/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/select.c,v 2.70.1.20 1993/06/07 02:59:32 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "drawing.e"
#include "dup.e"
#include "exec.e"
#include "file.e"
#include "grid.e"
#include "group.e"
#include "mainloop.e"
#include "mark.e"
#include "move.e"
#include "msg.e"
#include "names.e"
#include "obj.e"
#include "page.e"
#include "raster.e"
#include "rect.e"
#include "ruler.e"
#include "setup.e"
#include "stretch.e"

#define FORWARD 0
#define REVERSE 1

int		selLtX, selLtY, selRbX, selRbY;
int		selObjLtX, selObjLtY, selObjRbX, selObjRbY;
int		numObjSelected=0;
int		numObjLocked=0;
struct SelRec	* topSel = NULL, * botSel = NULL;
struct VSelRec	* topVSel = NULL, * botVSel = NULL;

void CalcBBox (X1, Y1, X2, Y2, LtX, LtY, RbX, RbY)
   int X1, Y1, X2, Y2, * LtX, * LtY, * RbX, * RbY;
{
   if (X1 < X2)
      if (Y1 < Y2)
      {
         *LtX = X1; *LtY = Y1; *RbX = X2; *RbY = Y2;
      }
      else
      {
         *LtX = X1; *LtY = Y2; *RbX = X2; *RbY = Y1;
      }
   else
      if (Y1 < Y2)
      {
         *LtX = X2; *LtY = Y1; *RbX = X1; *RbY = Y2;
      }
      else
      {
         *LtX = X2; *LtY = Y2; *RbX = X1; *RbY = Y1;
      }
}

void CalcVertexBBox (LtX, LtY, RbX, RbY)
   int	* LtX, * LtY, * RbX, * RbY;
{
   register int		i, *x_ptr, *y_ptr;
   struct VSelRec	* vsel_ptr;

   *LtX = selRbX; *LtY = selRbY; *RbX = selLtX; *RbY = selLtY;

   for (vsel_ptr=topVSel; vsel_ptr!=NULL; vsel_ptr=vsel_ptr->next)
      for (i=0, x_ptr=vsel_ptr->x, y_ptr=vsel_ptr->y; i < vsel_ptr->n;
            i++, x_ptr++, y_ptr++)
      {
         if (*x_ptr < *LtX) *LtX = *x_ptr;
         if (*y_ptr < *LtY) *LtY = *y_ptr;
         if (*x_ptr > *RbX) *RbX = *x_ptr;
         if (*y_ptr > *RbY) *RbY = *y_ptr;
      }
}

void UnSelNonVertexObjs (highlight)
   int	highlight;
{
   register struct ObjRec	* obj_ptr;
   register struct SelRec	* sel_ptr, * prev_sel;

   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = prev_sel)
   {
      prev_sel = sel_ptr->prev;
      obj_ptr = sel_ptr->obj;

      if ((obj_ptr->type==OBJ_POLY || obj_ptr->type==OBJ_POLYGON) &&
            !obj_ptr->locked)
         continue;

      if (highlight) HighLightAnObj (obj_ptr);

      if (sel_ptr->prev == NULL)
         topSel = sel_ptr->next;
      else
         sel_ptr->prev->next = sel_ptr->next;

      if (sel_ptr->next == NULL)
         botSel = sel_ptr->prev;
      else
         sel_ptr->next->prev = sel_ptr->prev;

      cfree (sel_ptr);
   }
}

void JustRemoveAllVSel ()
{
   register struct VSelRec	* next_vsel;

   while (topVSel != NULL)
   {
      next_vsel = topVSel->next;
      cfree (topVSel->v_index);
      cfree (topVSel->x);
      cfree (topVSel->y);
      cfree (topVSel);
      topVSel = next_vsel;
   }
   botVSel = NULL;
}

void RemoveAllSel ()
{
   register struct SelRec	* next_sel;
   register struct VSelRec	* next_vsel;

   while (topSel != NULL)
   {
      next_sel = topSel->next;
      cfree (topSel);
      topSel = next_sel;
   }
   botSel = NULL;

   while (topVSel != NULL)
   {
      next_vsel = topVSel->next;
      cfree (topVSel->v_index);
      cfree (topVSel->x);
      cfree (topVSel->y);
      cfree (topVSel);
      topVSel = next_vsel;
   }
   botVSel = NULL;
   numObjSelected = 0;
}

static
struct ObjRec * FindAVertex (XOff, YOff, VIndex, AbsX, AbsY)
   int	XOff, YOff, * VIndex, * AbsX, * AbsY;
   /* XOff and YOff are screen offsets */
{
   register struct ObjRec	* obj_ptr;
   struct PolyRec		* poly_ptr;
   struct PolygonRec		* polygon_ptr;
   struct SelRec		* sel_ptr;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;

      if (obj_ptr->type != OBJ_POLY && obj_ptr->type != OBJ_POLYGON) continue;
      if (!(XOff >= OFFSET_X(obj_ptr->bbox.ltx)-3 &&
            YOff >= OFFSET_Y(obj_ptr->bbox.lty)-3 &&
            XOff <= OFFSET_X(obj_ptr->bbox.rbx)+3 &&
            YOff <= OFFSET_Y(obj_ptr->bbox.rby)+3)) continue;

      switch (obj_ptr->type)
      {
         case OBJ_POLY:
            poly_ptr = obj_ptr->detail.p;
            if (PtInPolyMark (XOff,YOff,poly_ptr->n,poly_ptr->vlist,VIndex))
            {
               *AbsX = poly_ptr->vlist[*VIndex].x;
               *AbsY = poly_ptr->vlist[*VIndex].y;
               return (obj_ptr);
            }
            break;
         case OBJ_POLYGON:
            polygon_ptr = obj_ptr->detail.g;
            if (PtInPolyMark (XOff, YOff, polygon_ptr->n, polygon_ptr->vlist,
                  VIndex))
            {
               *AbsX = polygon_ptr->vlist[*VIndex].x;
               *AbsY = polygon_ptr->vlist[*VIndex].y;
               return (obj_ptr);
            }
            break;
      }
   }
   return (NULL);
}

struct ObjRec * FindAnObj (XOff, YOff, OwnerObj)
   int			XOff, YOff;
   struct ObjRec	* * OwnerObj;
   /* XOff and YOff are screen offsets */
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;
   struct ObjRec		* sub_obj;

   *OwnerObj = NULL;

   for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
   {
      for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
      {
         if (attr_ptr->shown &&
               XOff >= OFFSET_X(attr_ptr->obj->bbox.ltx)-3 &&
               YOff >= OFFSET_Y(attr_ptr->obj->bbox.lty)-3 &&
               XOff <= OFFSET_X(attr_ptr->obj->bbox.rbx)+3 &&
               YOff <= OFFSET_Y(attr_ptr->obj->bbox.rby)+3)
         {
            *OwnerObj = obj_ptr;
            return (attr_ptr->obj);
         }
      }

      if (XOff >= OFFSET_X(obj_ptr->bbox.ltx)-3 &&
            YOff >= OFFSET_Y(obj_ptr->bbox.lty)-3 &&
            XOff <= OFFSET_X(obj_ptr->bbox.rbx)+3 &&
            YOff <= OFFSET_Y(obj_ptr->bbox.rby)+3)
      {
         switch (obj_ptr->type)
         {
            case OBJ_TEXT: return (obj_ptr);
            case OBJ_XBM:
               if (FindGoodXBm (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_XPM:
               if (FindGoodXPm (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_BOX:
               if (FindGoodBox (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_OVAL:
               if (FindGoodOval (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_POLY:
               if (FindGoodPoly (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_POLYGON:
               if (FindGoodPolygon (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_ARC:
               if (FindGoodArc (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_RCBOX:
               if (FindGoodRCBox (XOff,YOff,obj_ptr)) return (obj_ptr); break;

            case OBJ_GROUP:
            case OBJ_SYM:
            case OBJ_ICON:
               if (FindGoodObj (XOff, YOff, obj_ptr->detail.r->first, &sub_obj))
               {
                  if (sub_obj == NULL)
                  {
                     *OwnerObj = NULL;
                     return (obj_ptr);
                  }
                  else
                  {
                     *OwnerObj = obj_ptr;
                     return (sub_obj);
                  }
               }
         }
      }
   }
   return (NULL);
}

static
int VertexAlreadySelected (ObjPtr, VIndex, VSelPtr)
   register struct ObjRec	* ObjPtr;
   int				VIndex;
   register struct VSelRec	* * VSelPtr;
{
   register int	i;

   for (*VSelPtr = topVSel; *VSelPtr != NULL; *VSelPtr = (*VSelPtr)->next)
      if ((*VSelPtr)->obj == ObjPtr)
      {
         for (i = 0; i < (*VSelPtr)->n; i++)
            if ((*VSelPtr)->v_index[i] == VIndex)
               return (TRUE);
         return (FALSE);
      }
   return (FALSE);
}

static
struct SelRec * AlreadySelected (ObjPtr)
   register struct ObjRec	* ObjPtr;
{
   register struct SelRec	* sel_ptr;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      if (sel_ptr->obj == ObjPtr)
         return (sel_ptr);
   return (NULL);
}

void AddSel (PrevPtr, NextPtr, SelPtr)
   struct SelRec	* PrevPtr, * NextPtr, * SelPtr;
   /* add SelPtr between PrevPtr and NextPtr */
{
   SelPtr->prev = PrevPtr;
   SelPtr->next = NextPtr;

   if (PrevPtr == NULL)
      topSel = SelPtr;
   else
      PrevPtr->next = SelPtr;

   if (NextPtr == NULL)
      botSel = SelPtr;
   else
      NextPtr->prev = SelPtr;
   numObjSelected++;
}

void AddNewSelObj (ObjPtr)
   register struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* obj_ptr = topObj;
   register struct SelRec	* sel_ptr = topSel, * new_sel_ptr;

   new_sel_ptr = (struct SelRec *) calloc (1, sizeof(struct SelRec));
   new_sel_ptr->obj = ObjPtr;

   for ( ; sel_ptr != NULL && obj_ptr != ObjPtr; obj_ptr = obj_ptr->next)
      if (obj_ptr == sel_ptr->obj)
         sel_ptr = sel_ptr->next;

   if (sel_ptr == NULL)
   {  /* the object is below the last selected object */
      if (botSel == NULL)
         topSel = new_sel_ptr;
      else
         botSel->next = new_sel_ptr;

      new_sel_ptr->prev = botSel;
      new_sel_ptr->next = NULL;
      botSel = new_sel_ptr;
   }
   else
   {  /* the object is between sel_ptr and sel_ptr->prev */
      if (sel_ptr->prev == NULL)
         topSel = new_sel_ptr;
      else
         sel_ptr->prev->next = new_sel_ptr;

      new_sel_ptr->next = sel_ptr;
      new_sel_ptr->prev = sel_ptr->prev;
      sel_ptr->prev = new_sel_ptr;
   }
   numObjSelected++;
}

void UpdSelBBox ()
   /* update selLtX, selLtY, selRbX, selRbY */
{
   register struct ObjRec	* obj_ptr;
   register struct SelRec	* sel_ptr;

   numObjSelected = 0;
   numObjLocked = 0;
   if ((sel_ptr = topSel) == NULL) return;

   numObjSelected++;
   obj_ptr = sel_ptr->obj;
   if (obj_ptr->locked) numObjLocked++;
   selLtX = obj_ptr->bbox.ltx;
   selLtY = obj_ptr->bbox.lty;
   selRbX = obj_ptr->bbox.rbx;
   selRbY = obj_ptr->bbox.rby;
   selObjLtX = obj_ptr->obbox.ltx;
   selObjLtY = obj_ptr->obbox.lty;
   selObjRbX = obj_ptr->obbox.rbx;
   selObjRbY = obj_ptr->obbox.rby;

   for (sel_ptr = topSel->next; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      numObjSelected++;
      obj_ptr = sel_ptr->obj;
      if (obj_ptr->locked) numObjLocked++;
      if (obj_ptr->bbox.ltx < selLtX) selLtX = obj_ptr->bbox.ltx;
      if (obj_ptr->bbox.lty < selLtY) selLtY = obj_ptr->bbox.lty;
      if (obj_ptr->bbox.rbx > selRbX) selRbX = obj_ptr->bbox.rbx;
      if (obj_ptr->bbox.rby > selRbY) selRbY = obj_ptr->bbox.rby;
      if (obj_ptr->obbox.ltx < selObjLtX) selObjLtX = obj_ptr->obbox.ltx;
      if (obj_ptr->obbox.lty < selObjLtY) selObjLtY = obj_ptr->obbox.lty;
      if (obj_ptr->obbox.rbx > selObjRbX) selObjRbX = obj_ptr->obbox.rbx;
      if (obj_ptr->obbox.rby > selObjRbY) selObjRbY = obj_ptr->obbox.rby;
   }
}
 
static
struct VSelRec * SelectOneVertex (XOff, YOff)
   int	XOff, YOff;
   /* XOff and YOff are screen offsets */
{
   register struct ObjRec	* obj_ptr;
   int				v_index, x, y;

   JustRemoveAllVSel ();
   if ((obj_ptr = FindAVertex (XOff, YOff, &v_index, &x, &y)) == NULL)
      return (NULL);

   topVSel = (struct VSelRec *) calloc (1, sizeof(struct VSelRec));
   topVSel->obj = obj_ptr;
   topVSel->max_v = 10;
   topVSel->v_index = (int *) calloc (10, sizeof(int));
   topVSel->x = (int *) calloc (10, sizeof(int));
   topVSel->y = (int *) calloc (10, sizeof(int));
   topVSel->v_index[0] = v_index;
   topVSel->x[0] = x;
   topVSel->y[0] = y;
   if (obj_ptr->type==OBJ_POLYGON && v_index==0)
   {
      topVSel->n = 2;
      topVSel->v_index[1] = obj_ptr->detail.g->n-1;
      topVSel->x[1] = x;
      topVSel->y[1] = y;
   }
   else
      topVSel->n = 1;
   topVSel->next = NULL;
   topVSel->prev = NULL;
   botVSel = topVSel;
   UpdSelBBox ();

   return (topVSel);
}

static
struct SelRec * SelectOneObj (XOff, YOff)
   int	XOff, YOff;
   /* XOff and YOff are screen offsets */
{
   register struct ObjRec	* obj_ptr;
   struct ObjRec		* owner_obj;

   RemoveAllSel ();
   if ((obj_ptr = FindAnObj (XOff,YOff,&owner_obj)) == NULL) return (NULL);

   if (owner_obj != NULL) obj_ptr = owner_obj;

   topSel = (struct SelRec *) calloc (1, sizeof(struct SelRec));
   topSel->next = NULL;
   topSel->obj = obj_ptr;
   topSel->prev = NULL;
   botSel = topSel;
   UpdSelBBox ();

   return (topSel);
}

static
int FindVertices (X1, Y1, X2, Y2, TopVSel, BotVSel)
   int			X1, Y1, X2, Y2;
   struct VSelRec	* * TopVSel, * * BotVSel;
   /* X1, Y1, X2, Y2 are absolute coordinates */
{
   register struct ObjRec	* obj_ptr;
   register struct VSelRec	* vsel_ptr;
   register int			i;
   struct SelRec		* sel_ptr;
   struct BBRec			bbox;
   XPoint			* v;
   int				n, count, max_count, j;

   *TopVSel = *BotVSel = NULL;

   bbox.ltx = X1; bbox.lty = Y1;
   bbox.rbx = X2; bbox.rby = Y2;
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      obj_ptr = sel_ptr->obj;

      if (obj_ptr->type != OBJ_POLY && obj_ptr->type != OBJ_POLYGON) continue;
      if (!BBoxIntersect (bbox, obj_ptr->bbox)) continue;

      v = (obj_ptr->type==OBJ_POLY) ? obj_ptr->detail.p->vlist :
            obj_ptr->detail.g->vlist;
      n = (obj_ptr->type==OBJ_POLY) ? obj_ptr->detail.p->n :
            obj_ptr->detail.g->n;
      for (i = 0, count = 0; i < n; i++)
         if (v[i].x >= X1 && v[i].x <= X2 && v[i].y >= Y1 && v[i].y <= Y2)
            count++;

      if (count != 0)
      {
         vsel_ptr = (struct VSelRec *) calloc (1, sizeof(struct VSelRec));
         vsel_ptr->obj = obj_ptr;
         vsel_ptr->next = *TopVSel;
         vsel_ptr->prev = NULL;
         if (*TopVSel == NULL)
            *BotVSel = vsel_ptr;
         else
            (*TopVSel)->prev = vsel_ptr;
         *TopVSel = vsel_ptr;
         vsel_ptr->n = count;
         max_count = ((count%10) == 0) ? 10*((int)(count/10)) :
               10*((int)(count/10)+1);
         vsel_ptr->max_v = max_count;
         vsel_ptr->v_index = (int *) calloc (max_count, sizeof(int));
         vsel_ptr->x = (int *) calloc (max_count, sizeof(int));
         vsel_ptr->y = (int *) calloc (max_count, sizeof(int));

         for (i = 0, j = 0; i < n; i++)
            if (v[i].x >= X1 && v[i].x <= X2 && v[i].y >= Y1 && v[i].y <= Y2)
            {
               vsel_ptr->v_index[j] = i;
               vsel_ptr->x[j] = v[i].x;
               vsel_ptr->y[j] = v[i].y;
               j++;
            }
      }
   }
   return (*TopVSel != NULL);
}
 
static
struct SelRec * FindObjects (X1, Y1, X2, Y2, TopSel, BotSel)
   int			X1, Y1, X2, Y2;
   struct SelRec	* * TopSel, * * BotSel;
   /* X1, Y1, X2, Y2 are absolute coordinates */
{
   register struct ObjRec	* obj_ptr;
   register struct SelRec	* sel_ptr;

   *TopSel = *BotSel = NULL;

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      if (X1 <= obj_ptr->bbox.ltx && X2 >= obj_ptr->bbox.rbx &&
            Y1 <= obj_ptr->bbox.lty && Y2 >= obj_ptr->bbox.rby)
      {
         sel_ptr = (struct SelRec *) calloc (1, sizeof(struct SelRec));
         sel_ptr->next = *TopSel;
         sel_ptr->obj = obj_ptr;
         sel_ptr->prev = NULL;
         if (*TopSel == NULL)
            *BotSel = sel_ptr;
         else
            (*TopSel)->prev = sel_ptr;
         *TopSel = sel_ptr;
      }

   return (*TopSel);
}
 
void SelBox (window, gc, x1, y1, x2, y2)
   Window	window;
   GC		gc;
   int		x1, y1, x2, y2;
{
   XPoint	sv[5];

   if (x1 == x2 || y1 == y2)
      XDrawLine (mainDisplay, window, gc, x1, y1, x2, y2);
   else
   {
      sv[0].x = (short)x1; sv[0].y = (short)y1;
      sv[1].x = (short)x1; sv[1].y = (short)y2;
      sv[2].x = (short)x2; sv[2].y = (short)y2;
      sv[3].x = (short)x2; sv[3].y = (short)y1;
      sv[4].x = (short)x1; sv[4].y = (short)y1;
      XDrawLines (mainDisplay, window, gc, sv, 5, CoordModeOrigin);
   }
}

static
struct ObjRec * PtInObjList (XOff, YOff, FirstObjPtr)
   int			XOff, YOff;
   struct ObjRec	* FirstObjPtr;
   /* XOff and YOff are screen offsets */
{
   register struct ObjRec	* obj_ptr, * obj_ptr1;
   register struct AttrRec	* attr_ptr;

   for (obj_ptr = FirstObjPtr; obj_ptr != NULL; obj_ptr = obj_ptr->next)
   {
      for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
         if (attr_ptr->shown &&
               XOff >= OFFSET_X(attr_ptr->obj->bbox.ltx)-3 &&
               YOff >= OFFSET_Y(attr_ptr->obj->bbox.lty)-3 &&
               XOff <= OFFSET_X(attr_ptr->obj->bbox.rbx)+3 &&
               YOff <= OFFSET_Y(attr_ptr->obj->bbox.rby)+3)
            return (attr_ptr->obj);

      if (XOff >= OFFSET_X(obj_ptr->bbox.ltx)-3 &&
            YOff >= OFFSET_Y(obj_ptr->bbox.lty)-3 &&
            XOff <= OFFSET_X(obj_ptr->bbox.rbx)+3 &&
            YOff <= OFFSET_Y(obj_ptr->bbox.rby)+3)
      {
         switch (obj_ptr->type)
         {
            case OBJ_TEXT: return (obj_ptr);
            case OBJ_XBM:
               if (FindGoodXBm (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_XPM:
               if (FindGoodXPm (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_BOX:
               if (FindGoodBox (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_OVAL:
               if (FindGoodOval (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_POLY:
               if (FindGoodPoly (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_POLYGON:
               if (FindGoodPolygon (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_ARC:
               if (FindGoodArc (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_RCBOX:
               if (FindGoodRCBox (XOff,YOff,obj_ptr)) return (obj_ptr); break;

            case OBJ_GROUP:
            case OBJ_SYM:
            case OBJ_ICON:
               obj_ptr1 = PtInObjList (XOff,YOff,obj_ptr->detail.r->first);
               if (obj_ptr1 != NULL) return (obj_ptr1);
               break;
         }
      }
   }
   return (NULL);
}

static
struct ObjRec * PtInSelected (XOff, YOff)
   int	XOff, YOff;
   /* XOff and YOff are screen offsets */
{
   register struct SelRec	* sel_ptr;
   register struct ObjRec	* obj_ptr, * obj_ptr1;
   register struct AttrRec	* attr_ptr;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;

      for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
         if (attr_ptr->shown &&
               XOff >= OFFSET_X(attr_ptr->obj->bbox.ltx)-3 &&
               YOff >= OFFSET_Y(attr_ptr->obj->bbox.lty)-3 &&
               XOff <= OFFSET_X(attr_ptr->obj->bbox.rbx)+3 &&
               YOff <= OFFSET_Y(attr_ptr->obj->bbox.rby)+3)
            return (attr_ptr->obj);

      if (XOff >= OFFSET_X(obj_ptr->bbox.ltx)-3 &&
            YOff >= OFFSET_Y(obj_ptr->bbox.lty)-3 &&
            XOff <= OFFSET_X(obj_ptr->bbox.rbx)+3 &&
            YOff <= OFFSET_Y(obj_ptr->bbox.rby)+3)
      {
         switch (obj_ptr->type)
         {
            case OBJ_TEXT: return (obj_ptr);
            case OBJ_XBM:
               if (FindGoodXBm (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_XPM:
               if (FindGoodXPm (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_BOX:
               if (FindGoodBox (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_OVAL:
               if (FindGoodOval (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_POLY:
               if (FindGoodPoly (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_POLYGON:
               if (FindGoodPolygon (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_ARC:
               if (FindGoodArc (XOff,YOff,obj_ptr)) return (obj_ptr); break;
            case OBJ_RCBOX:
               if (FindGoodRCBox (XOff,YOff,obj_ptr)) return (obj_ptr); break;

            case OBJ_GROUP:
            case OBJ_SYM:
            case OBJ_ICON:
               obj_ptr1 = PtInObjList (XOff,YOff,obj_ptr->detail.r->first);
               if (obj_ptr1 != NULL) return (obj_ptr1);
               break;
         }
      }
   }
   return (NULL);
}

static
void ToggleVertexSelection (ObjPtr, VIndex, AbsX, AbsY)
   struct ObjRec	* ObjPtr;
   int			VIndex, AbsX, AbsY;
{
   int			i, n;
   struct VSelRec	* vsel_ptr;
   int			j;

   if (!(ObjPtr->type==OBJ_POLYGON && ObjPtr->detail.g->n-1==VIndex))
   {
      MARK(drawWindow, revDefaultGC, OFFSET_X(AbsX), OFFSET_Y(AbsY));
      MARKV(drawWindow, revDefaultGC, OFFSET_X(AbsX), OFFSET_Y(AbsY));
   }
   if (VertexAlreadySelected (ObjPtr, VIndex, &vsel_ptr))
   {  /* de-select a vertex */
      if (vsel_ptr->n == 1)
      {
         if (vsel_ptr->prev == NULL)
            topVSel = vsel_ptr->next;
         else
            vsel_ptr->prev->next = vsel_ptr->next;

         if (vsel_ptr->next == NULL)
            botVSel = vsel_ptr->prev;
         else
            vsel_ptr->next->prev = vsel_ptr->prev;

         cfree (vsel_ptr->v_index);
         cfree (vsel_ptr->x);
         cfree (vsel_ptr->y);
         cfree (vsel_ptr);
      }
      else
      {
         for (j = 0; j < vsel_ptr->n; j++)
            if (vsel_ptr->v_index[j] == VIndex)
               break;
         if (j > vsel_ptr->n)
            fprintf (stderr, "%s.  Please send bug report.\n",
                  "Inconsistent vertex selection");
         for (i = j; i < vsel_ptr->n-1; i++)
         {
            vsel_ptr->v_index[i] = vsel_ptr->v_index[i+1];
            vsel_ptr->x[i] = vsel_ptr->x[i+1];
            vsel_ptr->y[i] = vsel_ptr->y[i+1];
         }
         vsel_ptr->n--;
      }
   }
   else
   {
      if (vsel_ptr == NULL)
      {
         vsel_ptr = (struct VSelRec *) calloc (1, sizeof(struct VSelRec));
         vsel_ptr->obj = ObjPtr;
         n = vsel_ptr->n = 1;
         vsel_ptr->max_v = 10;
         vsel_ptr->v_index = (int *) calloc (10, sizeof(int));
         vsel_ptr->x = (int *) calloc (10, sizeof(int));
         vsel_ptr->y = (int *) calloc (10, sizeof(int));

         vsel_ptr->prev = NULL;
         vsel_ptr->next = topVSel;
         if (topVSel == NULL)
            botVSel = vsel_ptr;
         else
            topVSel->prev = vsel_ptr;
         topVSel = vsel_ptr;
      }
      else
      {
         n = ++(vsel_ptr->n);
         if (n > vsel_ptr->max_v)
         {
            int	max_v;

            vsel_ptr->max_v += 10;
            max_v = vsel_ptr->max_v;
            vsel_ptr->v_index = (int *) realloc (vsel_ptr->v_index, max_v);
            vsel_ptr->x = (int *) realloc (vsel_ptr->x, max_v);
            vsel_ptr->y = (int *) realloc (vsel_ptr->y, max_v);
         }
      }
      vsel_ptr->v_index[n-1] = VIndex;
      vsel_ptr->x[n-1] = AbsX;
      vsel_ptr->y[n-1] = AbsY;
   }
}

static
void ToggleSelectedObjIfSelectedAlready (ObjPtr)
   register struct ObjRec	* ObjPtr;
{
   register struct SelRec	* sel_ptr;

   if ((sel_ptr = AlreadySelected (ObjPtr)) != NULL)
   {  /* de-select an object */
      HighLightAnObj (ObjPtr);

      if (sel_ptr->prev == NULL)
         topSel = sel_ptr->next;
      else
         sel_ptr->prev->next = sel_ptr->next;

      if (sel_ptr->next == NULL)
         botSel = sel_ptr->prev;
      else
         sel_ptr->next->prev = sel_ptr->prev;

      cfree (sel_ptr);
      numObjSelected--;
   }
   else
   {  /* add a newly selected object */
      AddNewSelObj (ObjPtr);
      HighLightAnObj (ObjPtr);
   }
}

static
void ContinueSel (XOff, YOff, ShiftKeyDown)
   int 	XOff, YOff, ShiftKeyDown;
   /* XOff and YOff are screen offsets, and they are not on grid */
{
   register int		i;
   int 			end_x, end_y, v_index;
   int 			done = FALSE, ltx, lty, rbx, rby, dx, dy, x, y;
   int 			new_end_x, new_end_y;
   XEvent		input, ev;
   XMotionEvent		* motion_ev;
   struct SelRec	* sel_ptr, * top_sel_ptr, * bot_sel_ptr;
   struct VSelRec	* vsel_ptr, * top_vsel_ptr, * bot_vsel_ptr;
   struct ObjRec	* obj_ptr, * owner_obj;

   end_x = XOff;
   end_y = YOff; 

   SelBox (drawWindow, revDefaultGC, XOff, YOff, end_x, end_y);
   XGrabPointer (mainDisplay, drawWindow, False,
         PointerMotionMask | ButtonReleaseMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
   
   while (!done)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonRelease)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         SelBox (drawWindow, revDefaultGC, XOff, YOff, end_x, end_y);
         done = TRUE;
      }
      else if (input.type == MotionNotify)
      {
         motion_ev = &(input.xmotion);
         new_end_x = motion_ev->x;
         new_end_y = motion_ev->y;

         SelBox (drawWindow, revDefaultGC, XOff, YOff, end_x, end_y);
         end_x = new_end_x; end_y = new_end_y;
         SelBox (drawWindow, revDefaultGC, XOff, YOff, end_x, end_y);

         MarkRulers (end_x, end_y);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }

   dx = abs (XOff - end_x);
   dy = abs (YOff - end_y);
   if (curChoice == VERTEXMODE)
   {
      if (dx <= 2 && dy <= 2)
      {
         if (topSel == NULL)
         {
            if (SelectOneObj (XOff, YOff) != NULL)
            {
               if ((topSel->obj->type == OBJ_POLY ||
                     topSel->obj->type == OBJ_POLYGON) && !topSel->obj->locked)
                  HighLightForward ();
               else
                  RemoveAllSel ();
            }
         }
         else if (ShiftKeyDown)
         {
            obj_ptr = FindAVertex (XOff, YOff, &v_index, &x, &y);
            if (obj_ptr != NULL)
            {
               ToggleVertexSelection (obj_ptr, v_index, x, y);
               if (obj_ptr->type==OBJ_POLYGON && v_index==0)
                  ToggleVertexSelection (obj_ptr,obj_ptr->detail.g->n-1,x,y);
               UpdSelBBox ();
            }
         }
         else
         {
            HighLightReverse ();
            SelectOneVertex (XOff, YOff);
            HighLightForward ();
         }
      }
      else
      {
         CalcBBox (XOff, YOff, end_x, end_y, &ltx, &lty, &rbx, &rby);
         if (topSel == NULL)
         {
            if (FindObjects (ABS_X(ltx), ABS_Y(lty), ABS_X(rbx), ABS_Y(rby),
                  &top_sel_ptr, &bot_sel_ptr) != NULL)
            {
               topSel = top_sel_ptr;
               botSel = bot_sel_ptr;
               UnSelNonVertexObjs (FALSE); /* do not highlight */
               UpdSelBBox ();
               HighLightForward ();
            }
         }
         else if (ShiftKeyDown)
         {
            if (FindVertices (ABS_X(ltx), ABS_Y(lty), ABS_X(rbx), ABS_Y(rby),
                  &top_vsel_ptr, &bot_vsel_ptr))
            {
               struct VSelRec	* next_vsel;

               for (vsel_ptr=top_vsel_ptr; vsel_ptr!=NULL; vsel_ptr=next_vsel)
               {
                  obj_ptr = vsel_ptr->obj;
                  for (i = 0; i < vsel_ptr->n; i++)
                     ToggleVertexSelection (obj_ptr, vsel_ptr->v_index[i],
                           vsel_ptr->x[i], vsel_ptr->y[i]);
                  next_vsel = vsel_ptr->next;
                  cfree(vsel_ptr->v_index);
                  cfree(vsel_ptr->x);
                  cfree(vsel_ptr->y);
                  cfree(vsel_ptr);
               }
               UpdSelBBox ();
            }
         }
         else
         {
            HighLightReverse ();
            JustRemoveAllVSel ();
            if (FindVertices (ABS_X(ltx), ABS_Y(lty), ABS_X(rbx), ABS_Y(rby),
                  &top_vsel_ptr, &bot_vsel_ptr))
            {
               topVSel = top_vsel_ptr;
               botVSel = bot_vsel_ptr;
               UpdSelBBox ();
            }
            HighLightForward ();
         }
      }
   }
   else
   {
      if (dx <= 2 && dy <= 2)
      {
         if (ShiftKeyDown)
         {
            obj_ptr = FindAnObj (XOff, YOff, &owner_obj);
            if (obj_ptr != NULL)
            {
               if (owner_obj != NULL) obj_ptr = owner_obj;

               ToggleSelectedObjIfSelectedAlready (obj_ptr);
               UpdSelBBox ();
            }
         }
         else
         {
            if (topSel != NULL) HighLightReverse ();

            if (SelectOneObj (XOff, YOff) != NULL)
               HighLightForward ();
         }
      }
      else
      {
         CalcBBox (XOff, YOff, end_x, end_y, &ltx, &lty, &rbx, &rby);
         if (ShiftKeyDown)
         {
            if (FindObjects (ABS_X(ltx), ABS_Y(lty), ABS_X(rbx), ABS_Y(rby),
                  &top_sel_ptr, &bot_sel_ptr) != NULL)
            {
               struct SelRec	* next_sel;

               for (sel_ptr=top_sel_ptr; sel_ptr!=NULL; sel_ptr=next_sel)
               {
                  next_sel = sel_ptr->next;
                  obj_ptr = sel_ptr->obj;
                  ToggleSelectedObjIfSelectedAlready (obj_ptr);
                  cfree(sel_ptr);
               }
               UpdSelBBox ();
            }
         }
         else
         {
            if (topSel != NULL) HighLightReverse ();
            RemoveAllSel ();
            if (FindObjects (ABS_X(ltx), ABS_Y(lty), ABS_X(rbx), ABS_Y(rby),
                  &top_sel_ptr, &bot_sel_ptr) != NULL)
            {
               topSel = top_sel_ptr;
               botSel = bot_sel_ptr;
               UpdSelBBox ();
               HighLightForward ();
            }
         }
      }
   }
}

static Time	selectLastClickTime;
static int	selectJustClicked = FALSE;

static XComposeStatus	c_stat;

void Select (input)
   XEvent	* input;
{
   register int		i;
   XButtonEvent		* button_ev;
   struct SelRec	* sel_ptr;
   struct VSelRec	* vsel_ptr;
   struct ObjRec	* obj_ptr;
   int			mouse_x, mouse_y, grid_x, grid_y, corner;
   char			s[80];
   KeySym		key_sym;
   Time			click_time;

   if (input->type == KeyPress)
   {
      int	delta, dx=0, dy=0;

      XLookupString (&(input->xkey), s, 80-1, &key_sym, &c_stat);
      if (topSel==NULL && topVSel==NULL)
      {
         XKeyEvent	* key_ev = &(input->xkey);

         if (key_ev->state & ControlMask)
         {
            XButtonEvent	button_ev;

            button_ev.state = ShiftMask;
            switch (key_sym)
            {
               case XK_Left: ScrollLeft (&button_ev); break;
               case XK_Up: ScrollUp (&button_ev); break;
               case XK_Right: ScrollRight (&button_ev); break;
               case XK_Down: ScrollDown (&button_ev); break;
            }
         }
         else
         {
            switch (key_sym)
            {
               case XK_Left: ScrollLeft (NULL); break;
               case XK_Up: ScrollUp (NULL); break;
               case XK_Right: ScrollRight (NULL); break;
               case XK_Down: ScrollDown (NULL); break;
            }
         }
         return;
      }
      if (key_sym!=XK_Left && key_sym!=XK_Up && key_sym!=XK_Right &&
            key_sym!=XK_Down)
         return;

      if (gridOn)
         delta = (gridSystem==ENGLISH_GRID) ? GRID_ABS_SIZE(xyEnglishGrid) :
               GRID_ABS_SIZE(xyMetricGrid);
      else
         delta = GRID_ABS_SIZE(1);
      HighLightReverse ();
      switch (key_sym)
      {
         case XK_Left:  dx = -delta; dy = 0; break;
         case XK_Up:    dx = 0;      dy = -delta; break;
         case XK_Right: dx = delta;  dy = 0; break;
         case XK_Down:  dx = 0;      dy = delta; break;
      }
      if (curChoice == VERTEXMODE)
         MoveAllSelVs (dx, dy);
      else if (numObjSelected == numObjLocked)
      {
         HighLightForward ();
         return;
      }
      else
         MoveAllSel (dx, dy);
      HighLightForward ();
      UpdSelBBox ();
      if (justDupped)
      {
         dupDx += dx;
         dupDy += dy;
      }
      SetFileModified (TRUE);
      return;
   }
   else if (input->type != ButtonPress)
      return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = button_ev->x;
      mouse_y = button_ev->y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);

      click_time = button_ev->time;
      if (curChoice==VERTEXMODE && topSel!=NULL && selectJustClicked &&
            (click_time-selectLastClickTime) < doubleClickInterval)
      {
         selectJustClicked = FALSE;
         HighLightReverse ();
         RemoveAllSel ();
         return;
      }
      selectJustClicked = TRUE;
      selectLastClickTime = click_time;

      if (button_ev->state & (ShiftMask | ControlMask))
      {
         ContinueSel (mouse_x, mouse_y, TRUE);
         justDupped = FALSE;
         return;
      }
      else if (curChoice == VERTEXMODE && topVSel != NULL)
      {
         int	found = FALSE;

         for (vsel_ptr=topVSel; vsel_ptr!=NULL && !found;
               vsel_ptr=vsel_ptr->next)
            for (i = 0; i < vsel_ptr->n; i++)
               if (PtInMark (mouse_x, mouse_y, OFFSET_X(vsel_ptr->x[i]),
                     OFFSET_Y(vsel_ptr->y[i])))
               {
                  found = TRUE;
                  break;
               }
         if (found)
         {
            MoveSelVs (grid_x, grid_y);
            return;
         }
      }
      else if (curChoice == NOTHING && topSel != NULL)
      {
         if ((sel_ptr = PtInSelMark (mouse_x, mouse_y, &corner)) != NULL)
         {
            StretchSel (grid_x, grid_y, sel_ptr->obj, corner);
            return;
         }
         else if ((obj_ptr = PtInSelected (mouse_x, mouse_y)) != NULL)
         {
            MoveSel (grid_x, grid_y, obj_ptr);
            return;
         }
      }
      ContinueSel (mouse_x, mouse_y, FALSE);
      justDupped = FALSE;
   }
}

struct AttrRec * FindAttrWithName (ObjPtr, AttrName)
   struct ObjRec	* ObjPtr;
   char			* AttrName;
{
   register struct AttrRec	* attr_ptr;
   struct AttrRec		* found_attr=NULL;
   int				count=1;

   for (attr_ptr=ObjPtr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
      if (strcmp (attr_ptr->name, AttrName) == 0)
      {
         found_attr = attr_ptr;
         break;
      }
   if (attr_ptr == NULL) return (NULL);

   for (attr_ptr=found_attr->next; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
      if (strcmp (attr_ptr->name, AttrName) == 0)
         if (attr_ptr->obj->color == colorIndex)
            break;
         else
            count++;

   if (attr_ptr != NULL)
      found_attr = attr_ptr;
   else if (count != 1)
   {
      char	msg[MAXSTRING+1];

      sprintf (msg, "Can not find '%s' attribute with color %s.",
            AttrName, colorMenuItems[colorIndex]);
      Msg (msg);
      return (NULL);
   }
   return (found_attr);
}

struct AttrRec * ValidAttrArg (c_ptr, obj_ptr, new_c_ptr)
   char			* c_ptr, * * new_c_ptr;
   struct ObjRec	* obj_ptr;
{
   char	name[MAXSTRING+1], * name_ptr;
   struct AttrRec	* attr_ptr;

   name_ptr = name;
   if (c_ptr[0] == '$' && c_ptr[1] == '(')
   {
      for (c_ptr = &c_ptr[2]; *c_ptr != '\0'; c_ptr++)
         switch (*c_ptr)
         {
            case '\\':
               c_ptr++;
               *name_ptr++ = *c_ptr;
               break;
            case ')':
               *name_ptr++ = '=';
               *name_ptr = '\0';
               *new_c_ptr = c_ptr;
               attr_ptr = FindAttrWithName (obj_ptr, name);
               if (attr_ptr == NULL)
               {
                  char	msg[MAXSTRING+1];

                  sprintf (msg, "Can not find '%s' attr.", name);
                  Msg (msg);
               }
               return (attr_ptr);
            default:
               *name_ptr++ = *c_ptr;
               break;
         }
   }
   return (NULL);
}

static
void DoTeleport (teleport_attr)
   struct AttrRec	* teleport_attr;
{
   char	file_name[MAXPATHLENGTH];
   int	do_not_save=FALSE;

   while (fileModified)
   {
      switch (YesNoCancel (
            "File modified, save file before open? [ync](y)",
            CONFIRM_YES))
      {
         case CONFIRM_YES: SaveFile (); break;
         case CONFIRM_NO:
               do_not_save = TRUE;
               SetFileModified (FALSE);
               break;
         case CONFIRM_CANCEL: return;
      }
   }
   if (*(teleport_attr->s) == '/')
   {
      if (!LoadFile (teleport_attr->s) && do_not_save)
         SetFileModified (TRUE);
   }
   else
   {
      sprintf (file_name, "%s/%s", curDir, teleport_attr->s);
      if (!LoadFile (file_name) && do_not_save)
         SetFileModified (TRUE);
   }
}

#define DO_PAGE_BY_NUM 0
#define DO_PAGE_BY_NAME 1

static
void DoPageTeleport (teleport_attr, do_by_page_name)
   struct AttrRec	* teleport_attr;
   int			do_by_page_name;
{
   int	i;
   char	msg[MAXSTRING+1], dummy_str[MAXSTRING+1];

   if (do_by_page_name)
   {
      struct PageRec	* page_ptr;

      for (i=1, page_ptr=firstPage; page_ptr!=NULL;
            page_ptr=page_ptr->next, i++)
         if (page_ptr->name != NULL && strcmp (page_ptr->name,
               teleport_attr->s) == 0)
         {
            if (curPageNum != i) SetCurPage (i);
            return;
         }
      sprintf (msg, "Can not find page \"%s\" to warp to.", teleport_attr->s);
      Dialog (msg, "( <CR> or <ESC> to continue )", dummy_str);
   }
   else
   {
      i=atoi(teleport_attr->s);
      if (i >= 1 && i <= lastPageNum)
      {
         if (curPageNum != i) SetCurPage (i);
      }
      else
      {
         sprintf (msg, "Can not find page %1d to warp to.", i);
         Dialog (msg, "( <CR> or <ESC> to continue )", dummy_str);
      }
   }
}

void Teleport (button_ev)
   XButtonEvent	* button_ev;
{
   struct AttrRec	* teleport_attr, * launch_attr, * exec_attr;
   struct ObjRec	* obj_ptr, * owner_obj;
   char			msg[MAXSTRING], buf[MAXSTRING+1];
   int			len;

   if ((obj_ptr = FindAnObj (button_ev->x, button_ev->y, &owner_obj)) == NULL)
      return;

   if (owner_obj != NULL) obj_ptr = owner_obj;

   teleport_attr = FindAttrWithName (obj_ptr, TELEPORT_ATTR);
   if (teleport_attr != NULL) { DoTeleport (teleport_attr); return; }

   strcpy (buf, TELEPORT_ATTR);
   len = strlen (buf);
   if (buf[len-1] == '=')
   {
      sprintf (&buf[len-1], "_page#=");
      teleport_attr = FindAttrWithName (obj_ptr, buf);
      if (teleport_attr != NULL)
      {
         DoPageTeleport (teleport_attr, DO_PAGE_BY_NUM);
         return;
      }

      sprintf (&buf[len-1], "_page=");
      teleport_attr = FindAttrWithName (obj_ptr, buf);
      if (teleport_attr != NULL)
      {
         DoPageTeleport (teleport_attr, DO_PAGE_BY_NAME);
         return;
      }
   }

   launch_attr = FindAttrWithName (obj_ptr, LAUNCH_ATTR);
   if (launch_attr != NULL) { DoLaunch (launch_attr, obj_ptr); return; }

   exec_attr = FindAttrWithName (obj_ptr, EXEC_ATTR);
   if (exec_attr != NULL)
   {
      int	saved_intr_check_interval=intrCheckInterval;

      intrCheckInterval = 1;
      ShowInterrupt ();

      DoExec (exec_attr, obj_ptr);

      HideInterrupt ();
      intrCheckInterval = saved_intr_check_interval;

      return;
   }

   sprintf (msg, "Can not find interpretable attribute.");
   TwoLineMsg (msg, "No action taken.");
}

void SelAllObj (high_light)
   int	high_light;
{
   register struct SelRec	* sel_ptr;
   register struct ObjRec	* obj_ptr;

   TieLooseEnds ();
   SetCurChoice (NOTHING);
   if (topSel!=NULL) { HighLightReverse (); RemoveAllSel (); }

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      sel_ptr = (struct SelRec *) calloc (1, sizeof(struct SelRec));
      sel_ptr->next = topSel;
      sel_ptr->obj = obj_ptr;
      sel_ptr->prev = NULL;
      if (topSel == NULL)
         botSel = sel_ptr;
      else
         topSel->prev = sel_ptr;
      topSel = sel_ptr;
   }
   UpdSelBBox ();
   if (high_light) HighLightForward ();
   justDupped = FALSE;
}

static struct ObjRec	* tmpTopObj, * tmpBotObj;

static
void PushTmpObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   ObjPtr->next = tmpTopObj;
   ObjPtr->prev = NULL;

   if (tmpBotObj == NULL)
      tmpBotObj = ObjPtr;
   else
      tmpTopObj->prev = ObjPtr;

   tmpTopObj = ObjPtr;
}

static
void BreakSel ()
   /* break off selected objects from the main stream objects           */
   /* when returns, tmpTopObj points to the top of the selected objects */
   /*    and tmpBotObj points to the bottom of the selected objects     */
{
   register struct SelRec	* sel_ptr;

   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      UnlinkObj (sel_ptr->obj);
      PushTmpObj (sel_ptr->obj);
   }
}

void JustMoveSelToTop ()
{
   if (topSel == NULL) return;

   tmpTopObj = tmpBotObj = NULL;

   BreakSel ();
   tmpBotObj->next = topObj;
   if (topObj == NULL)
      curPage->bot = botObj = tmpBotObj;
   else
      topObj->prev = tmpBotObj;
   curPage->top = topObj = tmpTopObj;
}

void MoveSelToTop ()
{
   if (topSel == NULL) return;

   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   JustMoveSelToTop ();
   RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
}

void MoveSelToBot ()
{
   if (topSel == NULL) return;

   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   tmpTopObj = tmpBotObj = NULL;

   BreakSel ();
   tmpTopObj->prev = botObj;
   if (topObj == NULL)
      curPage->top = topObj = tmpTopObj;
   else
      botObj->next = tmpTopObj;
   curPage->bot = botObj = tmpBotObj;
   RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
}

void DelAllSelObj ()
{
   register struct ObjRec	* obj_ptr;
   register struct SelRec	* sel_ptr;
   register int			j, i;
   struct VSelRec		* vsel_ptr;
   XPoint			* v=NULL;
   struct PolyRec		* poly_ptr=NULL;
   struct PolygonRec		* polygon_ptr=NULL;
   int				n=0;
   short			* mark;

   if (topSel==NULL && topVSel==NULL) { Msg ("No object selected!"); return; }

   HighLightReverse ();
   if (curChoice == VERTEXMODE)
   {
      StartCompositeCmd ();
      for (vsel_ptr=botVSel; vsel_ptr!=NULL; vsel_ptr=vsel_ptr->prev)
      {
         int	delete_it=FALSE, extra_vertex=FALSE;

         obj_ptr = vsel_ptr->obj;

         PrepareToReplaceAnObj (obj_ptr);
         switch (obj_ptr->type)
         {
            case OBJ_POLY:
               poly_ptr = obj_ptr->detail.p;
               v = poly_ptr->vlist;
               n = poly_ptr->n;

               if (vsel_ptr->n >= n-1) delete_it = TRUE;
               break;
            case OBJ_POLYGON:
               polygon_ptr = obj_ptr->detail.g;
               v = polygon_ptr->vlist;
               n = polygon_ptr->n;

               for (j=0; j < vsel_ptr->n; j++)
                  if (vsel_ptr->v_index[j] == 0)
                  {
                     extra_vertex = TRUE;
                     break;
                  }

               if ((!extra_vertex && n-vsel_ptr->n <= 3) ||
                     (extra_vertex && n-vsel_ptr->n <= 2))
                  delete_it = TRUE;
               break;
         }
         if (delete_it)
         {
            struct SelRec	* saved_top_sel, * saved_bot_sel;

            for (sel_ptr=botSel; sel_ptr!=NULL; sel_ptr=sel_ptr->prev)
               if (sel_ptr->obj == obj_ptr)
                  break;

            if (sel_ptr->prev == NULL)
               topSel = sel_ptr->next;
            else
               sel_ptr->prev->next = sel_ptr->next;

            if (sel_ptr->next == NULL)
               botSel = sel_ptr->prev;
            else
               sel_ptr->next->prev = sel_ptr->prev;

            saved_top_sel = topSel;
            saved_bot_sel = botSel;
            topSel = botSel = sel_ptr;
            sel_ptr->next = sel_ptr->prev = NULL;
            DelObj (obj_ptr);
            topSel = saved_top_sel;
            botSel = saved_bot_sel;

            cfree (sel_ptr);
            ChangeReplaceOneCmdToDeleteCmd ();
         }
         else
         {
            mark = (short *) calloc (n, sizeof(short));
            for (j=0; j < n; j++) mark[j] = FALSE;
            for (j=0; j < vsel_ptr->n; j++) mark[vsel_ptr->v_index[j]] = TRUE;

            switch (obj_ptr->type)
            {
               case OBJ_POLY:
                  for (i=n-1; i >= 0; i--)
                  {
                     if (mark[i])
                     {
                        for (j=i+1; j < n; j++)
                           v[j-1] = v[j];
                        n--;
                     }
                  }
                  poly_ptr->n -= vsel_ptr->n;
                  AdjObjSplineVs (obj_ptr);
                  UpdPolyBBox (obj_ptr, poly_ptr->n, poly_ptr->vlist);
                  break;
               case OBJ_POLYGON:
                  for (i=n-1; i >= 0; i--)
                  {
                     if (mark[i])
                     {
                        for (j=i+1; j < n; j++) v[j-1] = v[j];
                        n--;
                     }
                  }
                  polygon_ptr->n -= vsel_ptr->n;
                  if (extra_vertex) v[polygon_ptr->n++] = v[0];
                  AdjObjSplineVs (obj_ptr);
                  UpdPolyBBox (obj_ptr, polygon_ptr->n, polygon_ptr->vlist);
                  break;
            }
            cfree (mark);
            AdjObjBBox (obj_ptr);
            RecordReplaceAnObj (obj_ptr);
         }
      }
      EndCompositeCmd ();
      JustRemoveAllVSel ();
   }
   else
   {
      PrepareToRecord (CMD_DELETE, topSel, botSel, numObjSelected);
      for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
      {
         UnlinkObj (sel_ptr->obj);
         FreeObj (sel_ptr->obj);
      }
      RemoveAllSel ();
      RecordCmd (CMD_DELETE, NULL, NULL, NULL, 0);
   }
   RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   SetFileModified (TRUE);
   justDupped = FALSE;
}

void GroupSingleObj ()
{
   tmpTopObj = tmpBotObj = NULL;

   BreakSel ();
   CreateGroupObj (tmpTopObj, tmpBotObj);
   RemoveAllSel ();

   topSel = botSel = (struct SelRec *) calloc (1, sizeof(struct SelRec));
   topSel->obj = topObj;
   topSel->next = topSel->prev = NULL;
   numObjSelected = 1;
}

void GroupSelObj ()
{
   if (topSel == NULL) { Msg ("No object to group!"); return; }
   if (curChoice==VERTEXMODE && topSel!=NULL)
   {
      Msg ("Can not group in vertex mode!");
      return;
   }
   if (topSel == botSel && topSel->obj->type != OBJ_POLY &&
         topSel->obj->type != OBJ_POLYGON)
   {
      Msg ("Can not group a single object!  Group aborted!");
      return;
   }

   tmpTopObj = tmpBotObj = NULL;

   HighLightReverse ();
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   BreakSel ();

   CreateGroupObj (tmpTopObj, tmpBotObj);

   RemoveAllSel ();
   topSel = botSel = (struct SelRec *) calloc (1, sizeof(struct SelRec));
   topSel->obj = topObj;
   topSel->next = topSel->prev = NULL;
   UpdSelBBox ();

   RecordCmd (CMD_MANY_TO_ONE, NULL, topSel, botSel, 1);
   RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   SetFileModified (TRUE);
   justDupped = FALSE;
}

void SelectTopObj ()
{
   if (topObj == NULL) return;

   topSel = botSel = (struct SelRec *) calloc (1, sizeof(struct SelRec));
   topSel->obj = topObj;
   topSel->next = topSel->prev = NULL;
   UpdSelBBox ();

   HighLightForward ();
   justDupped = FALSE;
}
