/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1995, William Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgif" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgif" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgif", the right to sell derivative works of
 * "tgif", and the right to distribute "tgif" for a fee) are
 * reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /u/multimedia/william/X11/TGIF2/RCS/font.c,v 2.104 1995/05/23 01:49:51 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "auxtext.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "exec.e"
#include "file.e"
#ifndef _NO_EXTERN
#include "font.e"
#endif
#include "mainmenu.e"
#include "mark.e"
#include "menu.e"
#include "msg.e"
#include "obj.e"
#include "pattern.e"
#include "page.e"
#include "prtgif.e"
#include "ps.e"
#include "raster.e"
#include "select.e"
#include "setup.e"
#include "text.e"
#include "util.e"

extern int	atoi ARGS_DECL((char *));

#define DEFAULT_FONT_SIZE 14

#define MAX_FONTS_MATCHED	20
#define MAX_FONTS_LIST		200
#define MAX_FONT_SIZE		50

#define ROT_NRM 0
#define ROT_MOV 1
#define ROT_NON 2
#define ROT_INV 3

static void	InitRotTbl();
static char	RotTbl21XX[256];

struct FontSizeRec {
   XFontStruct		* xfs;
   int			size, faked_size;
   struct FontSizeRec	* next;
};

struct FontFmlyRec {
   struct FontSizeRec	* fr[MAXFONTSTYLES];
         /* fr[i] points to the default size used for the given font */
         /* this font is used if the requested size is not found */
   char			* name_faked;
};

struct KanjiFontSizeRec {
   KanjiFontStruct		* kfs;
   int				size, faked_size;
   struct KanjiFontSizeRec	* next;
};

#define LOAD_EXIST	0
#define LOAD_SPECIFY	1
#define	LOAD_ALL	2

struct KanjiFontFmlyRec {
   struct KanjiFontSizeRec	* fr;
   char				* font_name;
   char				* font_encoding;
   char				* ps_font_name;
   char				* menu_str;
   Bool				vert;
   Bool				bold;
   int				policy;
   int				num_specify;
   int				* specify;
   char				* dummy;
};

XFontStruct	* canvasFontPtr=NULL;
KanjiFontStruct	* canvasKanjiFontPtr=NULL;
int	canvasFontHeight=0;
int	canvasFontAsc=0;
int	canvasFontDes=0;
int	canvasFontSize=INVALID;
int	cacheForKanjiFont=TRUE;

XFontStruct	* rulerFontPtr=NULL;
int	rulerFontWidth=0;
int	rulerFontHeight=0;
int	rulerFontAsc=0;
int	rulerFontDes=0;

XFontStruct	* defaultFontPtr=NULL;
int	defaultFontWidth=0;
int	defaultFontHeight=0;
int	defaultFontAsc=0;
int	defaultFontDes=0;

int	drawScaledKanji = TRUE;

static int	textBitmapWidth = INVALID;
static int	textBitmapHeight = INVALID;
static Pixmap	textBitmap;
static char	* scaleBuf = NULL;
static GC	baseGC = NULL;

XFontStruct	* menuFontPtr=NULL;
int	menuFontWidth=0;
int	menuFontHeight=0;
int	menuFontAsc=0;
int	menuFontDes=0;

int	curFont = FONT_COU;
int	curKanjiFont = KANJI_FONT_NONE;
int	curSize = 20;
int	curStyle = STYLE_NR;
int	curRotate = ROTATE0;
int	curHeight = 0 ;
int	curAsc = 0 ;
int	curDes = 0 ;

int	fakeTextVSpace = 0 ;

static int	defaultCurFont=(-1);
static int	defaultCurSize=(-1);
static int	defaultCurStyle=(-1);

static char	* initSizeMenuStr[] = {
      "8 ", "10", "11", "12", "14", "17", "18", "20", "24", "25", "34" };

char	* styleMenuStr[] = {
      "Roman             ^#o",
      "Bold              ^#b",
      "Italic            ^#t",
      "BoldItalic        ^#p",
      "---------------------",
      "Left              ^#l",
      "Center            ^#c",
      "Right             ^#r",
      "---------------------",
      "Scale/Rotate Japanese" };
char	* styleStr[] = { "Roman", "Bold", "Italic", "BoldItlic" };
char	* * fontMenuStr=NULL;
char	* * sizeMenuStr=NULL;
int	* fontSizes=NULL;
int	numFonts=MAXFONTS;
int	numKanjiFonts=MAX_KANJI_FONTS;
int	numFontSizes=MAXFONTS;

int	exitIfErrorInFindFontInfo = FALSE;

static int	defaultFontSize=DEFAULT_FONT_SIZE;
static int	numFakedFonts=0;
static int	numFakedKanjiFonts=0;
static char	fontNamePrefix[MAXSTRING+1];
static int	hasAlternateDefaultFonts=FALSE;

static struct FontFmlyRec	* fontFamilies=NULL;
static struct KanjiFontFmlyRec	* kanjiFontFamilies=NULL;

static char	* initFontInfoStr[]={
  "times-medium-r-normal", "iso8859-1", "Times-Roman",
  "times-bold-r-normal", "iso8859-1", "Times-Bold",
  "times-medium-i-normal", "iso8859-1", "Times-Italic",
  "times-bold-i-normal", "iso8859-1", "Times-BoldItalic",
  "courier-medium-r-normal", "iso8859-1", "Courier",
  "courier-bold-r-normal", "iso8859-1", "Courier-Bold",
  "courier-medium-o-normal", "iso8859-1", "Courier-Oblique",
  "courier-bold-o-normal", "iso8859-1", "Courier-BoldOblique",
  "helvetica-medium-r-normal", "iso8859-1", "Helvetica",
  "helvetica-bold-r-normal", "iso8859-1", "Helvetica-Bold",
  "helvetica-medium-o-normal", "iso8859-1", "Helvetica-Oblique",
  "helvetica-bold-o-normal", "iso8859-1", "Helvetica-BoldOblique",
  "new century schoolbook-medium-r-normal", "iso8859-1",
        "NewCenturySchlbk-Roman",
  "new century schoolbook-bold-r-normal", "iso8859-1",
        "NewCenturySchlbk-Bold",
  "new century schoolbook-medium-i-normal", "iso8859-1",
        "NewCenturySchlbk-Italic",
  "new century schoolbook-bold-i-normal", "iso8859-1",
        "NewCenturySchlbk-BoldItalic",
  "symbol-medium-r-normal", "adobe-fontspecific", "Symbol",
  "symbol-medium-r-normal", "adobe-fontspecific", "Symbol",
  "symbol-medium-r-normal", "adobe-fontspecific", "Symbol",
  "symbol-medium-r-normal", "adobe-fontspecific", "Symbol",
  NULL, NULL, NULL
};

static struct KanjiFontFmlyRec	initKanjiFontFamilies[] = {
   /* for KANJI_FONT_NONE */
   { NULL, NULL, NULL, NULL, " ", 0, 0, 0, 0, NULL, NULL },
   { NULL,
	"fixed-medium-r-normal",
	"jisx0208.1983-*",
	"Ryumin-Light-EUC-H",
	"Ryumin",
	FALSE,
	FALSE,
	LOAD_SPECIFY,
	0,
	NULL,
	"14"},
   { NULL,
	"fixed-medium-r-normal",
	"jisx0208.1983-*",
	"GothicBBB-Medium-EUC-H",
	"Gothic",
	FALSE,
	TRUE,
	LOAD_SPECIFY,
	0,
	NULL,
	"14"},
   { NULL,
	"fixed-medium-r-normal",
	"jisx0208.1983-*",
	"Ryumin-Light-EUC-V",
	"Ryumin-V",
	TRUE,
	FALSE,
	LOAD_SPECIFY,
	0,
	NULL,
	"14"},
   { NULL,
	"fixed-medium-r-normal",
	"jisx0208.1983-*",
	"GothicBBB-Medium-EUC-V",
	"Gothic-V",
	TRUE,
	TRUE,
	LOAD_SPECIFY,
	0,
	NULL,
	"14"},
};

static char	**altFontInfoStr=NULL;

static char	* * fontInfoStr=NULL;

static char	* charCodeToName[] =
{
/* \200 */ "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
/* \220 */ "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
/* \240 */ "x",
           "8#241 /exclamdown",
           "8#242 /cent",
           "8#243 /sterling",
           "8#244 /currency",
           "8#245 /yen",
           "8#246 /bar",
           "8#247 /section",
           "8#250 /dieresis",
           "8#251 /copyright",
           "8#252 /ordfeminine",
           "8#253 /guillemotleft",
           "8#254 /logicalnot",
           "8#255 /emdash",
           "8#256 /registered",
           "8#257 /macron",
/* \260 */ "8#260 /degree",
           "8#261 /plusminus",
           "8#262 /twosuperior",
           "8#263 /threesuperior",
           "8#264 /acute",
           "8#265 /mu",
           "8#266 /paragraph",
           "8#267 /periodcentered",
           "8#270 /cedilla",
           "x",
           "8#272 /ordmasculine",
           "8#273 /guillemotright",
           "x",
           "x",
           "x",
           "8#277 /questiondown",
/* \300 */ "8#300 /Agrave",
           "8#301 /Aacute",
           "8#302 /Acircumflex",
           "8#303 /Atilde",
           "8#304 /Adieresis",
           "8#305 /Aring",
           "8#306 /AE",
           "8#307 /Ccedilla",
           "8#310 /Egrave",
           "8#311 /Eacute",
           "8#312 /Ecircumflex",
           "8#313 /Edieresis",
           "8#314 /Igrave",
           "8#315 /Iacute",
           "8#316 /Icircumflex",
           "8#317 /Idieresis",
/* \320 */ "8#320 /Eth",
           "8#321 /Ntilde",
           "8#322 /Ograve",
           "8#323 /Oacute",
           "8#324 /Ocircumflex",
           "8#325 /Otilde",
           "8#326 /Odieresis",
           "8#327 /multiply",
           "8#330 /Oslash",
           "8#331 /Ugrave",
           "8#332 /Uacute",
           "8#333 /Ucircumflex",
           "8#334 /Udieresis",
           "8#335 /Yacute",
           "8#336 /Thorn",
           "8#337 /germandbls",
/* \340 */ "8#340 /agrave",
           "8#341 /aacute",
           "8#342 /acircumflex",
           "8#343 /atilde",
           "8#344 /adieresis",
           "8#345 /aring",
           "8#346 /ae",
           "8#347 /ccedilla",
           "8#350 /egrave",
           "8#351 /eacute",
           "8#352 /ecircumflex",
           "8#353 /edieresis",
           "8#354 /igrave",
           "8#355 /iacute",
           "8#356 /icircumflex",
           "8#357 /idieresis",
/* \360 */ "8#360 /eth",
           "8#361 /ntilde",
           "8#362 /ograve",
           "8#363 /oacute",
           "8#364 /ocircumflex",
           "8#365 /otilde",
           "8#366 /odieresis",
           "8#367 /divide",
           "8#370 /oslash",
           "8#371 /ugrave",
           "8#372 /uacute",
           "8#373 /ucircumflex",
           "8#374 /udieresis",
           "8#375 /yacute",
           "8#376 /thorn",
           "8#377 /ydieresis"
};

#define InfoIndex(FontIdx,StyleIdx) (((FontIdx)*MAXFONTSTYLES+(StyleIdx))*3)

static int	debugScalableFonts=FALSE; /* debug scalable font */
static char	gszAttemptedFontName[MAXSTRING+1];

static
struct FontSizeRec * FindFontInfo (font_index, style_index, size)
   int	font_index, style_index, size;
{
   struct FontSizeRec	* fs_ptr, * prev_fs=NULL, * new_fs_ptr;
   int			info_index, watch_cursor;
   XFontStruct		* xfs=NULL;
   int			faked_size = size, well_formed = FALSE;
   int			i, font_num ;
   char			font_name[MAXSTRING+1], msg[MAXSTRING];
   char			**fonts_matched = NULL;
   char			*p;

   *gszAttemptedFontName = '\0';
   for (fs_ptr=fontFamilies[font_index].fr[style_index]; fs_ptr != NULL;
         fs_ptr = fs_ptr->next)
   {
      if (fs_ptr->size == size)
         return(fs_ptr);
     
      if (fs_ptr->size > size) break;
      prev_fs = fs_ptr;
   }
   info_index = InfoIndex (font_index, style_index);
   if (strstr (fontInfoStr[info_index], "%d") != NULL)
      sprintf (gszAttemptedFontName, fontInfoStr[info_index], size);
   else
      sprintf (gszAttemptedFontName, "%s-%s-*-%1d-*-*-*-*-*-%s", fontNamePrefix,
            fontInfoStr[info_index], size, fontInfoStr[info_index+1]);

   watch_cursor = watchCursorOnMainWindow;
   if (!watch_cursor && mainWindow!=None)
   {
      SetWatchCursor (drawWindow);
      SetWatchCursor (mainWindow);
   }

   if (strstr (fontInfoStr[info_index], "%d") != NULL) {
      i = 0;
      p = fontInfoStr[info_index];
      if ( *p == '-' ) {
         for( ; *p != '\0'; p++ )
            if ( *p == '-' ) 
               i++;
         if ( i == 14 )
            well_formed = TRUE ;
      }
   }
   else {
      i = 9; 			/* reserve for "%s-%s-*-%1d-*-*-*-*-*-%s" */
      p = fontNamePrefix;
      if ( *p == '-' ) {
         for(; *p != '\0'; p ++)
            if ( *p == '-' )
               i++ ;
         for(p = fontInfoStr[info_index]; *p != '\0'; p ++)
            if(*p == '-')
               i++ ;
         for(p = fontInfoStr[info_index+1]; *p != '\0'; p ++)
            if(*p == '-')
               i ++;
         if ( i== 14 )
             well_formed = TRUE;
      }
   }

   if (xfs == NULL)
   {				/* Search fonts downward */
      for(faked_size = size; faked_size > 0; faked_size --) {
         if ( prev_fs && prev_fs->size == faked_size ) {
            xfs = prev_fs->xfs;
            faked_size = prev_fs->faked_size;
            break;
         }
         if (strstr (fontInfoStr[info_index], "%d") != NULL)
            sprintf(font_name, fontInfoStr[info_index], faked_size);
         else
            sprintf (font_name, "%s-%s-*-%1d-*-*-*-*-*-%s", fontNamePrefix,
                 fontInfoStr[info_index], faked_size,
                 fontInfoStr[info_index+1]);

         if(fonts_matched)
            XFreeFontNames(fonts_matched);
         if(well_formed) {
             fonts_matched = XListFonts(mainDisplay, &font_name[1],
                                       MAX_FONTS_MATCHED, &font_num);
             if(!fonts_matched && debugScalableFonts)
                 continue;
             if((xfs = XLoadQueryFont (mainDisplay, &font_name[1])) != NULL)
                 break;
             else if(debugScalableFonts)
                 continue;
             if(fonts_matched)
                 XFreeFontNames(fonts_matched);
         }
         fonts_matched = XListFonts(mainDisplay, font_name,
                                  MAX_FONTS_MATCHED, &font_num);
         if (fonts_matched &&
                     (xfs = XLoadQueryFont (mainDisplay, font_name)) != NULL)
              break;
         }
      }

      if (xfs == NULL)
      {					/* Search fonts upward */
         for(faked_size = size + 1; faked_size < MAX_FONT_SIZE; faked_size ++){
            if(fs_ptr && fs_ptr->size == faked_size) {
               xfs = fs_ptr->xfs;
               faked_size = fs_ptr->faked_size;
               break;
            }
            if (strstr (fontInfoStr[info_index], "%d") != NULL)
               sprintf (font_name, fontInfoStr[info_index], faked_size);
            else
               sprintf (font_name, "%s-%s-*-%1d-*-*-*-*-*-%s", fontNamePrefix,
                        fontInfoStr[info_index], faked_size,
                        fontInfoStr[info_index+1]);
            if(fonts_matched)
                 XFreeFontNames(fonts_matched);
            if(well_formed) {
               fonts_matched = XListFonts(mainDisplay, &font_name[1],
                                     MAX_FONTS_MATCHED, &font_num);
               if(!fonts_matched && debugScalableFonts)
                  continue;
               if((xfs = XLoadQueryFont (mainDisplay, &font_name[1])) != NULL)
                  break;
               else if(debugScalableFonts)
                  continue;
               if ( fonts_matched )
                  XFreeFontNames(fonts_matched);
            } 
            fonts_matched = XListFonts(mainDisplay, font_name,
                                     MAX_FONTS_MATCHED, &font_num);
            if (fonts_matched &&
                 (xfs = XLoadQueryFont (mainDisplay, font_name)) != NULL)
                    break;
         }
         if (defaultCurFont == (-1))
         {
            defaultCurFont = font_index;
            defaultCurStyle = style_index;
            defaultCurSize = defaultFontSize;
         }
   }
   else if (defaultCurFont == (-1))
   {
      defaultCurFont = font_index;
      defaultCurStyle = style_index;
      defaultCurSize = size;
   }

   if(fonts_matched != NULL)
      XFreeFontNames(fonts_matched);

   if (!watch_cursor && mainWindow!=None)
   {
      SetDefaultCursor (mainWindow);
      ShowCursor ();
   }
   if (xfs == NULL)
   {
      if (exitIfErrorInFindFontInfo) {
         if (strstr (fontInfoStr[info_index], "%d") != NULL)
            sprintf (font_name, fontInfoStr[info_index], size);
         else
            sprintf (font_name, "%s-%s-*-%1d-*-*-*-*-*-%s", fontNamePrefix,
                    fontInfoStr[info_index], size, fontInfoStr[info_index+1]);
         sprintf (msg, "Can not open '%s'!  Abort!", font_name);
         Error ("OpenFont()", msg );
      }
      return (NULL);
   }

   if (faked_size != size && !((prev_fs && prev_fs->xfs == xfs) ||
         (fs_ptr && fs_ptr->xfs == xfs))) {
      new_fs_ptr = (struct FontSizeRec *)calloc(1,sizeof(struct FontSizeRec));
      new_fs_ptr->next = fs_ptr;
      new_fs_ptr->xfs = xfs;
      new_fs_ptr->size = faked_size;
      if (prev_fs == NULL)
         fontFamilies[font_index].fr[style_index] = new_fs_ptr;
      else
         prev_fs->next = new_fs_ptr;

      if(faked_size < size)
         prev_fs = new_fs_ptr;
      else
         fs_ptr = new_fs_ptr;
   }
   new_fs_ptr = (struct FontSizeRec *)calloc(1,sizeof(struct FontSizeRec));
   new_fs_ptr->next = fs_ptr;
   new_fs_ptr->xfs = xfs;
   new_fs_ptr->size = size;
   new_fs_ptr->faked_size = faked_size;
   if (prev_fs == NULL)
      fontFamilies[font_index].fr[style_index] = new_fs_ptr;
   else
      prev_fs->next = new_fs_ptr;
   return (new_fs_ptr);
}

static
KanjiFontStruct * FindKanjiFontInfo (font_index, size)
    int		font_index, size;
{
   struct KanjiFontSizeRec	* fs_ptr, * prev_fs=NULL, * new_fs_ptr;
   int				watch_cursor;
   char				font_name[MAXSTRING+1], msg[MAXSTRING];
   XFontStruct			* xfs=NULL;
   int				faked_size = size, well_formed = FALSE ;
   char				** fonts_matched = NULL ;
   int				i, font_num ;
   int				policy, num_specify, * specify;
   KanjiFontStruct		* kfs;
   char				*p;

   if(font_index <= KANJI_FONT_NONE || font_index > numKanjiFonts)
      return( NULL );

   for (fs_ptr=kanjiFontFamilies[font_index].fr;
           fs_ptr != NULL;
           fs_ptr = fs_ptr->next)
   {
      if (fs_ptr->size == size)
         return(fs_ptr->kfs);
      if (fs_ptr->size > size)
         break;
      prev_fs = fs_ptr ;
   }

   policy = kanjiFontFamilies[font_index].policy;
   num_specify = kanjiFontFamilies[font_index].num_specify;
   specify = kanjiFontFamilies[font_index].specify;

   watch_cursor = watchCursorOnMainWindow;
   if (!watch_cursor && mainWindow!=None)
   {
      SetWatchCursor (drawWindow);
      SetWatchCursor (mainWindow);
   }

   if (strstr (kanjiFontFamilies[font_index].font_name, "%d") != NULL) {
      i = 0 ;
      p = kanjiFontFamilies[font_index].font_name ;
      if(*p == '-') {
          for(; *p != '\0'; p ++)
             if ( *p == '-' )
                 i++ ;
             if(i == 14)
                well_formed = TRUE ;
      }
   }
   else {
      i = 9 ;			/* reserve for "%s-%s-*-%1d-*-*-*-*-*-%s" */
      p = fontNamePrefix;
      if ( *p == '-' ) {
         for( ; *p != '\0'; p++ )
            if ( *p == '-' )
               i++;
         for(p = kanjiFontFamilies[font_index].font_name; *p != '\0'; p ++)
            if (*p == '-' )
               i++;
         for(p = kanjiFontFamilies[font_index].font_encoding; *p != '\0'; p ++)
            if (*p == '-')
               i++;
         if(i == 14)
              well_formed = TRUE;
      }
   }

   if (xfs == NULL)
   {                                  /* Search fonts downward */
      for(faked_size = size; faked_size > 0; faked_size --) {
         if(prev_fs && prev_fs->size == faked_size) {
            xfs = prev_fs->kfs->xfs;
            faked_size = prev_fs->faked_size;
            break;
         }
         if (policy == LOAD_SPECIFY)
            for(i = 0; i < num_specify; i ++)
               if(specify[i] == faked_size)
                  break;
         if (policy != LOAD_SPECIFY || i != num_specify) {
            if (strstr (kanjiFontFamilies[font_index].font_name, "%d") != NULL)
               sprintf (font_name, kanjiFontFamilies[font_index].font_name,
                     faked_size);
            else
               sprintf (font_name, "%s-%s-*-%1d-*-*-*-*-*-%s", fontNamePrefix,
                     kanjiFontFamilies[font_index].font_name, faked_size,
                     kanjiFontFamilies[font_index].font_encoding);
 
            if(fonts_matched)
               XFreeFontNames(fonts_matched);
            if(well_formed) {
               fonts_matched = XListFonts(mainDisplay, &font_name[1],
                     MAX_FONTS_MATCHED, &font_num);
               if(!fonts_matched && policy == LOAD_EXIST)
                  continue;
               if((xfs = XLoadQueryFont (mainDisplay, &font_name[1])) != NULL)
                     break;
               else if(policy == LOAD_EXIST)
                  continue;
               if(fonts_matched)
                  XFreeFontNames(fonts_matched);
            }
            fonts_matched = XListFonts(mainDisplay, font_name,
                  MAX_FONTS_MATCHED, &font_num);
            if (fonts_matched &&
               (xfs = XLoadQueryFont (mainDisplay, font_name)) != NULL)
                  break;
        }
      }
   }
   if (xfs == NULL)
   {                                  /* Search fonts upward */
      for(faked_size = size + 1; faked_size < MAX_FONT_SIZE; faked_size ++) {
         if(fs_ptr && fs_ptr->size == faked_size) {
            xfs = fs_ptr->kfs->xfs;
            faked_size = fs_ptr->faked_size;
            break;
         }
         if (policy == LOAD_SPECIFY)
            for(i = 0; i < num_specify; i ++)
               if(specify[i] == faked_size)
                  break;
         if (policy != LOAD_SPECIFY || i != num_specify) {
            if (strstr (kanjiFontFamilies[font_index].font_name, "%d") != NULL)
               sprintf (font_name, kanjiFontFamilies[font_index].font_name,
                     faked_size);
            else
               sprintf (font_name, "%s-%s-*-%1d-*-*-*-*-*-%s", fontNamePrefix,
                     kanjiFontFamilies[font_index].font_name, faked_size,
                     kanjiFontFamilies[font_index].font_encoding);
 
            if(fonts_matched)
               XFreeFontNames(fonts_matched);
            if(well_formed) {
               fonts_matched = XListFonts(mainDisplay, &font_name[1],
                     MAX_FONTS_MATCHED, &font_num);
               if(!fonts_matched && policy == LOAD_EXIST)
                  continue;
               if((xfs = XLoadQueryFont (mainDisplay, &font_name[1])) != NULL)
                     break;
               else if(policy == LOAD_EXIST)
                  continue;
               if(fonts_matched)
                  XFreeFontNames(fonts_matched);
            }
            fonts_matched = XListFonts(mainDisplay, font_name,
                  MAX_FONTS_MATCHED, &font_num);
            if (fonts_matched &&
               (xfs = XLoadQueryFont (mainDisplay, font_name)) != NULL)
                  break;
         }
      }
   }

   if(fonts_matched != NULL)
      XFreeFontNames(fonts_matched);

   if (!watch_cursor && mainWindow!=None)
   {
      SetDefaultCursor (mainWindow);
      ShowCursor ();
   }

   if (xfs == NULL)
   {
      if (strstr (kanjiFontFamilies[font_index].font_name, "%d") != NULL)
         sprintf (font_name, kanjiFontFamilies[font_index].font_name, size);
      else
         sprintf (font_name, "%s-%s-*-%1d-*-*-*-*-*-%s", fontNamePrefix,
             kanjiFontFamilies[font_index].font_name, size,
             kanjiFontFamilies[font_index].font_encoding);
 
      sprintf (msg, "Can not open '%s'!", font_name);
      TwoLineMsg (msg, "    Draw boxes instead of Japanese chars!");

      faked_size = size;
   }
 
   if (faked_size != size &&
       !((prev_fs && prev_fs->faked_size == faked_size) ||
       !((prev_fs && prev_fs->faked_size == faked_size) ||
         (fs_ptr && fs_ptr->faked_size == faked_size))))
   {
      kfs = (KanjiFontStruct *)calloc(1,sizeof(KanjiFontStruct));
      kfs->xfs = xfs;
      kfs->ascent = faked_size * 88 / 102;
      kfs->descent = faked_size - kfs->ascent;
      kfs->width = faked_size;
      kfs->vert = kanjiFontFamilies[font_index].vert;
      kfs->bold = kanjiFontFamilies[font_index].bold;

      new_fs_ptr = (struct KanjiFontSizeRec *)
                     calloc(1,sizeof(struct KanjiFontSizeRec));
      new_fs_ptr->next = fs_ptr;
      new_fs_ptr->kfs = kfs;
      new_fs_ptr->size = faked_size;
      new_fs_ptr->faked_size = faked_size;
      if (prev_fs == NULL)
         kanjiFontFamilies[font_index].fr = new_fs_ptr;
      else
         prev_fs->next = new_fs_ptr;
 
      if(faked_size < size)
         prev_fs = new_fs_ptr;
      else
         fs_ptr = new_fs_ptr;
   }

   kfs = (KanjiFontStruct *)calloc(1,sizeof(KanjiFontStruct));
   kfs->xfs = xfs;
   kfs->ascent = size * 88 / 102;
   kfs->descent = size - kfs->ascent;
   kfs->width = size;
   kfs->vert = kanjiFontFamilies[font_index].vert;
   kfs->bold = kanjiFontFamilies[font_index].bold;

   new_fs_ptr = (struct KanjiFontSizeRec *)
                      calloc(1,sizeof(struct KanjiFontSizeRec));
   new_fs_ptr->next = fs_ptr;
   new_fs_ptr->kfs = kfs;
   new_fs_ptr->size = size;
   new_fs_ptr->faked_size = faked_size;
   if (prev_fs == NULL)
      kanjiFontFamilies[font_index].fr = new_fs_ptr;
   else
      prev_fs->next = new_fs_ptr;

   return (new_fs_ptr->kfs);
}

int ValidCharCode (c_ptr)
   char	* c_ptr;
{
   register int	index = (int)(*c_ptr - (char)'\200');

   if (*charCodeToName[index] == '\0' || *charCodeToName[index] == '8' ||
         *charCodeToName[index] == '\\')
      return (TRUE);
   else
   {
      sprintf (gszMsgBox,
         "Unrecognized character code \\%o.  Character discarded!",
         (*c_ptr)&0xff);
      Msg (gszMsgBox);
      return (FALSE);
   }
}

char * CharCodeTranslate (c_ptr)
   char	* c_ptr;
{
   register int	index = (int)(*c_ptr - (char)'\200');

   if (*charCodeToName[index] == '\\' || *charCodeToName[index] == '8')
      return (charCodeToName[index]);
   else
      return (NULL);
}

static
void RemoveIllegalChars (TextPtr)
   struct TextRec	* TextPtr;
{
   register char	* c_ptr, * dest_c_ptr;
   struct StrRec	* str_ptr;

   for (str_ptr = TextPtr->last; str_ptr != NULL; str_ptr = str_ptr->prev)
   {
      for (c_ptr = dest_c_ptr = str_ptr->s; *c_ptr != '\0'; c_ptr++)
         if (!(((*c_ptr) & 0x80) && !ValidCharCode (c_ptr)))
            *dest_c_ptr++ = *c_ptr;
      *dest_c_ptr = '\0';
   }
}

static
void CheckLength (TextPtr)
   struct TextRec     * TextPtr;
{
   register char      * c_ptr, * dest_c_ptr;
   struct StrRec      * str_ptr;
   register           len;

   for (str_ptr = TextPtr->last; str_ptr != NULL; str_ptr = str_ptr->prev)
   {
      len = 0;
      for (c_ptr = dest_c_ptr = str_ptr->s; *c_ptr != '\0'; c_ptr++)
      if (*c_ptr & 0x80) {
         len ++;
         if ((*(c_ptr + 1) & 0x80) || (len & 1) == 0)
            *dest_c_ptr++ = *c_ptr;
      }
      else {
         len = 0;
         *dest_c_ptr++ = *c_ptr;
      }
      *dest_c_ptr = '\0';
   }
}

static int	* encodeFont=NULL;
static short	* * encodeCharFlags=NULL;
static int	encodeCharFlagsAllocated=FALSE;
static int	numEncodeCharFonts=0;

static
void CleanUpEncodeCharFonts ()
{
   register int	i;

   if (encodeCharFlagsAllocated)
   {
      for (i = 0; i < numEncodeCharFonts*MAXFONTSTYLES; i++)
         cfree (encodeCharFlags[i]);
      cfree (encodeFont);
      cfree (encodeCharFlags);
      encodeCharFlagsAllocated = FALSE;
      numEncodeCharFonts = 0;
      encodeFont = NULL;
      encodeCharFlags = NULL;
   }
}

static
void PrepareText (TextPtr)
   register struct TextRec	* TextPtr;
{
   register char	* c_ptr;
   struct StrRec	* str_ptr;
   int			index, byte_index;
   short		* flag_ptr;

   if ( TextPtr->kanji_font != KANJI_FONT_NONE )
   {
      PSUseKanjiText();
      return ;
   }

   if (TextPtr->font_name == NULL)
   {
      if (TextPtr->font == FONT_SYM )
	 return;
   }
   else if (strncmp (TextPtr->font_name, "Symbol", 6) == 0 ||
	 strchr(TextPtr->font_name, ' ') != NULL)
      return;

   index = TextPtr->font*MAXFONTSTYLES + TextPtr->style;
   for (str_ptr = TextPtr->last; str_ptr != NULL; str_ptr = str_ptr->prev)
   {
      for (c_ptr = str_ptr->s; *c_ptr != '\0'; c_ptr++)
         if (((*c_ptr)&0x80) &&
             *charCodeToName[(int)(*c_ptr-(char)'\200')]=='8')
         {
            encodeFont[index] = TRUE;
            flag_ptr = encodeCharFlags[index];
            byte_index = (int)(((*c_ptr) & 0x7f) / 8);
            flag_ptr[byte_index] |= (1<<(((*c_ptr) & 0x7f) % 8));
         }
   }
}

static
void PrepareObjFontInfo (ObjPtr)
   register struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;

   for (obj_ptr = ObjPtr; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      switch (obj_ptr->type)
      {
         case OBJ_TEXT: PrepareText (obj_ptr->detail.t); break;

         case OBJ_GROUP:
         case OBJ_SYM:
         case OBJ_ICON: PrepareObjFontInfo (obj_ptr->detail.r->last); break;
      }
      for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
         PrepareText (attr_ptr->obj->detail.t);
   }
}

void PrepareEightBitFontInfo ()
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;
   register int			j, i;
   short			* flag_ptr;
   struct PageRec		* page_ptr;

   if (encodeCharFlagsAllocated && numEncodeCharFonts < numFonts)
      CleanUpEncodeCharFonts ();

   if (!encodeCharFlagsAllocated)
   {
      int font_count=0;

      font_count = (PRTGIF ? MAXFONTS+numFakedFonts : numFonts+numFakedFonts);

      encodeCharFlags = (short **) calloc (font_count*MAXFONTSTYLES,
            sizeof(short*));
      encodeFont = (int *) calloc (font_count*MAXFONTSTYLES, sizeof(int));
      for (i = 0; i < font_count*MAXFONTSTYLES; i++)
         encodeCharFlags[i] = (short *) calloc (16, sizeof(short));
      encodeCharFlagsAllocated = TRUE;
      numEncodeCharFonts = font_count;
   }

   for (i = 0; i < numEncodeCharFonts*MAXFONTSTYLES; i++)
   {
      encodeFont[i] = FALSE;
      flag_ptr = encodeCharFlags[i];
      for (j = 0; j < 16; j++) flag_ptr[j] = 0;
   }

   for (page_ptr = firstPage; page_ptr != NULL; page_ptr = page_ptr->next)
      for (obj_ptr = page_ptr->bot; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      {
         switch (obj_ptr->type)
         {
            case OBJ_TEXT: PrepareText (obj_ptr->detail.t); break;

            case OBJ_GROUP:
            case OBJ_SYM:
            case OBJ_ICON: PrepareObjFontInfo (obj_ptr->detail.r->last); break;
         }
         for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
            PrepareText (attr_ptr->obj->detail.t);
      }
}

int NeedEncode (FontIndex, StyleIndex)
   int	FontIndex, StyleIndex;
{
   if (FontIndex == FONT_SYM)
      return (FALSE);
   else
      return (encodeFont[FontIndex*MAXFONTSTYLES+StyleIndex]);
}

void GetPSFontStr (FontIndex, StyleIndex, FontStr)
   int	FontIndex, StyleIndex;
   char	* FontStr;
{
   if (PRTGIF)
   {
      if (FontIndex < MAXFONTS)
         sprintf (FontStr, "/%s",
               initFontInfoStr[InfoIndex(FontIndex,StyleIndex)+2]);
      else
         sprintf (FontStr, "/%s", fontFamilies[FontIndex].name_faked);
   }
   else
   {
      if (FontIndex < numFonts)
         sprintf (FontStr, "/%s",
               fontInfoStr[InfoIndex(FontIndex,StyleIndex)+2]);
      else
         sprintf (FontStr, "/%s", fontFamilies[FontIndex].name_faked);
   }
}

void GetPSKanjiFontStr (KanjiFontIndex, FontStr)
   int	KanjiFontIndex;
   char	* FontStr;
{
   if(KanjiFontIndex == KANJI_FONT_NONE)
      *FontStr = '\0';
   else if (PRTGIF) {
      if (KanjiFontIndex < MAX_KANJI_FONTS)
	 sprintf (FontStr, "%s",
	       initKanjiFontFamilies[KanjiFontIndex].ps_font_name);
      else
	 sprintf (FontStr, "%s",
	       kanjiFontFamilies[KanjiFontIndex].ps_font_name);
   }
   else
      sprintf (FontStr, "%s", kanjiFontFamilies[KanjiFontIndex].ps_font_name);
}

int GetFontIndex (FontStr, StyleIndex, MustFind)
   char	* FontStr;
   int	StyleIndex;
   int	MustFind;
{
   register int	i;
   char	font_str[MAXSTRING];
   char	* c_ptr;

   strcpy(font_str, FontStr);
   if((c_ptr = strchr(font_str, '%')) != NULL)
      *c_ptr = '\0';

   if (PRTGIF)
   {
      for (i=0; i < MAXFONTS; i++)
         if (strcmp (initFontInfoStr[(i*MAXFONTSTYLES+StyleIndex)*3+2],
               font_str) == 0)
            return (i);

      for ( ; i < MAXFONTS+numFakedFonts; i++)
         if (strcmp (fontFamilies[i].name_faked, font_str) == 0)
            return (i);
   }
   else
   {
      for (i=0; i < numFonts; i++)
         if (strcmp (fontInfoStr[(i*MAXFONTSTYLES+StyleIndex)*3+2],
               font_str) == 0)
            return (i);

      for ( ; i < numFonts+numFakedFonts; i++)
         if (strcmp (fontFamilies[i].name_faked, font_str) == 0)
            return (i);
   }
   if (MustFind) return (INVALID);

   if (PRTGIF)
   {
      numFakedFonts++;
      if (fontFamilies == NULL)
         fontFamilies = (struct FontFmlyRec *) calloc (MAXFONTS+numFakedFonts,
               sizeof(struct FontFmlyRec));
      else
         fontFamilies = (struct FontFmlyRec *) realloc (fontFamilies,
               (MAXFONTS+numFakedFonts)*sizeof(struct FontFmlyRec));
      fontFamilies[MAXFONTS+numFakedFonts-1].name_faked =
            (char *) calloc (strlen(font_str)+1, sizeof(char));
      strcpy (fontFamilies[MAXFONTS+numFakedFonts-1].name_faked, font_str);
      for (i=0; i<MAXFONTSTYLES; i++)
         fontFamilies[MAXFONTS+numFakedFonts-1].fr[i] = NULL;
      return (MAXFONTS+numFakedFonts-1);
   }
   else
   {
      numFakedFonts++;
      if (fontFamilies == NULL)
         fontFamilies = (struct FontFmlyRec *) calloc (numFonts+numFakedFonts,
               sizeof(struct FontFmlyRec));
      else
         fontFamilies = (struct FontFmlyRec *) realloc (fontFamilies,
               (numFonts+numFakedFonts)*sizeof(struct FontFmlyRec));
      fontFamilies[numFonts+numFakedFonts-1].name_faked =
            (char *) calloc (strlen(font_str)+1, sizeof(char));
      strcpy (fontFamilies[numFonts+numFakedFonts-1].name_faked, font_str);
      for (i=0; i<MAXFONTSTYLES; i++)
         fontFamilies[numFonts+numFakedFonts-1].fr[i] = NULL;
      return (numFonts++);
   }
}

int GetKanjiFontIndex (FontStr, MustFind)
   char * FontStr;
   int  MustFind;
{
   register int i;
   char * font_str;

   if((font_str = strchr(FontStr, '%')) != NULL)
      font_str ++;
   else
      return KANJI_FONT_NONE;

   if (PRTGIF) {
      for (i=1 ; i < MAX_KANJI_FONTS; i++)
         if (strcmp (initKanjiFontFamilies[i].ps_font_name, font_str) == 0)
            return (i);

      for ( ; i < MAX_KANJI_FONTS + numFakedKanjiFonts; i++)
         if (strcmp (kanjiFontFamilies[i].ps_font_name, font_str) == 0)
            return (i);
   }
   else {
      for (i=1 ; i < numKanjiFonts; i++)
         if (strcmp (kanjiFontFamilies[i].ps_font_name, font_str) == 0)
            return (i);
   }
   if (MustFind) return (INVALID);

   numFakedKanjiFonts++;
   numKanjiFonts++;
   if (kanjiFontFamilies == NULL)
      kanjiFontFamilies = (struct KanjiFontFmlyRec *)calloc (numKanjiFonts,
            sizeof(struct KanjiFontFmlyRec));
   else
      kanjiFontFamilies = (struct KanjiFontFmlyRec *)realloc (kanjiFontFamilies,
            numKanjiFonts*sizeof(struct KanjiFontFmlyRec));
   kanjiFontFamilies[numKanjiFonts-1].ps_font_name =
         (char *) calloc (strlen(font_str)+1, sizeof(char));
   strcpy (kanjiFontFamilies[numKanjiFonts-1].ps_font_name, font_str);
   kanjiFontFamilies[numKanjiFonts-1].fr = NULL;
   kanjiFontFamilies[numKanjiFonts-1].specify = NULL;
   return (numKanjiFonts);
}

struct eucFontRec {
   char                 * name;
   struct eucFontRec    * next;
};

static struct eucFontRec        * eucFontTable = NULL;

void PrepareEucFontTable()
{
   struct eucFontRec    * cur, * next;

   for(cur = eucFontTable; cur != NULL; cur = next) {
      next = cur->next;
      cfree(cur);
   }
   eucFontTable = NULL;
}

int IsDefinedEucFont(name)
char    *name;
{
   struct eucFontRec    * cur;

   for(cur = eucFontTable; cur != NULL; cur = cur->next)
      if(!strcmp(cur->name, name))
         return TRUE;

   return FALSE;
}

void DefineEucFont(name)
char    *name;
{
   struct eucFontRec    * new;

   new = (struct eucFontRec *)calloc(1, sizeof(struct eucFontRec) +
                                strlen(name) +1 );
   new->name = (char *)&new[1];
   strcpy(new->name, name);

   new->next = eucFontTable;
   eucFontTable = new;
}

void CurFontMsg ()
{
   char	msg[MAXSTRING], style[20];

   switch (curStyle)
   {
      case STYLE_BI: strcpy (style, "BoldItalic"); break;
      case STYLE_BR: strcpy (style, "Bold"); break;
      case STYLE_NI: strcpy (style, "Italic"); break;
      case STYLE_NR: strcpy (style, "Roman"); break;
   }

   sprintf (msg,"%s-%s-%s-%1dpt",fontMenuStr[curFont],style,fontMenuStr[curKanjiFont+numFonts],curSize);
   Msg (msg);
}

void DumpEightBitFontInfo (FP)
   FILE	* FP;
{
   register int		k, j;
   register short	* flag_ptr, flag;
   int			font_index, style_index;
   char			font_str[MAXSTRING];
   int			font_count;

   font_count = (PRTGIF ? MAXFONTS+numFakedFonts : numFonts+numFakedFonts);

   for (font_index = 0; font_index < font_count; font_index++)
   {
      for (style_index = 0; style_index < MAXFONTSTYLES; style_index++)
      {
         if (NeedEncode (font_index, style_index))
         {
            GetPSFontStr (font_index, style_index, font_str);
            fprintf (FP, "%s-vec [\n", font_str);
            flag_ptr = encodeCharFlags[font_index*MAXFONTSTYLES+style_index];
            for (j = 0; j < 16; j++, flag_ptr++)
            {
               flag = *flag_ptr;
               if ((flag & 0xff) != 0)
               {
                  for (k = 0; k < 8; k++)
                     if (flag & (1<<k))
                        fprintf (FP, " %s\n", charCodeToName[(j<<3)+k]);
               }
            }
            fprintf (FP, " ] def\n");
            fprintf (FP, "%s %s-8 %s-vec tgifReEncodeSmall\n\n", font_str,
                  font_str, &font_str[1]);
            if (preDumpSetup) PSUseReencode(font_str);
         }
      }
   }
}

static int	pointSize75[] = { 8, 10, 12, 14, 18, 24 };
static int	pointSize100[] = { 11, 14, 17, 20, 25, 34 };

int GetCompatibleSize (font_dpi, font_size)
   int	font_dpi, font_size;
{
   switch (font_dpi)
   {
      case FONT_DPI_75: return (pointSize75[font_size]);
      case FONT_DPI_100: return (pointSize100[font_size]);
   }
   return (INVALID);
}

void SetCanvasFont ()
{
   struct FontSizeRec	* fs_ptr;
   KanjiFontStruct	* kanji_font;

   fs_ptr = FindFontInfo (curFont, curStyle, curSize);

   if (fs_ptr == NULL) {
      fs_ptr = FindFontInfo(defaultCurFont, defaultCurStyle, defaultCurSize);
   }
   if (fs_ptr == NULL)
   {
      canvasFontSize = INVALID;
      return;
   }

   canvasFontPtr = fs_ptr->xfs;
   canvasFontSize = fs_ptr->faked_size;

   if ( canvasFontSize <= 0 ) { /* Tsuchida <taiji@ap.isl.melco.co.jp> */
      canvasFontSize = INVALID ;
      return;
   }

   kanji_font = FindKanjiFontInfo (curKanjiFont, canvasFontSize);

   if ((canvasKanjiFontPtr = kanji_font) != NULL) {
      if(canvasKanjiFontPtr->xfs != NULL)
         cacheForKanjiFont = (canvasKanjiFontPtr->xfs->max_bounds.width !=
               canvasFontSize) || canvasKanjiFontPtr->vert;
      else
         cacheForKanjiFont = FALSE;

      canvasFontAsc = max (canvasFontPtr->max_bounds.ascent,
                           canvasKanjiFontPtr->ascent);
      canvasFontDes = max (canvasFontPtr->max_bounds.descent,
                           canvasKanjiFontPtr->descent);
   }
   else {
      cacheForKanjiFont = FALSE;

      canvasFontAsc = canvasFontPtr->max_bounds.ascent;
      canvasFontDes = canvasFontPtr->max_bounds.descent;
   }

   canvasFontHeight = canvasFontAsc + canvasFontDes;

   XSetFont (mainDisplay, drawGC, canvasFontPtr->fid);
   if(canvasKanjiFontPtr && canvasKanjiFontPtr->xfs)
      XSetFont (mainDisplay, drawKanjiGC, canvasKanjiFontPtr->xfs->fid);

   textCursorH = canvasFontHeight;
   curHeight =  canvasFontHeight * curSize / canvasFontSize;
   curAsc = canvasFontAsc * curSize / canvasFontSize;
   curDes = curHeight - curAsc;

   fakeTextVSpace = textVSpace * canvasFontSize / curSize;
}

struct TmpFontFmlyRec {
   char				* * font_strings;
   struct TmpFontFmlyRec	* next, * prev;
};

void InitFonts ()
{
   register int		i, j;
   int			fmly_index, style_index;
   int			len, ruler_font_size;
   char			* c_ptr, * buf;
   struct FontSizeRec	* fs_ptr;

   debugScalableFonts = FALSE;
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"DebugScalableFonts")) != NULL)
      if (strcmp ("True", c_ptr) == 0 || strcmp ("true", c_ptr) == 0)
         debugScalableFonts = TRUE;

   defaultFontSize = DEFAULT_FONT_SIZE;
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"DefaultFontSize")) != NULL)
   {
      if ((defaultFontSize = atoi (c_ptr)) <= 0)
      {
         defaultFontSize = DEFAULT_FONT_SIZE;
         fprintf (stderr, "%s '%s*DefaultFontSize'.  %1d used.\n",
               "Warning:  Error in processing", TOOL_NAME, DEFAULT_FONT_SIZE);
      }
   }

   strcpy (fontNamePrefix, "-*");
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"FontNamePrefix")) != NULL)
      strcpy (fontNamePrefix, c_ptr);

   fontSizes = NULL;
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"FontSizes")) != NULL)
   {
      char	* size_ptr;
      int	allocated=50, ok=TRUE;

      numFontSizes = 0;
      fontSizes = (int *) calloc (allocated, sizeof(int));
      sizeMenuStr = (char **) calloc (allocated, sizeof(char*));
      len = strlen (c_ptr);
      buf = (char *) calloc (len+1, sizeof(char));
      strcpy (buf, c_ptr);
      if ((size_ptr=strtok(buf," ,\t\n\r")) != NULL)
      {
         while (ok && size_ptr != NULL)
         {
            int	size=atoi(size_ptr);

            if (size <= 0 || size>=1000)
            {
               ok = FALSE;
               break;
            }
            if (numFontSizes >= allocated)
            {
               allocated += 50;
               fontSizes = (int *) realloc (fontSizes, allocated*sizeof(int));
               sizeMenuStr = (char **) realloc (sizeMenuStr,
                     allocated*sizeof(char*));
            }
            sizeMenuStr[numFontSizes] = (char*)calloc(4,sizeof(char));
            sprintf (sizeMenuStr[numFontSizes], "%-3d", size);
            fontSizes[numFontSizes++] = size;
            size_ptr = strtok (NULL, " ,\t\n\r");
         }
      }
      cfree (buf);
      if (!ok || numFontSizes <= 0)
      {
         for (i=0; i < numFontSizes; i++) cfree (sizeMenuStr[i]);
         cfree (sizeMenuStr);
         sizeMenuStr = NULL;
         cfree (fontSizes);
         fontSizes = NULL;
         numFontSizes = 0;
         fprintf (stderr, "%s '%s*FontSizes'.  %s.\n",
               "Warning:  Error in processing", TOOL_NAME,
               "Default font sizes used");
      }
   }
   if (fontSizes == NULL)
   {
      numFontSizes = (MAXFONTSIZES<<1)-1;
      fontSizes = (int *) calloc (numFontSizes, sizeof(int));
      sizeMenuStr = (char **) calloc (numFontSizes, sizeof(char*));
      for (i=0; i<numFontSizes; i++)
      {
         fontSizes[i] = atoi(initSizeMenuStr[i]);
         sizeMenuStr[i] = (char *) calloc (4, sizeof(char));
         strcpy (sizeMenuStr[i], initSizeMenuStr[i]);
      }
   }

   hasAlternateDefaultFonts = FALSE;
   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME,
         "HasAlternateDefaultFonts")) != NULL)
      if ((strcmp (c_ptr, "true") == 0) || (strcmp (c_ptr, "True") == 0))
         hasAlternateDefaultFonts = TRUE;

   fontFamilies = NULL;
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"AdditionalFonts")) != NULL)
   {
      char	* font_ptr;

      len = strlen (c_ptr);
      buf = (char *) calloc (len+1, sizeof(char));
      strcpy (buf, c_ptr);
      for (font_ptr=buf; *font_ptr != '\0' &&
            strchr(" ,\t\n\r",*font_ptr) != NULL; font_ptr++) ;
      if (font_ptr != NULL && *font_ptr != '\0')
      {
         int	ok=TRUE, num_new_fonts=0, index;
         char	* * font_strings;
         struct TmpFontFmlyRec	* first_fmly=NULL, * last_fmly=NULL, * fmly_ptr;

         while (ok && font_ptr != NULL && *font_ptr != '\0')
         {
            font_strings = (char * *) calloc (MAXFONTSTYLES*3, sizeof(char *));
            for (i=0, index=0; i<MAXFONTSTYLES; i++)
            {
               char	* rgstry, * ps_name;

               if ((font_ptr=strtok(font_ptr,",\t\n\r")) != NULL &&
                     (rgstry=strtok(NULL,",\t\n\r")) != NULL &&
                     (ps_name=strtok(NULL," ,\t\n\r")) != NULL)
               {
                  font_strings[index] = (char *) calloc (81, sizeof(char));
                  strcpy (font_strings[index], font_ptr);
                  UtilTrimBlanks (font_strings[index++]);
                  font_strings[index] = (char *) calloc (81, sizeof(char));
                  strcpy (font_strings[index], rgstry);
                  UtilTrimBlanks (font_strings[index++]);
                  font_strings[index] = (char *) calloc (81, sizeof(char));
                  strcpy (font_strings[index], ps_name);
                  UtilTrimBlanks (font_strings[index++]);
                  for (font_ptr=(&ps_name[strlen(ps_name)+1]);
                        *font_ptr != '\0' && 
                        strchr(" ,\t\n\r",*font_ptr) != NULL; font_ptr++) ;
               }
               else
               {
                  ok = FALSE;
                  break;
               }
            }
            if (ok)
            {
               num_new_fonts++;
               fmly_ptr = (struct TmpFontFmlyRec *) calloc(1,
                     sizeof(struct TmpFontFmlyRec));
               fmly_ptr->next = NULL;
               fmly_ptr->prev = last_fmly;
               fmly_ptr->font_strings = font_strings;
               if (last_fmly == NULL)
                  first_fmly = fmly_ptr;
               else
                  last_fmly->next = fmly_ptr;
               last_fmly = fmly_ptr;
            }
            if (font_ptr-buf >= len) break;
         }
         if (!ok)
         {
            fprintf (stderr, "%s '%s*AdditionalFonts'.\n",
                  "Warning:  Error in processing", TOOL_NAME);
         }
         else
         {
            struct TmpFontFmlyRec	* next_fmly;

            numFonts = MAXFONTS+num_new_fonts;
            fontFamilies = (struct FontFmlyRec *) calloc (numFonts,
                  sizeof(struct FontFmlyRec));
            fontInfoStr = (char **) calloc (numFonts*MAXFONTSTYLES*3,
                  sizeof(char *));
            fontMenuStr = (char **) calloc (numFonts, sizeof(char *));
            index = MAXFONTS*MAXFONTSTYLES*3;
            for (fmly_ptr=first_fmly, fmly_index=MAXFONTS; fmly_ptr != NULL;
                  fmly_ptr=next_fmly, fmly_index++)
            {
               char	s[81];

               next_fmly = fmly_ptr->next;
               strcpy (s, fmly_ptr->font_strings[2]);
               if ((c_ptr=strchr(s,'-')) != NULL) *c_ptr='\0';
               fontMenuStr[fmly_index] = (char *) calloc (strlen(s)+1,
                     sizeof(char));
               strcpy (fontMenuStr[fmly_index], s );
               for (i=0, j=0; i<MAXFONTSTYLES; i++)
               {
                  fontInfoStr[index++] = fmly_ptr->font_strings[j++];
                  fontInfoStr[index++] = fmly_ptr->font_strings[j++];
                  fontInfoStr[index++] = fmly_ptr->font_strings[j++];
               }
               cfree (fmly_ptr->font_strings);
               cfree (fmly_ptr);
               fontFamilies[fmly_index].fr[STYLE_NR] =
                     fontFamilies[fmly_index].fr[STYLE_BR] =
                     fontFamilies[fmly_index].fr[STYLE_NI] =
                     fontFamilies[fmly_index].fr[STYLE_BI] = NULL;
            }
         }
      }
      cfree (buf);
   }
   if (fontFamilies == NULL)
   {
      fontFamilies = (struct FontFmlyRec *) calloc (numFonts,
            sizeof(struct FontFmlyRec));
      fontInfoStr = (char **) calloc (numFonts*MAXFONTSTYLES*3, sizeof(char*));
      fontMenuStr = (char **) calloc (numFonts, sizeof(char*));
   }
   altFontInfoStr = (char **) calloc (MAXFONTS*MAXFONTSTYLES*3, sizeof(char*));
   for (j=0; j<MAXFONTS*MAXFONTSTYLES*3; j++) altFontInfoStr[j] = NULL;
   i = 0;
   fmly_index = 0;
   for (fmly_index=0; fmly_index < MAXFONTS; fmly_index++)
   {
      char	s[81];

      strcpy (s, initFontInfoStr[i+2]);
      if ((c_ptr=strchr(s,'-')) != NULL) *c_ptr='\0';
      fontMenuStr[fmly_index] = (char *) calloc (strlen(s)+1, sizeof(char));
      strcpy (fontMenuStr[fmly_index], s );
      for (style_index=0; style_index < MAXFONTSTYLES; style_index++)
      {
         if (hasAlternateDefaultFonts)
         {
            for (j=0; j<3; j++, i++)
            {
               if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME,
                     initFontInfoStr[i+2])) != NULL)
               {
                  if (strstr(c_ptr, "%d") != NULL)
                  {
                     len = strlen (c_ptr);
                     altFontInfoStr[i] = (char *) calloc (len+1, sizeof(char));
                     strcpy(altFontInfoStr[i], c_ptr);
                     fontInfoStr[i] = altFontInfoStr[i];
                  }
                  else
                  {
                     fprintf (stderr,
                           "%s '%s*%s'.\n\t'%s-%s-*-%%d-*-*-*-*-*-%s' used.\n",
                           "Warning:  Error in processing",
                           TOOL_NAME, initFontInfoStr[i+2],
                           fontNamePrefix, initFontInfoStr[i],
                           initFontInfoStr[i+1]);
                     fontInfoStr[i] = initFontInfoStr[i];
                  }
               }
               else
                  fontInfoStr[i] = initFontInfoStr[i];
            }
         }
         else
         {
            fontInfoStr[i] = initFontInfoStr[i]; i++;
            fontInfoStr[i] = initFontInfoStr[i]; i++;
            fontInfoStr[i] = initFontInfoStr[i]; i++;
         }
      }
      fontFamilies[fmly_index].fr[STYLE_NR] =
            fontFamilies[fmly_index].fr[STYLE_BR] =
            fontFamilies[fmly_index].fr[STYLE_NI] =
            fontFamilies[fmly_index].fr[STYLE_BI] = NULL;
   }

   kanjiFontFamilies = NULL;
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"NihongoFonts")) != NULL)
   {
      int	i;
      int	ok = TRUE;
      int	size;
      char	* str1, * str2, * str3;

      len = strlen (c_ptr);
      buf = (char *) calloc (len+2, sizeof(char));
      strcpy (buf, c_ptr);
      buf[len] = '\n';
      buf[len+1] = '\0';
      numKanjiFonts = 1;			/* for KANJI_FONT_NONE */
      for(c_ptr = buf; (c_ptr = strchr(c_ptr, '\n')) != NULL; c_ptr ++) {
	 for(;c_ptr[1] == '\n'; c_ptr ++);
	 numKanjiFonts ++;
      }

      if(numKanjiFonts > 1) {
	 kanjiFontFamilies = (struct KanjiFontFmlyRec *)calloc(numKanjiFonts,
	       sizeof(struct KanjiFontFmlyRec));
	 i = 1;					/* for KANJI_FONT_NONE */
	 c_ptr = strtok(buf, "\n");
	 do {
	    kanjiFontFamilies[i++].font_name = c_ptr;
	 } while ((c_ptr = strtok(NULL, "\n")) != NULL);

	 for(i = 1; ok && i < numKanjiFonts; i ++) {
	    for(c_ptr = kanjiFontFamilies[i].font_name; *c_ptr != '\0' &&
		  strchr(" \t\n\r", *c_ptr) != NULL; c_ptr ++);
	    str1 = c_ptr;
	    for(; *c_ptr != '\0' && *c_ptr != ','; c_ptr ++);
	    for(; c_ptr != str1 && strchr(" ,\t\n\r", *c_ptr) != NULL; c_ptr--);
	    if(c_ptr != str1) {
	       c_ptr ++;
	       *c_ptr = '\0';
	       kanjiFontFamilies[i].font_name = str1;
	       c_ptr ++;
	    }
	    else {
	       ok = FALSE;
	       break;
	    }
	    for(; *c_ptr != '\0' &&
	          strchr(" \t\n\r", *c_ptr) != NULL; c_ptr ++);
	    str1 = c_ptr;
	    for(; *c_ptr != '\0' && *c_ptr != ','; c_ptr ++);
	    for(; c_ptr != str1 && strchr(" ,\t\n\r", *c_ptr) != NULL; c_ptr--);
	    if(strchr(" ,\t\n\r", *c_ptr) == NULL)
	       c_ptr ++;
	    *c_ptr = '\0';
	    kanjiFontFamilies[i].font_encoding = str1;
	    c_ptr ++;
	    if((str1 = strtok(c_ptr, " ,\t\n\r")) != NULL &&
	       (str2 = strtok(NULL, " ,\t\n\r")) != NULL) {
	       kanjiFontFamilies[i].ps_font_name = str1;
	       kanjiFontFamilies[i].menu_str = str2;
	    }
	    else {
	       ok = FALSE;
	       break;
	    }
	    if((str1 = strtok(NULL, " ,\t\n\r")) != NULL &&
	       (str2 = strtok(NULL, " ,\t\n\r")) != NULL &&
	       (str3 = strtok(NULL, " ,\t\n\r")) != NULL) {
	       if (*str1 == 'V' || *str1 == 'v')
		  kanjiFontFamilies[i].vert = TRUE;
	       else
		  kanjiFontFamilies[i].vert = FALSE;

	       if (!strcasecmp(str2, "true"))
		  kanjiFontFamilies[i].bold = TRUE;
	       else
		  kanjiFontFamilies[i].bold = FALSE;

	       if (!strcasecmp(str3, "all"))
		  kanjiFontFamilies[i].policy = LOAD_ALL;
	       else if (!strcasecmp(str3, "exist"))
		  kanjiFontFamilies[i].policy = LOAD_EXIST;
	       else
		  kanjiFontFamilies[i].policy = LOAD_SPECIFY;
	    }
	    else {
	       ok = FALSE;
	       break;
	    }
	    if(kanjiFontFamilies[i].policy == LOAD_SPECIFY) {
	       int	size_array = 0;
	       int	* array = NULL;
	       int	num_use = 0;

	       ok = FALSE;
	       do {
		  size = atoi(str3);
		  if(size > 0 && size <= MAX_FONT_SIZE)
		     ok = TRUE;
		  else
		     continue;
		  if(array == NULL) {
		     size_array = 5;
		     array = (int *)calloc(size_array, sizeof(int));
		  }
		  else if(num_use >= size_array) {
		     size_array += 5;
		     array = (int *)realloc(array, size_array * sizeof(int));
		  }
		  array[num_use] = size;
		  num_use ++;
	       } while ((str3 = strtok(NULL, " ,\t\n\r")) != NULL);
	       kanjiFontFamilies[i].num_specify = num_use;
	       kanjiFontFamilies[i].specify = array;
	    }
	 }
      }
      else
	 ok = FALSE;

      if(ok) 
	 kanjiFontFamilies[0].dummy = buf;
      else {
	 int	i;

	 fprintf (stderr, "%s '%s*NihongoFonts'.\n",
	       "Warning:  Error in processing", TOOL_NAME);
	 for(i = 1; i < numKanjiFonts; i ++) {
	    if(kanjiFontFamilies[i].specify != NULL)
	       cfree(kanjiFontFamilies[i].specify);
	 }
	 cfree(kanjiFontFamilies);
	 kanjiFontFamilies = NULL;
	 numKanjiFonts = 0;
	 cfree(buf);
      }
   }
   if(kanjiFontFamilies == NULL) {
      int	i;
      int	size;
      char	* str3;

      numKanjiFonts = MAX_KANJI_FONTS;

      kanjiFontFamilies = (struct KanjiFontFmlyRec *)calloc(numKanjiFonts,
	    sizeof(struct KanjiFontFmlyRec));
      for(i = 1; i < numKanjiFonts; i ++) {
	 kanjiFontFamilies[i].font_name = initKanjiFontFamilies[i].font_name;
	 kanjiFontFamilies[i].font_encoding =
	       initKanjiFontFamilies[i].font_encoding;
	 kanjiFontFamilies[i].ps_font_name =
	       initKanjiFontFamilies[i].ps_font_name;
	 kanjiFontFamilies[i].menu_str = initKanjiFontFamilies[i].menu_str;
	 kanjiFontFamilies[i].vert = initKanjiFontFamilies[i].vert;
	 kanjiFontFamilies[i].bold = initKanjiFontFamilies[i].bold;
	 kanjiFontFamilies[i].policy = initKanjiFontFamilies[i].policy;

	 if(kanjiFontFamilies[i].policy == LOAD_SPECIFY) {
	    int	size_array = 0;
	    int	* array = NULL;
	    int	num_use = 0;

	    str3 = strtok(initKanjiFontFamilies[i].dummy, " ,\t\n\r");
	    do {
	       size = atoi(str3);
	       if(size <= 0 || size > MAX_FONT_SIZE)
		  continue;
	       if(array == NULL) {
		  size_array = 5;
		  array = (int *)calloc(size_array, sizeof(int));
	       }
	       else if(num_use >= size_array) {
		  size_array += 5;
		  array = (int *)realloc(array, size_array * sizeof(int));
	       }
	       array[num_use] = size;
	       num_use ++;
	    } while ((str3 = strtok(NULL, " ,\t\n\r")) != NULL);
	    kanjiFontFamilies[i].num_specify = num_use;
	    kanjiFontFamilies[i].specify = array;
	 }
      }
   }
   if (kanjiFontFamilies) {
      int	i;
      char	*s;

      kanjiFontFamilies[0].menu_str = initKanjiFontFamilies[0].menu_str;

      fontMenuStr = (char **) realloc (fontMenuStr,
	    (numFonts + numKanjiFonts) * sizeof(char *));
      for(i = numFonts; i < numFonts + numKanjiFonts; i ++) {
	 s = kanjiFontFamilies[i - numFonts].menu_str;
	 fontMenuStr[i] = (char *) calloc (strlen(s)+1, sizeof(char));
	 strcpy (fontMenuStr[i], s );
      }
   }

   curFont = FONT_COU;
   curKanjiFont = KANJI_FONT_NONE;
   curStyle = STYLE_NR;
   curSize = 17;

   ruler_font_size = 10;
   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME, "RulerFontSize")) != NULL)
   {
      ruler_font_size = atoi (c_ptr);
      if (ruler_font_size <= 0)
      {
         ruler_font_size = defaultFontSize;
         fprintf (stderr, "%s '%s', '%1d' is used.\n",
               "Warning:  can not set RulerFontSize to", c_ptr,
               ruler_font_size);
      }
   }
   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME, "InitialFontDPI")) != NULL)
   {
      fprintf (stderr, "Obsoleted %s*InitialFontDPI used.\n%s%s%s.\n",
            TOOL_NAME, "\tPlease use ", TOOL_NAME, "*InitialFontSize instead");
   }
   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME, "MsgFontSizeIndex")) != NULL)
   {
      fprintf (stderr, "Obsoleted %s*MsgFontSizeIndex used.\n%s%s%s.\n",
            TOOL_NAME, "\tPlease use ", TOOL_NAME, "*MsgFontSize instead");
   }
   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME, "MsgFontSize")) != NULL)
   {
      curSize = atoi (c_ptr);
      if (curSize <= 0)
      {
         curSize = defaultFontSize;
         fprintf (stderr, "%s '%s', '%1d' is used.\n",
               "Warning:  can not set MsgFontSize to", c_ptr, curSize);
      }
   }
   exitIfErrorInFindFontInfo = TRUE;
   fs_ptr = FindFontInfo (curFont, curStyle, curSize);
   exitIfErrorInFindFontInfo = FALSE;
   if (fs_ptr == NULL) {
      sprintf (gszMsgBox, "Can not open the Default(Msg)Font '%s'!  Abort!",
            gszAttemptedFontName);
      Error ("OpenFont()", gszMsgBox);
   }

   defaultFontPtr = fs_ptr->xfs;
   defaultFontWidth = defaultFontPtr->max_bounds.width;
   defaultFontAsc = defaultFontPtr->max_bounds.ascent;
   defaultFontDes = defaultFontPtr->max_bounds.descent;
   defaultFontHeight = defaultFontAsc + defaultFontDes;

   if (defaultFontWidth < 1 || defaultFontHeight < 1) {
      fprintf(stderr, "%s (%1d/%1d).\n\t%s.\n\t%s.\n",
            "Warning:  Very small default font width/height",
            defaultFontWidth, defaultFontHeight,
            "Possibly a problem with the font path",
            "Set default font width to 9 and height to 14 as a temporary fix");
      defaultFontWidth = 9;
      defaultFontHeight = 14;
   }

   exitIfErrorInFindFontInfo = TRUE;
   fs_ptr = FindFontInfo (curFont, curStyle, ruler_font_size);
   exitIfErrorInFindFontInfo = FALSE;
   if (fs_ptr == NULL) {
      sprintf (gszMsgBox, "Can not open the RulerFont '%s'!  Abort!",
            gszAttemptedFontName);
      Error ("OpenFont()", gszMsgBox);
   }

   rulerFontPtr = fs_ptr->xfs;
   rulerFontWidth = rulerFontPtr->max_bounds.width;
   rulerFontAsc = rulerFontPtr->max_bounds.ascent;
   rulerFontDes = rulerFontPtr->max_bounds.descent;
   rulerFontHeight = rulerFontAsc + rulerFontDes;

   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME, "InitialFont")) != NULL)
   {
      for (i = 0; i < numFonts; i++)
         if (strcmp (c_ptr, fontMenuStr[i]) == 0)
            break;
      if (i != numFonts)
         curFont = i;
      else
         fprintf (stderr, "Warning:  can not set InitialFont to '%s'\n", c_ptr);
   }
   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME, "InitialKanjiFont")) != NULL)
   {
      for (i = 1; i < numKanjiFonts; i++)
         if (strcmp (c_ptr, fontMenuStr[i + numFonts]) == 0)
            break;
      if (i != numKanjiFonts)
         curKanjiFont = i;
      else
         fprintf (stderr, "Warning:  can not set InitialKanjiFont to '%s'\n", c_ptr);
   }
   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME, "InitialFontStyle")) != NULL)
   {
      len = strlen (c_ptr);
      if (len < ((int) strlen (styleMenuStr[0])))
      {
         for (i = 0; i < MAXFONTSTYLES; i++)
            if (strncmp (c_ptr, styleMenuStr[i], len) == 0)
               break;
         if (i != MAXFONTSTYLES)
            curStyle = i;
         else
            fprintf (stderr, "%s '%s'\n",
                  "Warning:  can not set InitialFontStyle to", c_ptr);
      }
      else
         fprintf (stderr, "Warning:  can not set InitialFontStyle to '%s'\n",
               c_ptr);
   }
   if ((c_ptr=XGetDefault (mainDisplay, TOOL_NAME, "InitialFontJust")) != NULL)
   {
      len = strlen (c_ptr);
      if (len < ((int) strlen (styleMenuStr[0])))
      {
         for (i = MAXFONTSTYLES+1; i < MAXFONTSTYLES+1+MAXJUSTS; i++)
            if (strncmp (c_ptr, styleMenuStr[i], len) == 0)
               break;
         if (i != MAXFONTSTYLES+1+MAXJUSTS)
            textJust = i-MAXFONTSTYLES-1;
         else
            fprintf (stderr, "%s '%s'\n",
                  "Warning:  can not set InitialFontJust to", c_ptr);
      }
      else
         fprintf (stderr, "Warning:  can not set InitialFontJust to '%s'\n",
               c_ptr);
   }
   if ((c_ptr=XGetDefault (mainDisplay,TOOL_NAME,"InitialFontSizeIndex"))!=NULL)
   {
      fprintf (stderr, "Obsoleted %s*InitialFontSizeIndex used.\n%s%s%s.\n",
            TOOL_NAME, "\tPlease use ", TOOL_NAME, "*InitialFontSize instead");
   }
   if ((c_ptr=XGetDefault (mainDisplay,TOOL_NAME,"InitialFontSize"))!=NULL)
   {
      curSize = atoi (c_ptr);
      if (curSize <= 0)
      {
         curSize = defaultFontSize;
         fprintf (stderr, "%s '%s'\n",
               "Warning:  can not set InitialFontSize to", c_ptr);
      }
   }

   if ((c_ptr=XGetDefault (mainDisplay,TOOL_NAME,"ScaleRotateJapanese"))!=NULL)
      if ((strcmp (c_ptr, "false") == 0) || (strcmp (c_ptr, "False") == 0))
	 drawScaledKanji = FALSE;

   InitRotTbl();
}

int ChangeObjTextStyle (ObjPtr, StyleIndex)
   register struct ObjRec	* ObjPtr;
   register int			StyleIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->style != StyleIndex)
         {
            int	rc, saved_style=ObjPtr->detail.t->style;
            int	saved_faked=ObjPtr->detail.t->faked;

            ObjPtr->detail.t->style = StyleIndex;
            rc = UpdTextBBox (ObjPtr);
            switch (rc)
            {
               case INVALID:
                  ObjPtr->detail.t->style = saved_style;
                  ObjPtr->detail.t->faked = saved_faked;
                  UpdTextBBox (ObjPtr);
                  break;
               case FALSE:
                  Msg ("Invalid vertical spacing for a text object.");
                  Msg ("    That object's vertical spacing reset to 0.");
                  ObjPtr->detail.t->v_space = 0;
                  UpdTextBBox (ObjPtr);
	       default:
		  if(ObjPtr->detail.t->cached_bitmap != None)
		     XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);
		  ObjPtr->detail.t->cached_zoom = 0;
		  ObjPtr->detail.t->cached_bitmap = None;
		  changed = TRUE;
                  break;
            }
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextStyle (obj_ptr, StyleIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontStyle (StyleIndex)
   int	StyleIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby, text_obj_created;
   int				changed = FALSE, text_cursor_shown;
   int				saved_style=curStyle;
   char				msg[MAXSTRING+1];

   if (StyleIndex == INVALID) return;

   if (topSel == NULL || stickyMenuSelection)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curStyle = StyleIndex;
      SetCanvasFont ();
      if (canvasFontSize == INVALID)
      {
         char	msg[80];

         sprintf (msg, "%s-%s-%s %1d not available.", fontMenuStr[curFont],
	       styleStr[curStyle], fontMenuStr[curKanjiFont+numFonts], curSize);
         Dialog (msg, NULL, NULL);
         curStyle = saved_style;
	 /* StyleIndex = saved_style; */
         SetCanvasFont ();
      }
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_STYLE);
      if (topSel == NULL) return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextStyle (sel_ptr->obj, StyleIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

int ChangeObjTextJust (ObjPtr, JustIndex)
   register struct ObjRec	* ObjPtr;
   register int			JustIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->just != JustIndex)
         {
            ObjPtr->detail.t->just = JustIndex;
            UpdTextBBox (ObjPtr);

            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextJust (obj_ptr, JustIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontJust (JustIndex)
   int JustIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby;
   int				text_obj_created, text_cursor_shown;
   int				changed = FALSE;

   if (JustIndex == INVALID) return;

   if (topSel == NULL || stickyMenuSelection)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      textJust = JustIndex;
      ShowJust ();
      UpdateSubMenu (MENU_STYLE);
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      if (topSel == NULL) return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextJust (sel_ptr->obj, JustIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

static void FreeKanjiTextCache (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* ptr;
   register struct AttrRec	* attr_ptr;
   struct TextRec		* text_ptr;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
	 text_ptr = ObjPtr->detail.t;

	 if (text_ptr->kanji_font != KANJI_FONT_NONE) {
	    if (text_ptr->cached_bitmap != None)
	       XFreePixmap (mainDisplay, text_ptr->cached_bitmap);
	    text_ptr->cached_zoom = 0;
	    text_ptr->cached_bitmap = None;
	 }
         break;

      case OBJ_SYM:
      case OBJ_ICON:
      case OBJ_GROUP:
         for (ptr = ObjPtr->detail.r->first; ptr != NULL; ptr = ptr->next)
            FreeKanjiTextCache (ptr);
         break;
   }
   for (attr_ptr=ObjPtr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
      if (attr_ptr->shown)
         FreeKanjiTextCache (attr_ptr->obj);
}

static void FreeKanjiTextCaches ()
{
   register struct ObjRec	* obj_ptr, * ptr;
   register struct AttrRec	* attr_ptr;

   if (topObj == NULL) return;

   for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
   {
      switch (obj_ptr->type)
      {
         case OBJ_TEXT:
            FreeKanjiTextCache (obj_ptr);
            break;
         case OBJ_SYM:
         case OBJ_ICON:
         case OBJ_GROUP:
            for (ptr = obj_ptr->detail.r->first; ptr != NULL; ptr = ptr->next)
               FreeKanjiTextCache (ptr);
            break;
      }
      for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
         if (attr_ptr->shown)
            FreeKanjiTextCache (attr_ptr->obj);
   }
}

void StyleSubMenu (index)
   int	index;
{
   if (index < MAXFONTSTYLES)
      ChangeFontStyle (index);
   else if (index > MAXFONTSTYLES && index < MAXFONTSTYLES+1+MAXJUSTS)
      ChangeFontJust (index - MAXFONTSTYLES - 1);
   else if (index == MAXFONTSTYLES+1+MAXJUSTS+1) {
      drawScaledKanji = (drawScaledKanji)?FALSE:TRUE;
      FreeKanjiTextCaches();
      UpdateSubMenu (MENU_STYLE);
      ClearAndRedrawDrawWindow ();
   }
}

static char * styleMenuDescription[] =
{
   "Roman text style",
   "Bold text style",
   "Italic text style",
   "Bold-italic text style",
   "",
   "Left justified text",
   "Center justified text",
   "Right justified text",
   "",
   "Scale/Rotate Japanese",      /* Yuuki Harano */
   NULL
};

int StyleMenu (X, Y, TrackMenubar)
   int  X, Y, TrackMenubar;
{
   register int	index;
   int		* fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXFONTSTYLES+MAXJUSTS+3, &fore_colors, &valid,
         &init_rv, NULL);

   init_rv[curStyle] = TRUE;
   init_rv[MAXFONTSTYLES+1+textJust] = TRUE;
   init_rv[MAXFONTSTYLES+1+MAXJUSTS+1] = drawScaledKanji;
   activeMenu = MENU_STYLE;
   index = TextMenuLoop (X, Y, styleMenuStr, MAXFONTSTYLES+MAXJUSTS+3,
         fore_colors, valid, init_rv, styleMenuDescription, SINGLECOLOR,
         TrackMenubar);

   if (index >= 0 ) StyleSubMenu (index);
   return (index);
}

int ChangeObjTextSize (ObjPtr, SizeIndex)
   register struct ObjRec	* ObjPtr;
   register int			SizeIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;
   int				size=fontSizes[SizeIndex];

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->size != size)
         {
            int	rc, saved_size=ObjPtr->detail.t->size;
            int	saved_faked=ObjPtr->detail.t->faked;

            ObjPtr->detail.t->size = size;
            rc = UpdTextBBox (ObjPtr);
            switch (rc)
            {
               case INVALID:
                  ObjPtr->detail.t->size = saved_size;
                  ObjPtr->detail.t->faked = saved_faked;
                  UpdTextBBox (ObjPtr);
                  break;
               case FALSE:
                  Msg ("Invalid vertical spacing for a text object.");
                  Msg ("    That object's vertical spacing reset to 0.");
                  ObjPtr->detail.t->v_space = 0;
                  UpdTextBBox (ObjPtr);
	       default:
		  if(ObjPtr->detail.t->cached_bitmap != None)
		     XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);
		  ObjPtr->detail.t->cached_zoom = 0;
		  ObjPtr->detail.t->cached_bitmap = None;
		  changed = TRUE;
                  break;
            }
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextSize (obj_ptr, SizeIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontSize (SizeIndex)
   int	SizeIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby, text_obj_created;
   int				text_cursor_shown, changed=FALSE;
   int				saved_size=curSize;
   char				msg[MAXSTRING+1];

   if (SizeIndex == INVALID) return;

   if (topSel == NULL || stickyMenuSelection)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curSize = fontSizes[SizeIndex];
      SetCanvasFont ();
      if (canvasFontSize == INVALID)
      {
         char	msg[80];

         sprintf (msg, "%s-%s-%s %1d not available.", fontMenuStr[curFont],
              styleStr[curStyle], fontMenuStr[curKanjiFont+numFonts], curSize);
         MsgBox (msg, TOOL_NAME, INFO_MB);
         curSize = saved_size;
         SizeIndex = GetSizeMenuIndex ();
         SetCanvasFont ();
      }
      else if (curSize != canvasFontSize)
      {
         sprintf (msg, "Can not change size to %1d.  %1d used.",
               curSize, canvasFontSize);
         MsgBox (msg, TOOL_NAME, INFO_MB);
         curSize = canvasFontSize;
         SizeIndex = GetSizeMenuIndex ();
      }
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_SIZE);
      if (topSel == NULL) return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextSize (sel_ptr->obj, SizeIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

int GetSizeMenuIndex ()
{
   register int	i;

   for (i=0; i < numFontSizes; i++)
      if (fontSizes[i] == curSize) return i;
   return (INVALID);
}

int SizeMenu (X, Y, TrackMenubar)
   int  X, Y, TrackMenubar;
{
   register int	i;
   int		index, * fore_colors, * valid, * init_rv;
   char		* * desc=(char **) calloc (numFontSizes+1, sizeof(char*));

   if (desc == NULL) fprintf (stderr, "Can not calloc().\n");
   for (i=0; i < numFontSizes; i++)
   {
      desc[i] = (char*) calloc (80, sizeof(char));
      if (desc[i] == NULL) fprintf (stderr, "Can not caloc().\n");
      sprintf (desc[i], "Set font size to %1d", fontSizes[i]);
   }
   desc[i] = NULL;
   DefaultColorArrays (numFontSizes, &fore_colors, &valid, &init_rv, NULL);

   index = GetSizeMenuIndex ();
   if (index >= 0) init_rv[index] = TRUE;
   activeMenu = MENU_SIZE;
   index = TextMenuLoop (X, Y, sizeMenuStr, numFontSizes, fore_colors,
         valid, init_rv, desc, SINGLECOLOR, TrackMenubar);

   if (index >= 0) ChangeFontSize (index);
   if (desc != NULL)
   {
      for (i=0; i < numFontSizes; i++)
         if (desc[i] != NULL) cfree (desc[i]);
      cfree (desc);
   }
   return (index);
}

int ChangeObjTextFont (ObjPtr, FontIndex)
   register struct ObjRec	* ObjPtr;
   register int			FontIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->font != FontIndex)
         {
            int	rc, saved_font=ObjPtr->detail.t->font;
	    int saved_faked=ObjPtr->detail.t->faked;

            if (ObjPtr->detail.t->font == FONT_SYM &&
		ObjPtr->detail.t->kanji_font == KANJI_FONT_NONE)
               RemoveIllegalChars (ObjPtr->detail.t);
            ObjPtr->detail.t->font = FontIndex;
            rc = UpdTextBBox (ObjPtr);
            switch (rc)
            {
               case INVALID:
                  ObjPtr->detail.t->font = saved_font;
		  ObjPtr->detail.t->faked = saved_faked;
                  UpdTextBBox (ObjPtr);
                  break;
               case FALSE:
                  Msg ("Invalid vertical spacing for a text object.");
                  Msg ("    That object's vertical spacing reset to 0.");
                  ObjPtr->detail.t->v_space = 0;
                  UpdTextBBox (ObjPtr);
	       default:
		  if(ObjPtr->detail.t->cached_bitmap != None)
		     XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);
		  ObjPtr->detail.t->cached_zoom = 0;
		  ObjPtr->detail.t->cached_bitmap = None;
		  changed = TRUE;
                  break;
            }
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextFont (obj_ptr, FontIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFont (FontIndex)
   int	FontIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby, text_obj_created;
   int				text_cursor_shown, changed=FALSE;
   int				saved_font=curFont;

   if (FontIndex == INVALID) return;

   if (topSel == NULL || stickyMenuSelection)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curFont = FontIndex;
      SetCanvasFont ();
      if (canvasFontSize == INVALID)
      {
         char   msg[80];

         sprintf (msg, "%s-%s-%s %1d not available.", fontMenuStr[curFont],
              styleStr[curStyle], fontMenuStr[curKanjiFont+numFonts], curSize);
         MsgBox (msg, TOOL_NAME, INFO_MB);
         curFont = FontIndex = saved_font;
         SetCanvasFont ();
      }
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_FONT);
      if (topSel == NULL) return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextFont (sel_ptr->obj, FontIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

int ChangeObjTextKanjiFont (ObjPtr, KanjiFontIndex)
   register struct ObjRec	* ObjPtr;
   register int			KanjiFontIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->kanji_font != KanjiFontIndex)
         {
            int	rc, saved_font=ObjPtr->detail.t->kanji_font;
	    int saved_faked=ObjPtr->detail.t->faked;

            if (ObjPtr->detail.t->kanji_font != KANJI_FONT_NONE &&
		ObjPtr->detail.t->font != FONT_SYM &&
		KanjiFontIndex == KANJI_FONT_NONE)
               RemoveIllegalChars (ObjPtr->detail.t);
	    if (ObjPtr->detail.t->kanji_font == KANJI_FONT_NONE)
	       CheckLength (ObjPtr->detail.t);
            ObjPtr->detail.t->kanji_font = KanjiFontIndex;
            rc = UpdTextBBox (ObjPtr);
            switch (rc)
            {
               case INVALID:
                  ObjPtr->detail.t->kanji_font = saved_font;
		  ObjPtr->detail.t->faked = saved_faked;
                  UpdTextBBox (ObjPtr);
                  break;
               case FALSE:
                  Msg ("Invalid vertical spacing for a text object.");
                  Msg ("    That object's vertical spacing reset to 0.");
                  ObjPtr->detail.t->v_space = 0;
                  UpdTextBBox (ObjPtr);
	       default:
		  if(ObjPtr->detail.t->cached_bitmap != None)
		     XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);
		  ObjPtr->detail.t->cached_zoom = 0;
		  ObjPtr->detail.t->cached_bitmap = None;
		  changed = TRUE;
                  break;
            }
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextKanjiFont (obj_ptr, KanjiFontIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeKanjiFont (KanjiFontIndex)
   int	KanjiFontIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby, text_obj_created;
   int				text_cursor_shown, changed=FALSE;
   int				saved_font=curKanjiFont;

   if (KanjiFontIndex == INVALID) return;

   if (topSel == NULL || stickyMenuSelection)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curKanjiFont = KanjiFontIndex;
      SetCanvasFont ();
      if (canvasFontSize == INVALID)
      {
         char   msg[80];

         sprintf (msg, "%s-%s-%s %1d not available.", fontMenuStr[curFont],
	       styleStr[curStyle], fontMenuStr[curKanjiFont+numFonts], curSize);
         Dialog (msg, NULL, NULL);
         curKanjiFont = KanjiFontIndex = saved_font;
         SetCanvasFont ();
      }
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      /* There is no need for update Panel (Choice) Window
      ShowCurFont ();
      ShowTextSize ();
      */
      UpdateSubMenu (MENU_FONT);
      if (topSel == NULL) return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextKanjiFont (sel_ptr->obj, KanjiFontIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

void FontSubMenu (index)
   int	index;
{
   if (index < numFonts)
      ChangeFont (index);
   else if(index >= numFonts && index < numFonts+numKanjiFonts)
      ChangeKanjiFont (index - numFonts);
}


int FontMenu (X, Y, TrackMenubar)
   int  X, Y, TrackMenubar;
{
   register int	i;
   int		index, * fore_colors, * valid, * init_rv;
   char		* * desc=(char **) calloc (numFonts+numKanjiFonts+1, sizeof(char*));

   if (desc == NULL) fprintf (stderr, "Can not calloc().\n");
   for (i=0; i < numFonts+numKanjiFonts; i++)
   {
      desc[i] = (char*) calloc (80, sizeof(char));
      if (desc[i] == NULL) fprintf (stderr, "Can not caloc().\n");
      sprintf (desc[i], "Set font to '%s'", fontMenuStr[i]);
   }
   desc[i] = NULL;
   DefaultColorArrays ( numFonts + numKanjiFonts,
                             &fore_colors, &valid, &init_rv, NULL);

   init_rv[curFont] = TRUE;
   init_rv[curKanjiFont + numFonts] = TRUE;
   activeMenu = MENU_FONT;
   index = TextMenuLoop (X, Y, fontMenuStr, numFonts + numKanjiFonts,
                fore_colors, valid, init_rv, desc, SINGLECOLOR, TrackMenubar);

   if (index >= 0)
         FontSubMenu (index);
   if (desc != NULL)
   {
      for (i=0; i < numFonts+numKanjiFonts; i++)
         if (desc[i] != NULL) cfree (desc[i]);
      cfree (desc);
   }
   return (index);
}

int ChangeObjVSpace (ObjPtr, VSpace)
   register struct ObjRec	* ObjPtr;
   register int			VSpace;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;
   int				saved_v_space;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->v_space != VSpace)
         {
            saved_v_space = ObjPtr->detail.t->v_space;
            ObjPtr->detail.t->v_space = VSpace;
            if (UpdTextBBox (ObjPtr) == FALSE)
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("Vertical spacing for that object not changed.");
               ObjPtr->detail.t->v_space = saved_v_space;
               UpdTextBBox (ObjPtr);
            }
            else
            {
               if (ObjPtr->detail.t->cached_bitmap != None)
                  XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

               ObjPtr->detail.t->cached_zoom = 0;
               ObjPtr->detail.t->cached_bitmap = None;

               changed = TRUE;
            }
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjVSpace (obj_ptr, VSpace))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeVSpace (VSpace)
   int	VSpace;
{
   register struct SelRec	* sel_ptr;
   int				changed = FALSE, text_cursor_shown;
   int				ltx, lty, rbx, rby, text_obj_created;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      if (curHeight+textVSpace <= 0)
         Msg ("Text vertical spacing too small.  No change.");
      else
      {
         textVSpace = VSpace;
         SetCanvasFont ();
         ShowTextVSpace ();
      }
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjVSpace (sel_ptr->obj, VSpace))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

static int	savedFont, savedKanjiFont, savedSize, savedStyle;
static int	savedJust, savedRotate, savedPen, savedFill, savedVSpace;

void SaveCurFont ()
{
   savedFont = curFont;
   savedKanjiFont = curKanjiFont;
   savedSize = curSize;
   savedStyle = curStyle;
   savedJust = textJust;
   savedRotate = curRotate;
   savedPen = penPat;
   savedFill = objFill;
   savedVSpace = textVSpace;
}

void RestoreCurFont ()
{
   curFont = savedFont;
   curKanjiFont = savedKanjiFont;
   curSize = savedSize;
   curStyle = savedStyle;
   textJust = savedJust;
   curRotate = savedRotate;
   penPat = savedPen;
   objFill = savedFill;
   textVSpace = savedVSpace;

   SetCanvasFont ();
}

static int	pushedFont, pushedKanjiFont, pushedSize, pushedStyle, pushedJust;
static int	pushedColorIndex, pushedRotate, pushedPen, pushedVSpace;
static int	pushedFill;

void PushCurFont ()
{
   pushedFont = curFont;
   pushedKanjiFont = curKanjiFont;
   pushedSize = curSize;
   pushedStyle = curStyle;
   pushedJust = textJust;
   pushedRotate = curRotate;
   pushedPen = penPat;
   pushedFill = objFill;
   pushedVSpace = textVSpace;
   pushedColorIndex = colorIndex;
}

void PopCurFont ()
{
   curFont = pushedFont;
   curKanjiFont = pushedKanjiFont;
   curSize = pushedSize;
   curStyle = pushedStyle;
   textJust = pushedJust;
   curRotate = pushedRotate;
   penPat = pushedPen;
   objFill = pushedFill;
   textVSpace = pushedVSpace;
   colorIndex = pushedColorIndex;

   SetCanvasFont ();
}

void CleanUpFonts ()
{
   register int		i;
   int			fmly_index, style_index;
   struct FontSizeRec	* fs_ptr, * next_fs;
   struct KanjiFontSizeRec	* kfs_ptr, * next_kfs;

   for (i=0; i<numFontSizes; i++) cfree (sizeMenuStr[i]);
   cfree (sizeMenuStr);
   sizeMenuStr = NULL;

   cfree (fontSizes);
   fontSizes = NULL;
   numFontSizes = 0;

   CleanUpEncodeCharFonts ();
   PrepareEucFontTable ();

   for (fmly_index=0; fmly_index<numFonts; fmly_index++)
   {
      for (style_index=0; style_index<MAXFONTSTYLES; style_index++)
      {
         for (fs_ptr=fontFamilies[fmly_index].fr[style_index];
               fs_ptr != NULL; fs_ptr = next_fs)
         {
            next_fs = fs_ptr->next;
            if (fs_ptr->xfs != NULL && fs_ptr->size == fs_ptr->faked_size)
               XFreeFont (mainDisplay, fs_ptr->xfs);
            cfree (fs_ptr);
         }
      }
   }
   for (fmly_index=0; fmly_index<numFakedFonts; fmly_index++)
      cfree (fontFamilies[fmly_index+numFonts].name_faked);
   cfree (fontFamilies);
   fontFamilies = NULL;

   for (fmly_index=0; fmly_index<numKanjiFonts; fmly_index++)
   {
      for (kfs_ptr=kanjiFontFamilies[fmly_index].fr;
	    kfs_ptr != NULL; kfs_ptr = next_kfs)
      {
	 next_kfs = kfs_ptr->next;
	 if(kfs_ptr->kfs != NULL) {
	    if (kfs_ptr->kfs->xfs != NULL &&
		  kfs_ptr->size == kfs_ptr->faked_size)
	       XFreeFont (mainDisplay, kfs_ptr->kfs->xfs);
	    cfree(kfs_ptr->kfs);
	 }
	 cfree (kfs_ptr);
      }
      if(kanjiFontFamilies[fmly_index].specify != NULL)
	 cfree(kanjiFontFamilies[fmly_index].specify);
   }
   for (fmly_index=1; fmly_index<=numFakedKanjiFonts; fmly_index++)
      cfree (kanjiFontFamilies[numKanjiFonts-fmly_index].ps_font_name);
   if(kanjiFontFamilies[0].dummy)
      cfree(kanjiFontFamilies[0].dummy);
   cfree (kanjiFontFamilies);
   kanjiFontFamilies = NULL;

   for (i=MAXFONTS*MAXFONTSTYLES*3; i<numFonts*MAXFONTSTYLES*3; i++)
      cfree (fontInfoStr[i]);
   if (altFontInfoStr)
   {
      for (i=0; i<MAXFONTS*MAXFONTSTYLES*3; i++)
         if (altFontInfoStr[i] != NULL) cfree (altFontInfoStr[i]);
      cfree (altFontInfoStr);
      altFontInfoStr = NULL;
   }
   cfree (fontInfoStr);
   fontInfoStr = NULL;

   for (fmly_index=0; fmly_index<numFonts; fmly_index++)
      cfree (fontMenuStr[fmly_index]);
   cfree (fontMenuStr);
   fontMenuStr = NULL;

   numFonts = MAXFONTS;
   numFakedFonts = 0;
}

/* The following procedure is used to generate pdrawFontAsc[] and */
/*    pDrawFontDes[], to be used in ``prtgif.c''.  It is supposed */
/*    to be called within dbx and not tgif.                       */

/*
 * static
 * void GenerateFontInfo ()
 * {
 *    register int	i, j, num_rows;
 * 
 *    for (i = 0; i < FONTTABLESIZE; i++)
 *       if (!myFontInfo[i].valid)
 *       {
 *          if ((myFontInfo[i].xfs =
 *                XLoadQueryFont (mainDisplay, fontNameStr[i])) == NULL)
 *          {
 *             printf ("Can not open %s.  Abort.\n\n", fontNameStr[i]);
 *             exit (-1);
 *          }
 *          myFontInfo[i].valid = TRUE;
 *       }
 * 
 *    num_rows = FONTTABLESIZE / MAXFONTSIZES;
 *    printf ("short\tpDrawFontAsc[] =\n{\n");
 *    for (i = 0; i < num_rows; i++)
 *    {
 *       printf ("   ");
 *       for (j = 0; j < MAXFONTSIZES; j++)
 *          if (i == num_rows-1 && j == MAXFONTSIZES-1)
 *             printf ("%2d ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.ascent);
 *          else
 *             printf ("%2d, ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.ascent);
 *       printf ("\n");
 *    }
 *    printf ("};\n\n");
 * 
 *    printf ("short\tpDrawFontDes[] =\n{\n");
 *    for (i = 0; i < num_rows; i++)
 *    {
 *       printf ("   ");
 *       for (j = 0; j < MAXFONTSIZES; j++)
 *          if (i == num_rows-1 && j == MAXFONTSIZES-1)
 *             printf ("%2d ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.descent);
 *          else
 *             printf ("%2d, ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.descent);
 *       printf ("\n");
 *    }
 *    printf ("};\n");
 * }
 */

int
XuTextWidth(xfs, kfs, str, len)
XFontStruct *	xfs;
KanjiFontStruct *	kfs;
char *	str;
int	len;
{
   int	width = 0;
   register char *	head;
   register char *	tail;
   register char	check;
   register int		mask = 0x7f;
   register int		i;
   XChar2b		str16[MAXSTRING / 2];

   head = str;
   check = *head & 0x80;
   if(kfs == NULL || kfs->xfs == NULL ||
      (kfs->xfs->min_byte1 & 0x80) || (kfs->xfs->max_byte1 & 0x80))
      mask = 0xff;

   while(head - str < len) {
      for(tail = head, i = 0; tail - str < len && (*tail & (char)0x80) == check; tail ++, i ++) {
	 if(i & 1)
	    str16[i >> 1].byte2 = (*tail & mask);
	 else
	    str16[i >> 1].byte1 = (*tail & mask);
      }
      if(check == 0x00 || kfs == NULL) {
	 width += XTextWidth(xfs, head, tail - head);
      }
      else if(kfs->xfs == NULL) {
	 width += kfs->width * ((tail - head) >> 1);
      }
      else if(kfs->xfs->max_bounds.width == kfs->width) {
	 width += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1);
      }
      else {
	 width += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1) *
	       kfs->width / kfs->xfs->max_bounds.width;
      }
      head = tail;
      check = *head & 0x80;
   }

   return width;
}

static void
AllocTextBitmapAndScaleBuf(display, d, width, height)
Display		*display;
Drawable	d;
int		width;
int		height;
{
   if(height > textBitmapHeight || width > textBitmapWidth) {
      if(textBitmapHeight != INVALID && textBitmapWidth != INVALID) {
	 XFreePixmap(display, textBitmap);
	 free(scaleBuf);
      }
      
      textBitmapHeight = height;
      textBitmapWidth = width;
      textBitmap = XCreatePixmap(display, d,
	    textBitmapWidth, textBitmapHeight, 1);
      scaleBuf = (char *)malloc(textBitmapHeight * sizeof(char));
   }
}

static void
CreateBaseGC(display)
Display	*display;
{
   XGCValues		values;

   values.foreground = 1;
   values.background = 0;
   values.fill_style = FillSolid;
   values.function = GXcopy;
   baseGC = XCreateGC (display, textBitmap,
	 GCForeground | GCBackground | GCFillStyle | GCFunction, &values);
}

static void
InitRotTbl()
{
   int	i;

   for(i = 0; i < 256; i ++)
      RotTbl21XX[i] = ROT_NRM;

   RotTbl21XX[0x22] = ROT_MOV; RotTbl21XX[0x23] = ROT_MOV;

   RotTbl21XX[0x31] = ROT_NON; RotTbl21XX[0x32] = ROT_NON;
   RotTbl21XX[0x3c] = ROT_INV;
   RotTbl21XX[0x3d] = ROT_NON; RotTbl21XX[0x3e] = ROT_NON;

   RotTbl21XX[0x41] = ROT_INV;
   RotTbl21XX[0x42] = ROT_NON; RotTbl21XX[0x43] = ROT_NON;
   RotTbl21XX[0x44] = ROT_NON; RotTbl21XX[0x45] = ROT_NON;
   RotTbl21XX[0x4a] = ROT_NON; RotTbl21XX[0x4b] = ROT_NON;
   RotTbl21XX[0x4c] = ROT_NON; RotTbl21XX[0x4d] = ROT_NON;
   RotTbl21XX[0x4e] = ROT_NON; RotTbl21XX[0x4f] = ROT_NON;

   RotTbl21XX[0x50] = ROT_NON; RotTbl21XX[0x51] = ROT_NON;
   RotTbl21XX[0x52] = ROT_NON; RotTbl21XX[0x53] = ROT_NON;
   RotTbl21XX[0x54] = ROT_NON; RotTbl21XX[0x55] = ROT_NON;
   RotTbl21XX[0x56] = ROT_NON; RotTbl21XX[0x57] = ROT_NON;
   RotTbl21XX[0x58] = ROT_NON; RotTbl21XX[0x59] = ROT_NON;
   RotTbl21XX[0x5a] = ROT_NON; RotTbl21XX[0x5b] = ROT_NON;

   RotTbl21XX[0x61] = ROT_NON;
}

static void
RotChars(num_chars, from_image, to_image, size, str16)
int	num_chars;
XImage	*from_image;
XImage	*to_image;
int	size;
XChar2b	*str16;
{
   register int	i;
   register int	x1;
   register int	x2;
   register int	y2;
   register int size2;

   size2 = size >> 1;
   for(i = num_chars, x1 = 0; i > 0; i --, x1 += size, str16++) {
      switch(((str16->byte1 & 0x7f) == 0x21)
	    ? RotTbl21XX[str16->byte2 & 0x7f]
	    : ROT_NRM) {
	 case ROT_NON :
	    for(x2 = 0; x2 < size ; x2 ++) {
	       for(y2 = 0; y2 < size; y2 ++) {
		  if(XGetPixel(from_image, x2 + x1, y2))
		     XPutPixel(to_image, x2 + x1, y2, 1);
	       }
	    }
	    break;
	 case ROT_INV :
	    for(x2 = 0; x2 < size ; x2 ++) {
	       for(y2 = 0; y2 < size; y2 ++) {
		  if(XGetPixel(from_image, x2 + x1, y2))
		     XPutPixel(to_image, x2 + x1, size - y2, 1);
	       }
	    }
	    break;
	 case ROT_MOV:
	    for(x2 = 0; x2 < size ; x2 ++) {
	       for(y2 = 0; y2 < size; y2 ++) {
		  if(XGetPixel(from_image, x2 + x1, y2))
		     XPutPixel(to_image, (y2 + size2) % size + x1,
			   size - (x2 + size2) % size, 1);
	       }
	    }
	    break;
	 case ROT_NRM :
	 default :
	    for(x2 = 0; x2 < size ; x2 ++) {
	       for(y2 = 0; y2 < size; y2 ++) {
		  if(XGetPixel(from_image, x2 + x1, y2))
		     XPutPixel(to_image, y2 + x1, size - x2, 1);
	       }
	    }
	    break;
      }
   }
}

static void
ScaleUp(from_image, to_image, width, base_size, size)
XImage	*from_image;
XImage	*to_image;
int	width;
int	base_size;
int	size;
{
   register int	i;
   register int	j;
   register int	x1;
   register int	y1;
   register int	x2;
   register int	y2;

   i = -1;
   for(x2 = 0; x2 < width; x2 ++) {
      x1 = x2 * base_size / size;
      if(x1 != i) {
	 j = -1;
	 for(y2 = 0; y2 < size; y2 ++) {
	    y1= y2 * base_size / size;
	    if(y1 != j) {
	       if((scaleBuf[y2] = XGetPixel(from_image, x1, y1)) == 1)
		  XPutPixel(to_image, x2, y2, 1);
	       j = y1;
	    }
	    else {
	       if((scaleBuf[y2] = scaleBuf[y2 - 1]) == 1)
		  XPutPixel(to_image, x2, y2, 1);
	    }
	 }
	 i = x1;
      }
      else {
	 for(y2 = 0; y2 < size; y2 ++) {
	    if(scaleBuf[y2] == 1)
	       XPutPixel(to_image, x2, y2, 1);
	 }
      }
   }
}

static void
ScaleDown(from_image, to_image, width, base_size, size)
XImage	*from_image;
XImage	*to_image;
int	width;
int	base_size;
int	size;
{
   register int	x1;
   register int	y1;
   register int	x2;
   register int	y2;

   for(x2 = 0; x2 < width; x2 ++) {
      x1 = x2 * base_size / size;
      for(y2 = 0; y2 < size; y2 ++) {
	 y1= y2 * base_size / size;
	 if(XGetPixel(from_image, x1, y1) == 1)
	    XPutPixel(to_image, x2, y2, 1);
      }
   }
}

static int
DrawScaleRotateString16(display, d, kgc, x, y, str16, len, kfs, clipGC, flag)
Display		*display;
Drawable	d;
GC		kgc;
int		x;
int		y;
XChar2b		*str16;
int		len;
KanjiFontStruct	*kfs;
GC		*clipGC;
int		flag;
{
   int		base_size;
   int		size;
   int		base_width;
   int		width;
   XImage	*from_image;
   XImage	*via_image;
   XImage	*to_image;
   XGCValues	values;


   base_size = kfs->xfs->max_bounds.width;
   size = kfs->width;
   base_width = XTextWidth16(kfs->xfs, str16, len);
   width = base_width * size / base_size;

   AllocTextBitmapAndScaleBuf(display, d, max(width, base_width),
	 max(size, base_size));

   if(baseGC == NULL)
      CreateBaseGC(display);

   XSetForeground(display, baseGC, 0);
   XFillRectangle(display, textBitmap, baseGC, 0, 0,
	 textBitmapWidth, textBitmapHeight);
   if(flag == DRAW_FOR_DISPLAY && kfs->vert && size != base_size)
      via_image = XGetImage(display, textBitmap, 0, 0,
	    textBitmapWidth, textBitmapHeight, 1, ZPixmap);
   to_image = XGetImage(display, textBitmap, 0, 0,
	 textBitmapWidth, textBitmapHeight, 1, ZPixmap);
   
   values.foreground = 1;
   values.font = kfs->xfs->fid;
   XChangeGC(display, baseGC, GCForeground | GCFont, &values);

   XDrawString16(display, textBitmap, baseGC, 0, kfs->xfs->ascent, str16, len);
   if(kfs->bold)
      XDrawString16(display, textBitmap, baseGC, 1, kfs->xfs->ascent, str16, len);
   from_image = XGetImage(display, textBitmap, 0, 0,
	 textBitmapWidth, textBitmapHeight, 1, ZPixmap);

   if(size == base_size) {
      RotChars(len, from_image, to_image, size, str16);
   }
   else if(size > base_size) { /* Scale up */
      if(flag == DRAW_FOR_DISPLAY && kfs->vert) {
	 RotChars(len, from_image, via_image, base_size, str16);
	 ScaleUp(via_image, to_image, width, base_size, size);
      }
      else {
	 ScaleUp(from_image, to_image, width, base_size, size);
      }
   }
   else { /* Scale Down */
      if(flag == DRAW_FOR_DISPLAY && kfs->vert) {
	 ScaleDown(from_image, via_image, width, base_size, size);
	 RotChars(len, via_image, to_image, size, str16);
      }
      else {
	 ScaleDown(from_image, to_image, width, base_size, size);
      }
   }

   XPutImage(display, textBitmap, baseGC, to_image, 0, 0, 0, 0, width, size);

   if(*clipGC == NULL) {
      values.clip_x_origin = x;
      values.clip_y_origin = y - kfs->ascent;
      values.clip_mask = textBitmap;
      *clipGC = XCreateGC(display, d, 
	    GCClipMask | GCClipXOrigin | GCClipYOrigin, &values);
      XCopyGC(display, kgc,
	    ~0L & ~GCClipMask & ~GCClipXOrigin & ~GCClipYOrigin, *clipGC);
   }
   else {
      values.clip_x_origin = x;
      values.clip_y_origin = y - kfs->ascent;
      values.clip_mask = textBitmap;
      XChangeGC(display, *clipGC, GCClipMask | GCClipXOrigin | GCClipYOrigin,
	    &values);
   }

   XFillRectangle(display, d, *clipGC, x, y - kfs->ascent, width, size);

   XDestroyImage(from_image);
   if(flag == DRAW_FOR_DISPLAY && kfs->vert && size != base_size)
      XDestroyImage(via_image);
   XDestroyImage(to_image);

   return width;
}

void
XuDrawString (display, d, gc, kgc, x, y, str, len, xfs, kfs, flag)
Display *	display;
Drawable	d;
GC		gc;
GC		kgc;
int		x;
int		y;
char *		str;
int		len;
XFontStruct *	xfs;
KanjiFontStruct *	kfs;
int		flag;
{
   register char *	head;
   register char *	tail;
   register char	check;
   register int		mask = 0x7f;
   register int		i;
   XChar2b		str16[MAXSTRING / 2];
   XChar2b		* p16;
   int			size;
   int			x1;
   int			y1;
   GC			clipGC = NULL;

   head = str;
   check = *head & 0x80;
   if(kfs == NULL || kfs->xfs == NULL ||
      (kfs->xfs->min_byte1 & 0x80) || (kfs->xfs->max_byte1 & 0x80))
      mask = 0xff;

   while(head - str < len) {
      for(tail = head, i = 0; tail - str < len && (*tail & (char)0x80) == check; tail ++, i ++) {
	 if(i & 1)
	    str16[i >> 1].byte2 = (*tail & mask);
	 else
	    str16[i >> 1].byte1 = (*tail & mask);
      }

      if(check == 0x00 || kfs == NULL) {
	 /* ISO characters */
	 XDrawString(display, d, gc, x, y, head, tail - head);
	 x += XTextWidth(xfs, head, tail - head);
      }
      else if(kfs->xfs == NULL) {
	 /* JAPANESE characters, but no font */
	 size = kfs->width;
	 x1 = x;
	 y1 = y - kfs->ascent;
	 for(i = (tail - head) >> 1; i > 0; i --) {
	    XFillRectangle(display, d, kgc, x1 + 1, y1 + 1, size - 2, size - 2);
	    x1 += size;
	 }
	 x += size * ((tail - head) >> 1);
      }
      else if(kfs->xfs->max_bounds.width == kfs->width && !kfs->vert) {
	 /* JAPANESE characters, no scale or rotate */
	 y1 = y - kfs->ascent + kfs->xfs->ascent;
	 XDrawString16(display, d, kgc, x, y1, str16, (tail - head) >> 1);
	 if(kfs->bold)
	    XDrawString16(display, d, kgc, x+1, y1, str16, (tail - head) >> 1);
	 x += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1);
      }
      else if(!drawScaledKanji) {
	 /* JAPANESE characters, scale or rotate with black box */
	 size = kfs->width;
	 x1 = x + 1;
	 y1 = y - kfs->ascent + 1;
	 for(i = (tail - head) >> 1; i > 0; i --) {
	    XFillRectangle(display, d, kgc, x1, y1, size - 2, size - 2);
	    x1 += size;
	 }
	 x += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1) *
	       kfs->width / kfs->xfs->max_bounds.width;
      }
      else if(flag == DRAW_FOR_EDIT &&
	    kfs->xfs->max_bounds.width - 2 <= kfs->width) {
	 size = (kfs->width - kfs->xfs->max_bounds.width) / 2;
	 x1 = x + size;
	 y1 = y - kfs->ascent + size + kfs->xfs->ascent;
	 size = kfs->width;
	 for(i = (tail - head) >> 1, p16 = str16; i > 0; i --, p16++) {
	    XDrawString16(display, d, kgc, x1, y1, p16, 1);
	    if(kfs->bold)
	       XDrawString16(display, d, kgc, x1+1, y1, p16, 1);
	    x1 += size;
	 }
	 x += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1) *
	       kfs->width / kfs->xfs->max_bounds.width;
      }
      else {
	 /* JAPANESE characters, scale or rotate */
	 x += DrawScaleRotateString16(display, d, kgc, x, y, str16,
	       (tail - head) >> 1, kfs, &clipGC, flag);
      }
      head = tail;
      check = *head & 0x80;
   }

   if(clipGC != NULL)
      XFreeGC(display, clipGC);
}
