/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/attr.c,v 2.64 1993/05/27 04:52:07 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "auxtext.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#include "file.e"
#include "font.e"
#include "mainloop.e"
#include "mainmenu.e"
#include "mark.e"
#include "menu.e"
#include "msg.e"
#include "names.e"
#include "obj.e"
#include "raster.e"
#include "select.e"
#include "setup.e"
#include "stk.e"
#include "text.e"

#define PAINT 0
#define ERASE 1

int		dropObsIconAttrWhenUpdate = FALSE;

static char		attrDummyStr[MAXSTRING+1];
static struct AttrRec	* topAttr=NULL, * botAttr=NULL;

void LinkInAttr (PrevPtr, NextPtr, AttrPtr)
   struct AttrRec	* PrevPtr, * NextPtr, * AttrPtr;
   /* add AttrPtr between PrevPtr and NextPtr */
{
   AttrPtr->prev = PrevPtr;
   AttrPtr->next = NextPtr;

   if (PrevPtr == NULL)
      topAttr = AttrPtr;
   else
      PrevPtr->next = AttrPtr;

   if (NextPtr == NULL)
      botAttr = AttrPtr;
   else
      NextPtr->prev = AttrPtr;
}

void FreeAttr (AttrPtr)
   struct AttrRec	* AttrPtr;
   /* This routine only frees the attribute record, not   */
   /*    the text record, which must be freed explicitly. */
{
   cfree (AttrPtr);
}

void UnlinkAttr (AttrPtr)
   struct AttrRec	* AttrPtr;
{
   struct ObjRec	* own_ptr;
   struct AttrRec	* * top_attr_ad;
   struct AttrRec	* * bot_attr_ad;

   own_ptr = AttrPtr->owner;
 
   top_attr_ad = &(own_ptr->fattr);
   bot_attr_ad = &(own_ptr->lattr);
     
   if (*top_attr_ad == AttrPtr)
      *top_attr_ad = AttrPtr->next;
   else
      AttrPtr->prev->next = AttrPtr->next;

   if (*bot_attr_ad == AttrPtr)
      *bot_attr_ad = AttrPtr->prev;
   else
      AttrPtr->next->prev = AttrPtr->prev;
}

static
char * FindEqual(s)
   register char	* s;
{
   while (*s != '=' && *s != '\0') s++;
   return ((*s == '=') ? (s) : (char *)NULL);
}

static
void ParseAttrStr(Str, name, s)
   char	* Str, * name, * s;
{
   char	* eq_ptr, * str_ptr, * ptr;

   if ((eq_ptr = FindEqual(Str)) != NULL)
   {
      eq_ptr++;
      ptr = name;
      str_ptr = Str;
      do
      {
         *ptr = *str_ptr; 
         ptr++;
         str_ptr++;
      } while (str_ptr != eq_ptr);
 
      *ptr = '\0';

      ptr = s;
      do 
      {
         *ptr = *str_ptr; 
         ptr++;
         str_ptr++;
      } while (*str_ptr != '\0');

      *ptr = '\0';
   }
   else
   {
      *name = '\0';
      strcpy (s, Str);
   }
} 

void UpdateAttr (TextPtr, AttrPtr)
   struct TextRec	* TextPtr;
   struct AttrRec	* AttrPtr;
   /* This routine updates the name and value in the AttrRec */
   /*   and its ObjRec after an attribute was edited.        */
{
   char	s[MAXSTRING+1], name[MAXSTRING+1];

   if (AttrPtr->nameshown)
   {
      ParseAttrStr (TextPtr->first->s, name, s);
      strcpy (AttrPtr->s, s);
      strcpy (AttrPtr->name, name);
   }
   else
   {
      strcpy (s, TextPtr->first->s);
      strcpy (AttrPtr->s, s);
   }

   if (!(AttrPtr->shown))
      TextPtr->first->s[0] = '\0';
   else
   {
      if (AttrPtr->nameshown)
      {
         strcat (name, s);
         strcpy (TextPtr->first->s, name);
      }
      else
          strcpy (TextPtr->first->s, s);
   }
   UpdTextBBox (AttrPtr->obj);
}

void DrawAttrs (Win, XOff, YOff, AttrPtr)
   Window		Win;
   int			XOff, YOff;
   struct AttrRec	* AttrPtr;
{
   struct AttrRec	* ptr;

   for (ptr = AttrPtr; ptr != NULL; ptr = ptr->next)
      if (ptr->shown)
         DrawTextObj(Win, XOff, YOff, ptr->obj);
}

void MoveAttrs (AttrPtr, Dx, Dy)
   int			Dx, Dy;
   struct AttrRec	* AttrPtr;
{
   struct AttrRec	* ptr;

   for (ptr = AttrPtr; ptr != NULL; ptr = ptr->next)
   {
      ptr->obj->x += Dx;
      ptr->obj->y += Dy;
      ptr->obj->bbox.ltx += Dx;
      ptr->obj->bbox.lty += Dy;
      ptr->obj->bbox.rbx += Dx;
      ptr->obj->bbox.rby += Dy;
      ptr->obj->obbox.ltx += Dx;
      ptr->obj->obbox.lty += Dy;
      ptr->obj->obbox.rbx += Dx;
      ptr->obj->obbox.rby += Dy;
   }
}

void DelAllAttrs (AttrPtr)
   struct AttrRec	* AttrPtr;
{
   struct AttrRec	* ptr, * next_attr;

   for (ptr = AttrPtr; ptr != NULL; ptr = next_attr)
   {
      next_attr = ptr->next;
      FreeTextObj(ptr->obj);
      FreeAttr(ptr);
   }
}

static
struct AttrRec	* NewAttr (OwnerObjPtr, ObjPtr, Inherited)
   struct ObjRec	* ObjPtr;
   struct ObjRec	* OwnerObjPtr;
   short		Inherited;
{
   struct AttrRec	* attr_ptr;
    
   attr_ptr = (struct AttrRec *) calloc (1, sizeof(struct AttrRec));
   attr_ptr->shown = TRUE;
   attr_ptr->nameshown = TRUE;
   attr_ptr->inherited = Inherited;
   attr_ptr->obj = ObjPtr;
   attr_ptr->next= attr_ptr->prev = NULL;
   attr_ptr->owner = OwnerObjPtr;
   ObjPtr->detail.t->attr = attr_ptr;

   return attr_ptr;
}

static
void DupAnAttr (FromAttrPtr, ToAttrPtr)
   register struct AttrRec	* FromAttrPtr, * ToAttrPtr;
{
   struct ObjRec	* text_obj_ptr;

   text_obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   DupObjBasics (FromAttrPtr->obj, text_obj_ptr);
   DupTextObj (FromAttrPtr->obj->detail.t, text_obj_ptr);

   strcpy(ToAttrPtr->name, FromAttrPtr->name);
   strcpy(ToAttrPtr->s, FromAttrPtr->s);
   ToAttrPtr->shown = FromAttrPtr->shown;
   ToAttrPtr->nameshown = FromAttrPtr->nameshown;
   ToAttrPtr->inherited = FromAttrPtr->inherited;
   ToAttrPtr->obj = text_obj_ptr;
   ToAttrPtr->next= ToAttrPtr->prev = NULL;
   text_obj_ptr->detail.t->attr = ToAttrPtr;
}

void DupAttrs (FromObjPtr, ToObjPtr)
   register struct ObjRec	* FromObjPtr, * ToObjPtr;
{
   register struct AttrRec	* to_attr_ptr, * from_attr_ptr;

   topAttr = botAttr = NULL;
   from_attr_ptr = FromObjPtr->lattr;
   for ( ; from_attr_ptr != NULL; from_attr_ptr = from_attr_ptr->prev)
   {
      to_attr_ptr = (struct AttrRec *) calloc (1, sizeof(struct AttrRec));
      to_attr_ptr->owner = ToObjPtr;
      DupAnAttr (from_attr_ptr, to_attr_ptr);
      LinkInAttr ((struct AttrRec *)NULL, topAttr, to_attr_ptr);
   }
   ToObjPtr->fattr = topAttr;
   ToObjPtr->lattr = botAttr;
}

static
void AddAttr(ObjPtr, TextObjPtr)
   struct ObjRec	* ObjPtr, * TextObjPtr;
{
   struct AttrRec	* attr_ptr;
   struct TextRec	* text_ptr;
   char			name[MAXSTRING+1], value[MAXSTRING+1];

   text_ptr = TextObjPtr->detail.t; 

   ParseAttrStr(text_ptr->first->s, name, value);
   topAttr = ObjPtr->fattr;
   botAttr = ObjPtr->lattr;

   UnlinkObj (TextObjPtr);
   TextObjPtr->next = TextObjPtr->prev = NULL;
   attr_ptr = NewAttr (ObjPtr, TextObjPtr, FALSE); 
   UpdateAttr (text_ptr, attr_ptr); 
   LinkInAttr ((struct AttrRec *)NULL, topAttr, attr_ptr);

   ObjPtr->fattr = topAttr;
   ObjPtr->lattr = botAttr;
}

void AddAttrs ()
{
   struct ObjRec	* owner_ptr = NULL;
   struct SelRec	* sel_ptr;
   int			text_count=0, sel_ltx, sel_lty, sel_rbx, sel_rby;
   int			locked_text_count=0;

   if (topSel == NULL)
   {
      Dialog ("Must select at least one text object.",
            "( <CR> or <ESC> to continue )", attrDummyStr);
      return;
   }
   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      switch (sel_ptr->obj->type)
      {
         case OBJ_TEXT: text_count++; break;

         case OBJ_BOX:
         case OBJ_OVAL:
         case OBJ_POLYGON:
         case OBJ_POLY:
         case OBJ_SYM:
         case OBJ_GROUP:
         case OBJ_ICON:
         case OBJ_ARC:
         case OBJ_RCBOX:
         case OBJ_XBM:
         case OBJ_XPM:
            if (owner_ptr != NULL)
            {
               Msg("Two non-text objects selected.");
               return;
            }
            owner_ptr = sel_ptr->obj;
            break; 
      }
 
   if (text_count == 0)
   {
      Dialog ("No text objects selected to add as attributes.",
            "( <CR> or <ESC> to continue )", attrDummyStr);
      return;
   }
   if (owner_ptr == NULL)
   {
      Dialog ("No objects (other than TEXT objects) selected.",
            "( <CR> or <ESC> to continue )", attrDummyStr);
      return;
   }
   HighLightReverse ();
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   sel_ltx = selLtX; sel_lty = selLtY;
   sel_rbx = selRbX; sel_rby = selRbY;

   for (sel_ptr = botSel;  sel_ptr != NULL; sel_ptr = sel_ptr->prev)
      if (sel_ptr->obj->type == OBJ_TEXT)
      {
         if (sel_ptr->obj->locked)
         {
            locked_text_count++;
            sel_ptr->obj->locked = FALSE;
         }
         AddAttr(owner_ptr, sel_ptr->obj);
      }

   RemoveAllSel ();
   UnlinkObj (owner_ptr);
   AddObj (NULL, topObj, owner_ptr);
   topSel = botSel = (struct SelRec *) calloc (1, sizeof(struct SelRec));
   topSel->obj = owner_ptr;
   topSel->prev = NULL;
   botSel->next = NULL;
   AdjObjBBox (owner_ptr);
   UpdSelBBox ();
   RecordCmd (CMD_MANY_TO_ONE, NULL, topSel, botSel, 1);
   RedrawAreas (botObj, sel_ltx-GRID_ABS_SIZE(1), sel_lty-GRID_ABS_SIZE(1),
         sel_rbx+GRID_ABS_SIZE(1), sel_rby+GRID_ABS_SIZE(1),
         selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   SetFileModified (TRUE);
   justDupped = FALSE;
   if (locked_text_count==0)
      Msg ("Text object(s) attached.");
   else
      Msg ("Text object(s) unlocked and attached.");
}

static
void SaveAttr (FP, AttrPtr)
   FILE			* FP;
   struct AttrRec	* AttrPtr;
{
   fprintf (FP, "attr(\"");
   SaveString (FP, AttrPtr->name);
   fprintf (FP, "\", \"");
   SaveString (FP, AttrPtr->s);
   fprintf (FP, "\", %1d, %1d, %1d,\n",
         AttrPtr->shown, AttrPtr->nameshown, AttrPtr->inherited);
  
   SaveTextObj (FP, AttrPtr->obj);
   fprintf (FP, ")");         
}

void SaveAttrs (FP, BotAttrPtr)
   FILE			* FP;
   struct AttrRec	* BotAttrPtr;
{
 
   struct AttrRec       * ptr;

   fprintf (FP, "[\n");

   for (ptr = BotAttrPtr; ptr != NULL; ptr = ptr->prev)
     {
       SaveAttr (FP, ptr);
       if (ptr->prev != NULL)
         fprintf (FP, ",\n");
     }

   if (BotAttrPtr == NULL)
      fprintf (FP, "]");
   else
      fprintf (FP, "\n]");
}

static
char * ReadAttrString (Str)
   char	* Str;
{
   register char	* s;

   for (s = Str; *s != '\0'; s++)
      if (*s == '"')
      {
         if (s[1] == '"')
            s++;
         else
            break;
      }
      else if (*s == '\\')
         s++;

   if (*s == '"') s++;
   return (s);
}

#define GETVALUE(val,name) ScanValue("%d", (char *) &(val), name, "attribute")

int ReadAttr (FP, AttrPtr)
   FILE			* FP;
   struct AttrRec	* * AttrPtr;
{
   struct ObjRec	* TextObj;
   char			inbuf[MAXSTRING+1], * s, * line, * c_ptr;
   char			* name, * value;
   int			len, shown, nameshown, inherited, cur_size;
   int			done, allocated = FALSE;
 
   fgets (inbuf, MAXSTRING, FP); 
   scanLineNum++;

   len = strlen(inbuf);
   if (inbuf[len-1] != '\r' && inbuf[len-1] != '\n')
   {  /* line longer than MAXSTRING characters */
      cur_size = 2*MAXSTRING-1;
      line = (char *) calloc (cur_size+1, sizeof(char));
      allocated = TRUE;
      strcpy (line, inbuf);
      c_ptr = &(line[MAXSTRING-1]);

      done = FALSE;
      while (!done && fgets (inbuf, MAXSTRING, FP) != NULL)
      {
         scanLineNum++;
         len = strlen(inbuf);
         if (inbuf[len-1] == '\r' || inbuf[len-1] == '\n')
         {
            done = TRUE;
            inbuf[len-1] = '\0';
            strcpy (c_ptr, inbuf);
         }
         else
         {
            int	n = c_ptr - line;	/* NMH */

            cur_size += MAXSTRING-1;
            line = (char *) realloc (line, cur_size+1);
            c_ptr = line + n;		/* NMH */
            strcpy (c_ptr, inbuf);
            c_ptr += MAXSTRING-1;
         }
      }
      if (!done)
      {
         sprintf (inbuf, "%s, %d:  EOF in ReadAttr ().  Read aborted!",
               scanFileName, scanLineNum);
         if (PRTGIF)
            fprintf (stderr, "%s\n", inbuf);
         else
            Msg (inbuf);

         cfree (line);
         return (FALSE);
      }
   }
   else
   {
      line = inbuf;
      line[len-1] = '\0';
   }

   if (line[0] == ']')
   {
      if (allocated) cfree (line);
      return (FALSE);
   }

   *AttrPtr = NULL;

   len = strlen(line) - 1;
   line[len] = '\0';

   name = (char *) calloc (len+1, sizeof(char));
   s = FindChar ('"', line);
   strcpy(name, s);
   s = ReadAttrString (s);
   s = FindChar (',', s);
   value = (char *) calloc (len+1, sizeof(char));
   strcpy(value, FindChar ('"', s));
   s = ReadAttrString (value);
   s = FindChar (',', s);

   InitScan (s, "\t\n ,");
   if (GETVALUE (shown, "shown") == INVALID ||
       GETVALUE (nameshown, "name shown") == INVALID ||
       GETVALUE (inherited, "inherited") == INVALID)
   {
      if (allocated) cfree (line);
      cfree (name);
      cfree (value);
      return (FALSE);
   }

   *AttrPtr = (struct AttrRec *)  calloc (1, sizeof(struct AttrRec)); 

   s = ReadString (name);
   *(--s) = '\0';
   strcpy ((*AttrPtr)->name, name);
   s = ReadString (value);
   *(--s) = '\0';
   strcpy ((*AttrPtr)->s, value);

   cfree (name);
   cfree (value);

   (*AttrPtr)->shown = shown;
   (*AttrPtr)->nameshown = nameshown;
   (*AttrPtr)->inherited = inherited;

   if (ReadObj (FP, &TextObj) == FALSE)
   {
     cfree (*AttrPtr);
     *AttrPtr = NULL;
     if (allocated) cfree (line);
     return (FALSE);
   }
   TextObj->detail.t->attr = *AttrPtr;
   (*AttrPtr)->obj = TextObj;

   if (allocated) cfree (line);
   return (TRUE);
}

static
int ObjMightChangeForShowAndUpdAttrNames (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register struct AttrRec	* attr_ptr;

   if ((attr_ptr = ObjPtr->fattr) != NULL)
      for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
         if (!(attr_ptr->nameshown))
            return (TRUE);

   return (FALSE);
}

static
int DoShowAndUpdAttrNames (ObjPtr, Force)
   struct ObjRec	* ObjPtr;
   int			Force;
   /* Force will force attribute name to be shown whether the attribute */
   /*    is inherited or not.                                           */
{
   register struct AttrRec	* attr_ptr=ObjPtr->fattr;
   int				picture_changed = FALSE;
   int				len1, len2;
   char				* s, msg[80];

   for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
   {
      if (!(attr_ptr->nameshown) && (Force || !(attr_ptr->inherited)))
      {
         s = attr_ptr->obj->detail.t->first->s;
         strcpy (s, attr_ptr->name);

         len1 = strlen (attr_ptr->name);
         len2 = strlen (attr_ptr->s);
         if (len1+len2 >= MAXSTRING)
         {
            sprintf (msg, "String length exceeds %1d.  %s.",
                  MAXSTRING, "String truncated");
            Msg (msg);
            attr_ptr->s[MAXSTRING-len1] = '\0';
         }

         strcat (s, attr_ptr->s);
         attr_ptr->nameshown = TRUE;
         UpdTextBBox (attr_ptr->obj);
         if (attr_ptr->shown)
         {
            picture_changed = TRUE;

            if (attr_ptr->obj->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,
                     attr_ptr->obj->detail.t->cached_bitmap);
            attr_ptr->obj->detail.t->cached_zoom = 0;
            attr_ptr->obj->detail.t->cached_bitmap = None;
         }
      }
   }
   AdjObjBBox (ObjPtr);

   return (picture_changed);
}

static
int ShowAndUpdAttrNames ()
   /* returns TRUE if any attribute names are updated                   */
   /* This routine concatinate the 'name' and 's' first of every        */
   /*    attribute of the selected object and assign that to the        */
   /*    first line of the text object the attribute pointer points to. */
{
   struct SelRec	* sel_ptr;
   int			picture_changed = FALSE;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      if (ObjMightChangeForShowAndUpdAttrNames (sel_ptr->obj))
      {
         PrepareToReplaceAnObj (sel_ptr->obj);
         picture_changed = DoShowAndUpdAttrNames (sel_ptr->obj, TRUE);
         RecordReplaceAnObj (sel_ptr->obj);
      }

   return (picture_changed);
}

void ShowAllAttrNames ()
{
   HighLightReverse ();
   StartCompositeCmd ();
   if (ShowAndUpdAttrNames ())
   {
      UpdSelBBox ();
      RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   EndCompositeCmd ();
   HighLightForward ();
}

static
int ShowAndUpdAttrs ()
   /* returns TRUE if any attribute was not shown before */
{
   struct SelRec	* sel_ptr;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;
   int			picture_changed = FALSE;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;
      attr_ptr = obj_ptr->fattr;
      if (attr_ptr != NULL)
      {
         int	obj_changed = FALSE;

         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            if (!attr_ptr->shown)
            {
               obj_changed = TRUE;
               break;
            }

         if (obj_changed)
         {
            PrepareToReplaceAnObj (obj_ptr);
            for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL;
                  attr_ptr=attr_ptr->next)
               if (!attr_ptr->shown)
                  attr_ptr->shown = TRUE;

            picture_changed = TRUE;
            AdjObjBBox (obj_ptr);
            RecordReplaceAnObj (obj_ptr);
         }
      }
   }
   return (picture_changed);
}

void ShowAllAttrs ()
{
   HighLightReverse ();
   StartCompositeCmd ();
   if (ShowAndUpdAttrs ())
   {
      UpdSelBBox ();
      RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   EndCompositeCmd ();
   HighLightForward ();
}

static
int HideAndUpdAttrs ()
   /* returns TRUE if any attribute was shown */
{
   struct SelRec	* sel_ptr;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;
   int			picture_changed = FALSE;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;
      attr_ptr = obj_ptr->fattr;
      if (attr_ptr != NULL)
      {
         int	obj_changed = FALSE;

         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            if (attr_ptr->shown)
            {
               obj_changed = TRUE;
               break;
            }

         if (obj_changed)
         {
            PrepareToReplaceAnObj (obj_ptr);
            for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL;
                  attr_ptr=attr_ptr->next)
               if (attr_ptr->shown)
                  attr_ptr->shown = FALSE;

            picture_changed = TRUE;
            AdjObjBBox (obj_ptr);
            RecordReplaceAnObj (obj_ptr);
         }
      }
   }
   return (picture_changed);
}

void HideAllAttrs ()
{
   int	sel_ltx, sel_lty, sel_rbx, sel_rby;

   sel_ltx = selLtX; sel_lty = selLtY; sel_rbx = selRbX; sel_rby = selRbY;

   HighLightReverse ();
   StartCompositeCmd ();
   if (HideAndUpdAttrs ())
   {
      UpdSelBBox ();
      RedrawAnArea (botObj, sel_ltx-GRID_ABS_SIZE(1),
            sel_lty-GRID_ABS_SIZE(1), sel_rbx+GRID_ABS_SIZE(1),
            sel_rby+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   EndCompositeCmd ();
   HighLightForward ();
}

static
int HideAndUpdAttrNames ()
   /* returns TRUE if any attribute names are updated */
   /* For all the first line of the selected object's attributes,    */
   /*    this routine change them to the 's' field of the attribute. */
{
   struct SelRec	* sel_ptr;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;
   int			picture_changed = FALSE;
   char			* s;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;
      attr_ptr = obj_ptr->fattr;
      if (attr_ptr != NULL)
      {
         int	obj_change = FALSE;

         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            if (attr_ptr->nameshown && *(attr_ptr->name) != '\0')
            {
               obj_change = TRUE;
               break;
            }

         if (obj_change)
         {
            PrepareToReplaceAnObj (obj_ptr);
            for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL;
                  attr_ptr=attr_ptr->next)
            {
               if (attr_ptr->nameshown && *(attr_ptr->name) != '\0')
               {
                  attr_ptr->nameshown = FALSE;
                  s = attr_ptr->obj->detail.t->first->s;
                  strcpy (s, attr_ptr->s);
                  UpdTextBBox (attr_ptr->obj);
                  if (attr_ptr->shown)
                  {
                     picture_changed = TRUE;

                     if (attr_ptr->obj->detail.t->cached_bitmap != None)
                        XFreePixmap (mainDisplay,
                              attr_ptr->obj->detail.t->cached_bitmap);
                     attr_ptr->obj->detail.t->cached_zoom = 0;
                     attr_ptr->obj->detail.t->cached_bitmap = None;
                  }
               }
            }
            AdjObjBBox (obj_ptr);
            RecordReplaceAnObj (obj_ptr);
         }
      }
   }
   return (picture_changed);
}

void HideAllAttrNames ()
{
   int	sel_ltx, sel_lty, sel_rbx, sel_rby;

   sel_ltx = selLtX; sel_lty = selLtY; sel_rbx = selRbX; sel_rby = selRbY;

   HighLightReverse ();
   StartCompositeCmd ();
   if (HideAndUpdAttrNames ())
   {
      UpdSelBBox ();
      RedrawAnArea (botObj, sel_ltx-GRID_ABS_SIZE(1),
            sel_lty-GRID_ABS_SIZE(1), sel_rbx+GRID_ABS_SIZE(1),
            sel_rby+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   EndCompositeCmd ();
   HighLightForward ();
}

void DetachGroupAttrs (ObjPtr, TopSelPtr, BotSelPtr)
   struct ObjRec	* ObjPtr;
   struct SelRec	* * TopSelPtr, * * BotSelPtr;
{
   struct AttrRec	* attr_ptr, * next_attr;
   struct SelRec	* new_sel_ptr;
   int			len1, len2;
   char			* s, msg[80];

   for (attr_ptr=ObjPtr->fattr; attr_ptr!=NULL; attr_ptr=next_attr)
   {
      next_attr = attr_ptr->next;
      if (!(attr_ptr->nameshown))
      {
         s = attr_ptr->obj->detail.t->first->s;
         strcpy (s, attr_ptr->name);

         len1 = strlen (attr_ptr->name);
         len2 = strlen (attr_ptr->s);
         if (len1+len2 >= MAXSTRING)
         {
            sprintf (msg, "String length exceeds %1d.  String truncated.",
                  MAXSTRING);
            Msg (msg);
            attr_ptr->s[MAXSTRING-len1] = '\0';
         }

         strcat (s, attr_ptr->s);
         UpdTextBBox (attr_ptr->obj);
      }

      attr_ptr->obj->detail.t->attr = NULL;

      attr_ptr->obj->prev = NULL;
      attr_ptr->obj->next = ObjPtr->detail.r->first;

      if (attr_ptr->obj->next == NULL)
         ObjPtr->detail.r->last = attr_ptr->obj;
      else
          attr_ptr->obj->next->prev = attr_ptr->obj;
      ObjPtr->detail.r->first = attr_ptr->obj;

      new_sel_ptr = (struct SelRec *) calloc (1, sizeof(struct SelRec));
      new_sel_ptr->obj = attr_ptr->obj;

      new_sel_ptr->prev = NULL;
      new_sel_ptr->next = *TopSelPtr;

      if (new_sel_ptr->next == NULL)
         *BotSelPtr = new_sel_ptr;
      else
         (*TopSelPtr)->prev = new_sel_ptr;
      *TopSelPtr = new_sel_ptr;

      cfree (attr_ptr);
   }
}

void DetachAttrs ()
{
   struct SelRec	* sel_ptr, * new_sel_ptr;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr, * next_attr;
   int			picture_changed=FALSE, name_changed=FALSE;

   HighLightReverse ();
   StartCompositeCmd ();

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;
      attr_ptr = obj_ptr->fattr;
      if (attr_ptr != NULL)
      {
         int	obj_change = FALSE;

         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            if (!attr_ptr->inherited)
            {
               obj_change = TRUE;
               break;
            }

         if (obj_change)
         {
            struct SelRec	* tmp_top_sel, * tmp_bot_sel;
            struct SelRec	* tmp_sel_ptr, * next_sel;
            int			count;

            PrepareToReplaceAnObj (obj_ptr);
            if (DoShowAndUpdAttrNames (obj_ptr, FALSE)) name_changed = TRUE;

            tmp_top_sel = tmp_bot_sel = (struct SelRec *) calloc (1,
                  sizeof(struct SelRec));
            tmp_top_sel->next = tmp_top_sel->prev = NULL;
            tmp_top_sel->obj = obj_ptr;
            count = 1;

            topAttr = botAttr = NULL;
            for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL; attr_ptr=next_attr)
            {
               next_attr = attr_ptr->next;
               if (obj_ptr->type == OBJ_ICON && attr_ptr->inherited)
               {
                  LinkInAttr ((struct AttrRec *)NULL, topAttr, attr_ptr);
                  continue;
               }
               picture_changed = TRUE;

               tmp_sel_ptr = (struct SelRec *) calloc (1,sizeof(struct SelRec));
               tmp_sel_ptr->next = tmp_bot_sel;
               tmp_sel_ptr->obj = attr_ptr->obj;
               if (tmp_top_sel == tmp_bot_sel)
               {
                  tmp_sel_ptr->prev = NULL;
                  tmp_top_sel->prev = tmp_sel_ptr;
                  tmp_top_sel = tmp_sel_ptr;
               }
               else
               {
                  tmp_sel_ptr->prev = tmp_bot_sel->prev;
                  tmp_bot_sel->prev->next = tmp_sel_ptr;
                  tmp_bot_sel->prev = tmp_sel_ptr;
               }
               count++;

               attr_ptr->obj->detail.t->attr = NULL;
               AddObj (obj_ptr->prev, obj_ptr, attr_ptr->obj);
               new_sel_ptr = (struct SelRec *) calloc (1,sizeof(struct SelRec));
               new_sel_ptr->obj = obj_ptr->prev;
               AddSel (sel_ptr->prev, sel_ptr, new_sel_ptr);
               cfree (attr_ptr);
            }
            obj_ptr->fattr = topAttr;
            obj_ptr->lattr = botAttr;
            AdjObjBBox (obj_ptr);
            RecordCmd (CMD_ONE_TO_MANY, NULL, tmp_top_sel, tmp_bot_sel, count);

            for (tmp_sel_ptr=tmp_top_sel; tmp_sel_ptr!=NULL;
                  tmp_sel_ptr=next_sel)
            {
               next_sel = tmp_sel_ptr->next;
               cfree (tmp_sel_ptr);
            }
         }
      }
   }
   EndCompositeCmd ();
   UpdSelBBox ();
   if (picture_changed || name_changed)
   {
      RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

void UpdAttr (AttrPtr)
   struct AttrRec	* AttrPtr;
   /* Update the text object's string value associated with AttrPtr */
{
   int	len1, len2;
   char	msg[80];

   if (AttrPtr->nameshown)
   {
      strcpy (AttrPtr->obj->detail.t->first->s, AttrPtr->name);

      len1 = strlen (AttrPtr->name);
      len2 = strlen (AttrPtr->s);
      if (len1+len2 >= MAXSTRING)
      {
         sprintf (msg, "String length exceeds %1d.  String truncated.",
               MAXSTRING);
         Msg (msg);
         AttrPtr->s[MAXSTRING-len1] = '\0';
      }

      strcat (AttrPtr->obj->detail.t->first->s, AttrPtr->s);
   }
   else
      strcpy (AttrPtr->obj->detail.t->first->s, AttrPtr->s);
   UpdTextBBox(AttrPtr->obj);
}

static
int MoveOneAttr (ObjPtr, AttrPtr)
   struct ObjRec	* ObjPtr;
   struct AttrRec	* AttrPtr;
{
   struct ObjRec	* text_obj_ptr;
   int          	x, y, grid_x, grid_y, dx, dy, placing = TRUE;
   int          	ltx, lty, rbx, rby;
   int			orig_x, orig_y, grid_orig_x, grid_orig_y;
   XEvent		input, ev;

   text_obj_ptr = AttrPtr->obj;
   Msg ("LEFT--show and move.  MIDDLE--toggle name shown.  RIGHT--hide attr.");

   orig_x = OFFSET_X(text_obj_ptr->x);
   orig_y = OFFSET_Y(text_obj_ptr->y);
   GridXY (orig_x, orig_y, &grid_orig_x, &grid_orig_y);
   ltx = OFFSET_X(text_obj_ptr->bbox.ltx);
   lty = OFFSET_Y(text_obj_ptr->bbox.lty);
   rbx = OFFSET_X(text_obj_ptr->bbox.rbx)+1;
   rby = OFFSET_Y(text_obj_ptr->bbox.rby)+1;

   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonPressMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
   XWarpPointer (mainDisplay, None, drawWindow, 0, 0, 0, 0, orig_x, orig_y);

   dx = dy = 0;
   grid_x = grid_orig_x; grid_y = grid_orig_y;

   SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
   while (placing)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
      {
         SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
         ExposeEventHandler (&input, TRUE);
         SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
      }
      else if (input.type == MotionNotify)
      {
         x = input.xmotion.x;
         y = input.xmotion.y;
         GridXY (x, y, &grid_x, &grid_y);

         if (grid_x != grid_orig_x+dx || grid_y != grid_orig_y+dy)
         {
            SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
            dx = grid_x - grid_orig_x;
            dy = grid_y - grid_orig_y;
            SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
            MarkRulers (grid_x, grid_y);
         }
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
      else if (input.type == ButtonPress)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         placing = FALSE;
         SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
         Msg ("");
         switch (input.xbutton.button)
         {
            case Button1:
               if (dx != 0 || dy != 0)
               {
                  if (ObjPtr->locked)
                  {
                     Msg ("Can not move attributes of locked object.");
                     return (FALSE);
                  }
                  HighLightReverse ();
                  PrepareToReplaceAnObj (ObjPtr);
                  AttrPtr->shown = TRUE;
                  MoveObj (text_obj_ptr, GRID_ABS_SIZE(dx), GRID_ABS_SIZE(dy));
                  AdjObjBBox (ObjPtr);
                  RecordReplaceAnObj (ObjPtr);
                  return (TRUE);
               }
               else if (!(AttrPtr->shown))
               {
                  HighLightReverse ();
                  PrepareToReplaceAnObj (ObjPtr);
                  AttrPtr->shown = TRUE;
                  AdjObjBBox (ObjPtr);
                  RecordReplaceAnObj (ObjPtr);
                  return (TRUE);
               }
               return (FALSE);
            case Button2:
               if (*(AttrPtr->name) != '\0') PrepareToReplaceAnObj (ObjPtr);
               if (!(AttrPtr->nameshown && *(AttrPtr->name)=='\0'))
                  AttrPtr->nameshown = !AttrPtr->nameshown;
               UpdAttr (AttrPtr);
               if (AttrPtr->shown)
               {
                  HighLightReverse ();
                  AdjObjCache (ObjPtr);
                  AdjObjBBox (ObjPtr);
                  if (*(AttrPtr->name) != '\0') RecordReplaceAnObj (ObjPtr);
                  return (TRUE);
               }
               if (*(AttrPtr->name) != '\0') RecordReplaceAnObj (ObjPtr);
               return (FALSE);
            case Button3:
               if (AttrPtr->shown)
               {
                  HighLightReverse ();
                  PrepareToReplaceAnObj (ObjPtr);
                  AttrPtr->shown = FALSE;
                  AdjObjBBox (ObjPtr);
                  RecordReplaceAnObj (ObjPtr);
                  return (TRUE);
               }
               return (FALSE);
         }
      }
   }
   return (FALSE);
}

static
int ChangeAttrJust (ObjPtr, AttrPtr)
   struct ObjRec	* ObjPtr;
   struct AttrRec	* AttrPtr;
{
   struct ObjRec	* text_obj_ptr;
   int          	x, y, grid_x, grid_y, dx, dy, placing = TRUE;
   int          	ltx, lty, rbx, rby;
   int			orig_x, orig_y, grid_orig_x, grid_orig_y;
   int			old_just, new_just = 0;
   XEvent		input, ev;

   text_obj_ptr = AttrPtr->obj;
   Msg ("LEFT--left, MIDDLE--center, RIGHT--right justified.");

   orig_x = OFFSET_X(text_obj_ptr->x);
   orig_y = OFFSET_Y(text_obj_ptr->y);
   GridXY (orig_x, orig_y, &grid_orig_x, &grid_orig_y);
   ltx = OFFSET_X(text_obj_ptr->bbox.ltx);
   lty = OFFSET_Y(text_obj_ptr->bbox.lty);
   rbx = OFFSET_X(text_obj_ptr->bbox.rbx)+1;
   rby = OFFSET_Y(text_obj_ptr->bbox.rby)+1;

   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonPressMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
   XWarpPointer (mainDisplay, None, drawWindow, 0, 0, 0, 0, orig_x, orig_y);

   dx = dy = 0;
   grid_x = grid_orig_x; grid_y = grid_orig_y;

   SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
   while (placing)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
      {
         SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
         ExposeEventHandler (&input, TRUE);
         SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
      }
      else if (input.type == MotionNotify)
      {
         x = input.xmotion.x;
         y = input.xmotion.y;
         GridXY (x, y, &grid_x, &grid_y);

         if (grid_x != grid_orig_x+dx || grid_y != grid_orig_y+dy)
         {
            SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
            dx = grid_x - grid_orig_x;
            dy = grid_y - grid_orig_y;
            SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
            MarkRulers (grid_x, grid_y);
         }
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
      else if (input.type == ButtonPress)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         placing = FALSE;
         SelBox (drawWindow, revDefaultGC, ltx+dx, lty+dy, rbx+dx, rby+dy);
         Msg ("");
         old_just = text_obj_ptr->detail.t->just;
         switch (input.xbutton.button)
         {
            case Button1: new_just = JUST_L; break;
            case Button2: new_just = JUST_C; break;
            case Button3: new_just = JUST_R; break;
         }
         if (old_just != new_just)
         {
            HighLightReverse ();
            PrepareToReplaceAnObj (ObjPtr);
            text_obj_ptr->detail.t->just = new_just;
            AdjObjCache (ObjPtr);
            UpdTextBBox (text_obj_ptr);
            AdjObjBBox (ObjPtr);
            RecordReplaceAnObj (ObjPtr);
            return (TRUE);
         }
      }
   }
   return (FALSE);
}

void MoveAttr ()
{
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr, * attr_ptr1;
   int			num_attrs = 0, i, index, x, y;
   int			sel_ltx, sel_lty, sel_rbx, sel_rby;
   int			* fore_colors, * pixel_ptr, * valid, * flag_ptr;
   int			len1, len2, * init_rv;
   char			* * attrStrs, * s, buf[MAXSTRING], msg[80];
   unsigned int		button;

   if (topSel != botSel || topSel == NULL)
   { Msg ("Please select only ONE object."); return; }

   obj_ptr = topSel->obj;
   attr_ptr1 = attr_ptr = obj_ptr->fattr;
   for ( ; attr_ptr1 != NULL; attr_ptr1 = attr_ptr1->next, num_attrs++) ;

   if (num_attrs == 0)
   { Msg ("Selected object currently has NO attributes."); return; }

   attrStrs = (char * *) calloc (num_attrs, sizeof(char *));
   fore_colors = pixel_ptr = (int *) calloc (num_attrs, sizeof(int));
   valid = flag_ptr = (int *) calloc (num_attrs, sizeof(int));
   init_rv = (int *) calloc (num_attrs, sizeof(int));

   attr_ptr1 = attr_ptr;
   for (i = 0; i < num_attrs; i++, attr_ptr1 = attr_ptr1->next)
   {
      s = (char *) calloc (MAXSTRING, sizeof(char));
      attrStrs[i] = s;
      strcpy (s, attr_ptr1->name);

      len1 = strlen (attr_ptr1->name);
      len2 = strlen (attr_ptr1->s);
      if (len1+len2 >= MAXSTRING)
      {
         sprintf (msg, "String length exceeds %1d.  String truncated.",
               MAXSTRING);
         Msg (msg);
         attr_ptr1->s[MAXSTRING-len1] = '\0';
      }

      strcat (s, attr_ptr1->s);
      *pixel_ptr++ = colorPixels[attr_ptr1->obj->color];
      *flag_ptr++ = TRUE;
      init_rv[i] = FALSE;
   }

   Msg ("Hold down left button to see attributes.");
   strcpy (buf, "Left button move/see attributes.  ");
   strcat (buf, "Middle button change attribute justifications.");
   Msg (buf);
   button = CornerLoop (&x, &y);
   activeMenu = INVALID;
   index = TextMenuLoop (x, y, attrStrs, num_attrs, fore_colors, valid, init_rv,
         MULTICOLOR);
   if (index != INVALID)
   {
      attr_ptr1 = attr_ptr;
      for (i = 0; i < index; i++, attr_ptr1 = attr_ptr1->next) ;
      sel_ltx = selLtX; sel_lty = selLtY; sel_rbx = selRbX; sel_rby = selRbY;
      if (button == Button1)
      {
         if (MoveOneAttr (obj_ptr, attr_ptr1))
         {  /* HighLightReverse () is expected to be called */
            UpdSelBBox ();
            RedrawAreas (botObj, sel_ltx-GRID_ABS_SIZE(1),
                  sel_lty-GRID_ABS_SIZE(1),
                  sel_rbx+GRID_ABS_SIZE(1), sel_rby+GRID_ABS_SIZE(1),
                  selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
                  selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
            SetFileModified (TRUE);
            HighLightForward ();
         }
      }
      else if (button == Button2)
      {
         if (ChangeAttrJust (obj_ptr, attr_ptr1))
         {  /* HighLightReverse () is expected to be called */
            UpdSelBBox ();
            RedrawAreas (botObj, sel_ltx-GRID_ABS_SIZE(1),
                  sel_lty-GRID_ABS_SIZE(1), sel_rbx+GRID_ABS_SIZE(1),
                  sel_rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
                  selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
                  selRbY+GRID_ABS_SIZE(1));
            SetFileModified (TRUE);
            HighLightForward ();
         }
      }
   }

   for (i = 0; i < num_attrs; i++) cfree (attrStrs[i]);
   cfree (attrStrs);
   justDupped = FALSE;
}

void CopyAndUpdateAttrs (ToObjPtr, FromObjPtr)
   struct ObjRec	* ToObjPtr, * FromObjPtr;
{
   register struct AttrRec	* to_attr_ptr, * from_attr_ptr;
   char				msg[MAXSTRING];
   int				confirm_status;

   topAttr = botAttr = NULL;
   from_attr_ptr = FromObjPtr->fattr;
   for ( ; from_attr_ptr != NULL; from_attr_ptr = from_attr_ptr->next)
   {
      to_attr_ptr = ToObjPtr->fattr;
      for ( ; to_attr_ptr != NULL; to_attr_ptr = to_attr_ptr->next)
      {
         if (from_attr_ptr->obj->color == to_attr_ptr->obj->color &&
               strcmp (from_attr_ptr->name, to_attr_ptr->name) == 0)
         {
            if (*(from_attr_ptr->s) != '\0')
            {
               strcpy (to_attr_ptr->s, from_attr_ptr->s);
               UpdAttr (to_attr_ptr);
            }
            break;
         }
      }
      if (to_attr_ptr == NULL)
      {  /* can not find the same attribute in the new one */
         confirm_status = CONFIRM_NO;

         if (from_attr_ptr->inherited)
         {  /* obsolete attribute? */
            confirm_status = (dropObsIconAttrWhenUpdate) ? CONFIRM_YES :
                  CONFIRM_CANCEL;
            if (*from_attr_ptr->name == '\0')
               sprintf (msg, "Is the attribute '%s' %s for icon '%s'?  [yn](y)",
                     from_attr_ptr->s, "obsolete", FromObjPtr->detail.r->s);
            else
               sprintf (msg, "Is the attribute '%s' %s for icon '%s'?  [yn](y)",
                     from_attr_ptr->name, "obsolete", FromObjPtr->detail.r->s);
            while (confirm_status == CONFIRM_CANCEL)
            {
               confirm_status = YesNoCancel (msg, CONFIRM_YES);
               if (confirm_status == CONFIRM_CANCEL)
                  Msg ("CANCEL is not a valid option.");
            }
         }

         if (confirm_status == CONFIRM_NO)
         {  /* new attribute */
            to_attr_ptr = (struct AttrRec *) calloc (1, sizeof(struct AttrRec));
            to_attr_ptr->owner = ToObjPtr;
            DupAnAttr (from_attr_ptr, to_attr_ptr);
            LinkInAttr ((struct AttrRec *)NULL, topAttr, to_attr_ptr);
         }
      }
      else
      {
         to_attr_ptr->shown = from_attr_ptr->shown;
         to_attr_ptr->nameshown = from_attr_ptr->nameshown;
      }
   }
   if (topAttr != NULL)
   {
      topAttr->prev = NULL;
      botAttr->next = ToObjPtr->fattr;

      if (ToObjPtr->fattr != NULL) ToObjPtr->fattr->prev = botAttr;
      ToObjPtr->fattr = topAttr;
      if (ToObjPtr->lattr == NULL) ToObjPtr->lattr = botAttr;
   }
   AdjObjBBox (ToObjPtr);
}

void ToggleNamedAttrShown (attr_name)
   char	* attr_name;
{
   char			* paran_ptr, msg[MAXSTRING+1];
   struct AttrRec	* attr_ptr;
   int			ltx, lty, rbx, rby;

   if (topSel != botSel || topSel == NULL)
   { Msg ("Please select only ONE object."); return; }

   if ((paran_ptr = strchr (attr_name, ')')) == NULL)
   { Msg ("Invalid format in shortcut specification."); return; }

   *paran_ptr = '\0';
   if ((attr_ptr = FindAttrWithName (topSel->obj, attr_name)) == NULL)
   {
      sprintf (msg, "Can not find a '%s' attribute for the selected object.",
            attr_name);
      Msg (msg);
      return;
   }
   ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
   HighLightReverse ();
   PrepareToReplaceAnObj (topSel->obj);
   attr_ptr->shown = !attr_ptr->shown;
   AdjObjBBox (topSel->obj);
   RecordReplaceAnObj (topSel->obj);
   UpdSelBBox ();
   RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
         rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1),
         selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   SetFileModified (TRUE);
   HighLightForward ();
}

void AddFileAttrs ()
{
   struct SelRec	* sel_ptr;
   int			ltx, lty, rbx, rby;

   if (topSel == NULL)
   {
      Dialog ("Must select at least one text object.",
            "( <CR> or <ESC> to continue )", attrDummyStr);
      return;
   }
   for (sel_ptr=topSel; sel_ptr!=NULL; sel_ptr=sel_ptr->next)
   {
      if (sel_ptr->obj->type != OBJ_TEXT)
      {
         Dialog ("Can not attach non-text objects.",
               "( <CR> or <ESC> to continue )", attrDummyStr);
         return;
      }
      else if (sel_ptr->obj->detail.t->first != sel_ptr->obj->detail.t->last)
      {
         Dialog ("A multiline text can NOT be a file attribute.",
               "( <CR> or <ESC> to continue )", attrDummyStr);
         return;
      }
   }

   ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;

   HighLightReverse ();
   AddObj (NULL, topObj, tgifObj);
   AddNewSelObj (topObj);
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);

   for (sel_ptr=topSel->next; sel_ptr!=NULL; sel_ptr=sel_ptr->next)
      AddAttr (topObj, sel_ptr->obj);
   RemoveAllSel ();
   AddNewSelObj (topObj);
   UpdSelBBox ();

   recordCmdIncludeTgifObj = TRUE;
   RecordCmd (CMD_MANY_TO_ONE, NULL, topSel, botSel, 1);
   recordCmdIncludeTgifObj = FALSE;
   RemoveAllSel ();
   UnlinkObj (topObj);
   RedrawAnArea (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
         rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1));
   SetFileModified (TRUE);
   justDupped = FALSE;
}

void DetachFileAttrs ()
{
   struct AttrRec	* attr_ptr, * next_attr;
   struct SelRec	* sel_ptr;

   MakeQuiescent ();

   if (tgifObj->fattr == NULL)
   {
      Dialog ("Current file contains no attributes.",
            "( <CR> or <ESC> to continue )", attrDummyStr);
      return;
   }

   AddObj (NULL, topObj, tgifObj);
   AddNewSelObj (topObj);
   PrepareToReplaceAnObj (topObj);

   for (attr_ptr=topObj->fattr; attr_ptr!=NULL; attr_ptr=next_attr)
   {
      next_attr = attr_ptr->next;

      sel_ptr = (struct SelRec *) calloc (1,sizeof(struct SelRec));
      sel_ptr->obj = attr_ptr->obj;
      AddSel (topSel, topSel->next, sel_ptr);

      attr_ptr->obj->detail.t->attr = NULL;
      AddObj (topObj, topObj->next, attr_ptr->obj);
      cfree (attr_ptr);
   }
   topObj->fattr = topObj->lattr = NULL;
   recordCmdIncludeTgifObj = TRUE;
   RecordCmd (CMD_ONE_TO_MANY, NULL, topSel, botSel, numObjSelected);
   recordCmdIncludeTgifObj = FALSE;

   UnlinkObj (topObj);

   sel_ptr = topSel;
   topSel = topSel->next;
   topSel->prev = NULL;
   cfree (sel_ptr);
   numObjSelected--;
   UpdSelBBox ();

   RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   SetFileModified (TRUE);
   justDupped = FALSE;
}

static XComposeStatus	c_stat;
static DspList		* fileAttrNameDspPtr=NULL;

static
DspList * FileAttrNameListing ()
{
   register int		i;
   DspList		* dsp_ptr;
   struct AttrRec	* attr_ptr;

   for (nameEntries=0, attr_ptr=tgifObj->fattr; attr_ptr!=NULL;
         attr_ptr=attr_ptr->next)
      nameEntries++;

   fileAttrNameDspPtr = (DspList *) calloc (nameEntries, sizeof(DspList));
   for (i=1, dsp_ptr=fileAttrNameDspPtr, attr_ptr=tgifObj->fattr;
         i<=nameEntries; i++, dsp_ptr++, attr_ptr=attr_ptr->next)
   {
      strcpy (dsp_ptr->itemstr, attr_ptr->obj->detail.t->first->s);
      dsp_ptr->next = ((i == nameEntries) ? NULL : &dsp_ptr[1]);
   }
   return (fileAttrNameDspPtr);
}

static
int EditFileAttrNames (TopStr)
   char	* TopStr;
{
   int			button_widths, str_width, graph_width;
   int			str_start, button_start, graph_start;
   int			dsp_w, dsp_h, w, h, i, button_selected = INVALID;
   XEvent		input, ev;
   int			changing = TRUE, name_index, exposed = FALSE;
   char			buf[80], name[MAXPATHLENGTH];
   XKeyEvent		* key_ev;
   XButtonEvent		* button_ev;
   KeySym		key_sym;
   XWMHints		wmhints;
   XSizeHints		sizehints;
   XSetWindowAttributes	win_attrs;
   int			win_x, win_y;

   dsp_w = DisplayWidth (mainDisplay, mainScreen);
   dsp_h = DisplayHeight (mainDisplay, mainScreen);

   button_widths = ButtonWidth("OK", 8) + ButtonWidth("CANCEL", 8) +
         defaultFontWidth;
   numButtons = 2;
   strcpy (buttonStr[0], "OK");
   strcpy (buttonStr[1], "CANCEL");

   str_width = defaultFontWidth * strlen (TopStr);
   graph_width = nameDspWinW + scrollBarW + 2 * brdrW;

   if (str_width > graph_width)
   {
      w = str_width + 4 * defaultFontWidth;
      str_start = 2 * defaultFontWidth;
      graph_start = (w - graph_width) / 2;
   }
   else
   {
      w = graph_width + 4 * defaultFontWidth;
      str_start = (w - str_width) / 2;
      graph_start = 2 * defaultFontWidth;
   }
   button_start = (w - button_widths) / 2;
   h = (8 + ITEM_DSPED) * ROW_HEIGHT;

   win_x = (w > dsp_w) ? 0 : (dsp_w - w)/2;
   win_y = (h > dsp_h) ? 0 : (dsp_h - h)/3;

   if ((nameBaseWin = XCreateSimpleWindow (mainDisplay, rootWindow,
         win_x, win_y, w, h, brdrW, myBorderPixel, myBgPixel)) == 0)
      Error ("ChooseAName()", "Can not XCreateSimpleWindow() for nameBaseWin");

   XDefineCursor (mainDisplay, nameBaseWin, defaultCursor);

   if ((nameDspWin = XCreateSimpleWindow (mainDisplay, nameBaseWin, graph_start,
         5*ROW_HEIGHT, nameDspW, nameDspH, brdrW, myBorderPixel,
         myBgPixel)) == 0)
      Error ("ChooseAName()", "Can not XCreateSimpleWindow() for nameDspWin");

   if ((nameScrollWin = XCreateSimpleWindow (mainDisplay, nameBaseWin,
         graph_start+nameDspWinW, 5*ROW_HEIGHT, scrollBarW, nameDspH,
         brdrW, myBorderPixel, myBgPixel)) == 0)
      Error ("ChooseAName()","Can not XCreateSimpleWindow() for nameScrollWin");

   win_attrs.save_under = True;
   XChangeWindowAttributes (mainDisplay, nameBaseWin, CWSaveUnder, &win_attrs);

   wmhints.flags = InputHint | StateHint;
   wmhints.input = True;
   wmhints.initial_state = NormalState;
   XSetWMHints (mainDisplay, nameBaseWin, &wmhints);

   sizehints.flags = PPosition | PSize | USPosition | PMinSize | PMaxSize;
   sizehints.x = win_x;
   sizehints.y = win_y;
   sizehints.width = sizehints.min_width = sizehints.max_width = w;
   sizehints.height = sizehints.min_height = sizehints.max_height = h;
#ifdef NOTR4MODE
   XSetNormalHints (mainDisplay, nameBaseWin, &sizehints);
#else
   XSetWMNormalHints (mainDisplay, nameBaseWin, &sizehints);
#endif

   XSetTransientForHint (mainDisplay, nameBaseWin, mainWindow);

#ifdef MAPBEFORESELECT
   XMapWindow (mainDisplay, nameBaseWin);
   XSelectInput (mainDisplay, nameBaseWin,
         KeyPressMask | ButtonPressMask | ExposureMask | StructureNotifyMask);
   XMapWindow (mainDisplay, nameDspWin);
   XSelectInput (mainDisplay, nameDspWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow (mainDisplay, nameScrollWin);
   XSelectInput (mainDisplay, nameScrollWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
#else
   XSelectInput (mainDisplay, nameBaseWin,
         KeyPressMask | ButtonPressMask | ExposureMask | StructureNotifyMask);
   XMapWindow (mainDisplay, nameBaseWin);
   XSelectInput (mainDisplay, nameDspWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow (mainDisplay, nameDspWin);
   XSelectInput (mainDisplay, nameScrollWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow (mainDisplay, nameScrollWin);
#endif

   if (warpToWinCenter)
      XWarpPointer (mainDisplay, None, nameBaseWin, 0, 0, 0, 0,
            (int)(w/2), (int)(h/2));

   XSync (mainDisplay, False);

   justClicked = FALSE;

   Msg ("");

   if (nameMarked == INVALID)
   {
      name[0] = '\0';
      name_index = 0;
   }
   else
   {
      if (nameMarked >= ITEM_DSPED)
      {
         if (nameMarked < nameEntries-ITEM_DSPED)
            nameFirst = nameMarked;
         else
            nameFirst = nameEntries-ITEM_DSPED;
      }
      strcpy (name, nameDspPtr[nameMarked]);
      name_index = strlen (name);
   }

   while (changing)
   {
      XNextEvent (mainDisplay, &input);

      if ((input.type==MapNotify && input.xany.window==nameBaseWin) ||
            (input.type==Expose && (input.xany.window==nameBaseWin ||
            input.xany.window==nameScrollWin ||
            input.xany.window==nameDspWin)) ||
            (!exposed &&
            (XCheckWindowEvent (mainDisplay,nameBaseWin,ExposureMask,&ev) ||
            XCheckWindowEvent (mainDisplay,nameScrollWin,ExposureMask,&ev) ||
            XCheckWindowEvent (mainDisplay,nameDspWin,ExposureMask,&ev) ||
            XCheckWindowEvent (mainDisplay,nameBaseWin,StructureNotifyMask,
            &ev))))
      {
         while (XCheckWindowEvent (mainDisplay,nameBaseWin,ExposureMask,&ev)) ;
         while (XCheckWindowEvent (mainDisplay,nameScrollWin,ExposureMask,
               &ev))
            ;
         while (XCheckWindowEvent (mainDisplay,nameDspWin,ExposureMask,&ev)) ;
         while (XCheckWindowEvent (mainDisplay,nameBaseWin,StructureNotifyMask,
               &ev))
            ;

         RedrawNameBaseWindow (TopStr, name, str_start, graph_start,
               button_start, w, h);
         RedrawNameScrollWin ();
         RedrawDspWindow ();

         exposed = TRUE;
         XSync (mainDisplay, False);

         if ((input.type==MapNotify && input.xany.window==nameBaseWin) ||
               (input.type==Expose && (input.xany.window==nameBaseWin ||
               input.xany.window==nameScrollWin ||
               input.xany.window==nameDspWin)))
            continue;
      }

      if (input.type==Expose)
         ExposeEventHandler (&input, FALSE);
      else if (input.type==VisibilityNotify && input.xany.window==mainWindow &&
            input.xvisibility.state==VisibilityUnobscured)
      {
         while (XCheckWindowEvent (mainDisplay, mainWindow,
               VisibilityChangeMask, &ev)) ;
         if (pinnedMainMenu) XMapRaised (mainDisplay, mainMenuWindow);
         for (i = 0; i < numExtraWins; i++)
            if (extraWinInfo[i].mapped && extraWinInfo[i].raise &&
                  extraWinInfo[i].window != None)
               XMapRaised (mainDisplay, extraWinInfo[i].window);
         XMapRaised (mainDisplay, nameBaseWin);
      }
      else if (input.type == KeyPress)
      {
         key_ev = &(input.xkey);
         XLookupString (key_ev, buf, 80-1, &key_sym, &c_stat);
         TranslateKeys (buf, &key_sym);

         if ((buf[0]=='\r' && (key_sym & 0xff)=='\r') ||
             (buf[0]=='\n' && (key_sym & 0xff)=='\n'))
         {
            changing = FALSE;
            button_selected = BUTTON_OK;
         }
         else if (buf[0]=='\033' && (key_sym & 0xff)=='\033')
         {
            changing = FALSE;
            button_selected = BUTTON_CANCEL;
         }
         else if ((buf[0] == '\b'  && (key_sym & 0xff)=='\b') ||
               (buf[0] == '\b' && (key_sym & 0xff)=='h') ||
               (buf[0] == '\177' && (key_sym & 0x7f)=='\177') ||
               key_sym==XK_Left)
         {
            if (nameMarked != INVALID && name_index != 0)
            {
               name[--name_index] = '\0';
               if (nameMarked != INVALID)
                  strcpy (nameDspPtr[nameMarked], name);

               if (exposed)
               {
                  RedrawNamePath (name, graph_start,
                        3*ROW_HEIGHT+defaultFontAsc+2);
                  RedrawDspWindow ();
               }
            }
         }
         else if (nameEntries != 0 && ((key_sym>'\040' && key_sym<='\177' &&
               (key_ev->state & ControlMask)) || key_sym==XK_Up ||
               key_sym==XK_Down))
         {
            if ((i = ControlChar (key_ev, key_sym)) != BAD)
            {
               if (i == INVALID)
               {
                  name[0] = '\0';
                  name_index = 0;
                  nameFirst = 0;
                  nameMarked = INVALID;
               }
               else
               {
                  strcpy (name, nameDspPtr[i]);
                  name_index = strlen (name);

                  if (i < nameFirst)
                     nameFirst = i;
                  else if (i >= nameFirst+ITEM_DSPED)
                  {
                     if (i < nameEntries-ITEM_DSPED)
                        nameFirst = i;
                     else
                        nameFirst = nameEntries-ITEM_DSPED;
                  }
                  nameMarked = i;
               }

               if (exposed)
               {
                  RedrawNamePath (name, graph_start,
                        3*ROW_HEIGHT+defaultFontAsc+2);
                  RedrawNameScrollWin ();
                  RedrawDspWindow ();
               }
            }
         }
         else if (nameMarked != INVALID &&
               key_sym>='\040' && key_sym<='\177' && nameEntries != 0)
         {
            name[name_index++] = buf[0];
            name[name_index] = '\0';
            strcpy (nameDspPtr[nameMarked], name);

            if (exposed)
            {
               RedrawNamePath (name, graph_start,
                     3*ROW_HEIGHT+defaultFontAsc+2);
               RedrawNameScrollWin ();
               RedrawDspWindow ();
            }
         }
      }
      else if (input.type == ButtonPress)
      {
         button_ev = &(input.xbutton);
         if (button_ev->window == nameBaseWin)
         {
            if (PointInBBox (button_ev->x, button_ev->y, buttonBBox[0]))
            {
               changing = FALSE;
               button_selected = BUTTON_OK;
            }
            else if (PointInBBox (button_ev->x, button_ev->y, buttonBBox[1]))
            {
               changing = FALSE;
               button_selected = 1;
            }
         }
         else if (button_ev->window == nameScrollWin)
            NameScrollHandler (button_ev);
         else if (button_ev->window == nameDspWin)
         {
            NameDspHandler (button_ev);
            if (nameMarked != INVALID)
            {
               strcpy (name, nameDspPtr[nameMarked]);
               name_index = strlen (name);
               RedrawNamePath (name, graph_start,
                     3*ROW_HEIGHT+defaultFontAsc+2);
            }
         }
      }
   }

   if (exposed && button_selected != INVALID)
      DisplayButton (nameBaseWin, buttonStr[button_selected], 8,
            &buttonBBox[button_selected], BUTTON_INVERT);

   XDestroyWindow (mainDisplay, nameBaseWin);
   if (warpToWinCenter)
      XWarpPointer (mainDisplay, None, drawWindow, 0, 0, 0, 0,
            (int)(ZOOMED_SIZE(drawWinW)>>1), (int)(ZOOMED_SIZE(drawWinH)>>1));
   return (button_selected == BUTTON_OK);
}

static
int BlankStr (s)
   register char	* s;
{
   while (*s == ' ') s++;
   return (*s == '\0');
}

void EditFileAttrs ()
{
   register int	i;
   DspList	* dsp_ptr;

   MakeQuiescent ();

   if (tgifObj->fattr == NULL)
   {
      Dialog ("Current file contains no attributes.",
            "( <CR> or <ESC> to continue )", attrDummyStr);
      return;
   }
   dsp_ptr = FileAttrNameListing ();
   nameDspPtr = MakeNameDspItemArray (nameEntries, dsp_ptr);
   nameFirst = 0;
   nameMarked = 0;
   if (EditFileAttrNames ("Edit File Attributes..."))
   {
      int		modified=FALSE;
      struct AttrRec	* attr_ptr, * next_attr;

      AddObj (NULL, topObj, tgifObj);
      AddNewSelObj (topObj);
      PrepareToReplaceAnObj (topObj);

      for (attr_ptr=tgifObj->fattr, i=0; attr_ptr!=NULL;
            attr_ptr=next_attr, i++)
      {
         int	blank_str=BlankStr(nameDspPtr[i]);

         next_attr = attr_ptr->next;
         if (blank_str || strcmp (attr_ptr->obj->detail.t->first->s,
               nameDspPtr[i]) != 0)
         {
            modified = TRUE;
            if (blank_str)
            {
               UnlinkAttr (attr_ptr);
               FreeTextObj (attr_ptr->obj);
               FreeAttr (attr_ptr);
            }
            else
            {
               strcpy (attr_ptr->obj->detail.t->first->s, nameDspPtr[i]);
               UpdateAttr (attr_ptr->obj->detail.t, attr_ptr);
            }
         }
      }
      if (modified)
      {
         recordCmdIncludeTgifObj = TRUE;
         RecordReplaceAnObj (topObj);
         recordCmdIncludeTgifObj = FALSE;

         UnlinkObj (topObj);

         SetFileModified (TRUE);
         Msg ("File attribute updated.");
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   cfree (dsp_ptr);
   cfree (*nameDspPtr);
   cfree (nameDspPtr);
   nameDspPtr = NULL;
   Msg ("");
}
