/*
 * top - a top users display for AIX 3.2.X
 *
 * SYNOPSIS:  any IBM RS6000 running aix 3.2.X
 *
 * DESCRIPTION:
 * Works for:
 *      AIX 3.2.X
 *
 * TERMCAP: curses
 * MODE: 2111
 * UID: root
 * GID: system
 * INSTALL: /usr/ucb/install
 *
 * AUTHOR: William L. Jones jones@chpc.utexas.edu
 */

#include <stdio.h>
#include <nlist.h>
#include <procinfo.h>
#include <sys/sysinfo.h>
#include <sys/vminfo.h>
#include <sys/proc.h>

#include "top.h"
#include "machine.h"


/*
 * Defines.
 */

#define NPROCS 10000				/* I hop it large enough */
#define pagetok(size) size
#define KMEM    "/dev/kmem"
#define CPUSTATES  CPU_NTIMES
#define VMUNIX  "/vmunix"


/*
 * Globals.
 */
static          int       kmem = -1;
static struct   procinfo  procinfo[NPROCS];
static struct	procinfo  *pref[NPROCS];
static          int       pref_len = 0;
static struct   userinfo  userinfo[NPROCS];
static struct   sysinfo   sysinfox;
static 		int       nproc= 0;
static 		int	  sysinfo_offset;
static          int       vmker_offset;
static          long      cp_old[CPUSTATES];
static          long      cp_diff[CPUSTATES];
static unsigned long 	  avenrun_offset;
static          int       avenrun[3];
static		uid_t	  real_uid = -1;;
 

struct oldproc {
    pid_t  pi_pid;
    double cpu;
    double pct;
    double wcpu;
} oldproc[NPROCS];

/* Stollen from  monitor-1.05 */
/* vmker struct is kernelstruct (undocumented) */
/* vmker seems to hold some kernels virtual memeory variables */
struct vmker {
  uint n0,n1,n2,n3,n4,n5,n6,n7,n8;
  uint totalmem;
  uint n10;
  uint freemem;
  uint n12,n13;
  uint totalvmem,freevmem;
} vmker;

/*
 * nlist arrary.
 */
#define X_VMKER         0
#define X_VMMINFO       1
#define X_SYSINFO       2
#define X_AVENRUN       3

static struct nlist nlst[] = {
    { "vmker" },	         /* 0 */
    { "vmminfo" }, 		 /* 1 */
    { "sysinfo" }, 		 /* 2 */
    { "avenrun"},		 /* 3 */
    { "" },
};
#define NLIST_NUM ((sizeof(nlst)/sizeof(struct nlist))-1)


/* get_process_info passes back a handle.  This is what it looks like: */

struct handle
{
    struct procinfo **next_proc;    /* points to next valid proc pointer */
    int remaining;              /* number of pointers remaining */
};


/*
 *  These definitions control the format of the per-process area
 */

static char header[] =
  "  PID X        PRI NICE  SIZE   RES STATE     TIME   WCPU      CPU COMMAND";

/* 0123456   -- field to fill in starts at header+6 */
#define UNAME_START 6

#define Proc_format \
        "%5d %-8.8s %3d %4d %5s %5s %-5s%6d:%02d %6.2f%% %6.2f%% %.14s"

int process_states[SSTOP+1];
char *procstatenames[] = {
    "", " sleeping, ", " ???, ", " running, ", " idle, ", " zombie, ", 
    " stopped, ",
    NULL
};

char *state_abbrev[] =
{
    "?", "sleep", "?",  "run", "T", "zombie", "sleep"
};


int cpu_states[CPUSTATES];
char *cpustatenames[] = {
    "idle", "user", "system", "wait", NULL
};

/* these are for detailing the memory statistics */

int memory_stats[4];
char *memorynames[] = {
    "M in use, ", "M free, ", "M available, ", NULL

};

/* useful externals */
extern int errno;
extern char *sys_errlist[];

long lseek();
long time();
long percentages();

machine_init(statics)

struct statics *statics;

{
    struct timeval now;
    int i;

    if ((kmem = open(KMEM, O_RDONLY)) == -1) {
        perror(KMEM);
        return(-1);
    }
     
    if (knlist(nlst, NLIST_NUM, sizeof(struct nlist)) == -1) {
	perror("nlist");
	return(-1);
    }

    sysinfo_offset = nlst[X_SYSINFO].n_value;
    vmker_offset = nlst[X_VMKER].n_value;
    avenrun_offset = nlst[X_AVENRUN].n_value;
    /* 
     * fill in the statics information 
     */
    statics->procstate_names = procstatenames;
    statics->cpustate_names = cpustatenames;
    statics->memory_names = memorynames;

    memset((char *)&oldproc, 0, sizeof(oldproc));

    for (i=0; i<NPROCS; i++) oldproc[i].pi_pid = -1;

    /* get the cp_time array */
    (void) getkval(sysinfo_offset, (int *)&sysinfox, sizeof(sysinfox),
                   "sysinfo");
    return(0);
}

char *format_header(uname_field)

register char *uname_field;

{
    register char *ptr;

    ptr = header + UNAME_START;
    while (*uname_field != '\0')
    {
	*ptr++ = *uname_field++;
    }

    return(header);
}

get_system_info(si)

struct system_info *si;

{
    long cpu[CPUSTATES];
    int i,j;
    int total;

    /* get the cp_time array */
    (void) getkval(sysinfo_offset, (int *)&sysinfox, sizeof(sysinfox),
                   "sysinfo");
     
    for (i=0; i<CPUSTATES; i++) {
	cpu[i] = sysinfox.cpu[i];
    }

    /* get load average array */
    (void) getkval(avenrun_offset, (int *)avenrun, sizeof(avenrun),
                   "_avenrun");

    for (i=0; i<3; i++) {
       si->load_avg[i] = (double)avenrun[i]/(double)(1<<16);
    }

    total = percentages(CPUSTATES, cpu_states, cpu, cp_old, cp_diff);
    si->cpustates = cpu_states;

    /* get the cp_time array */
    (void) getkval(vmker_offset, (int *)&vmker, sizeof(vmker),
                   "vmker");

    memory_stats[0] = (vmker.totalmem - vmker.freemem)*4.0/1024.0;
    memory_stats[1] = vmker.freemem*4.0/1024.0;
    memory_stats[2] = vmker.totalmem*4.0/1024.0;
    memory_stats[3] = -1;


    si->memory = memory_stats;
    si->procstates = process_states;
    si->p_total = 0;
    si->p_active = 0;
    si->last_pid = -1;
}
   

static struct handle handle;

caddr_t get_process_info(si, sel, compare)

struct system_info *si;
struct process_select *sel;
int (*compare)();

{
    register int i;
    register int total_procs;
    register int active_procs;
    register struct procinfo **prefp;
    register struct procinfo *pp;

    /* these are copied out of sel for speed */
    int show_idle;
    int show_system;
    int show_uid;
    int show_command;

    static struct timeval lasttime = {0, 0};
    struct timeval thistime;
    double timediff;
    double alpha, beta;
    double cpu;

    gettimeofday(&thistime, NULL);
    /*
     * To avoid divides, we keep times in nanoseconds.  This is
     * scaled by 1e7 rather than 1e9 so that when we divide we
     * get percent.
     */
    if (lasttime.tv_sec)
        timediff = (double)(thistime.tv_sec - lasttime.tv_sec) +
		   (double)(thistime.tv_usec - lasttime.tv_usec)/1000000.0;
    else
        timediff = 1.0;
    /*
     * constants for exponential average.  avg = alpha * new + beta * avg
     * The goal is 50% decay in 30 sec.  However if the sample period
     * is greater than 30 sec, there's not a lot we can do.
     */
    if (timediff < 30.0) {
        alpha = 0.5 * (timediff / 30.0);
        beta = 1.0 - alpha;
    } else {
        alpha = 0.5;
        beta = 0.5;
    }

    /* set up flags which define what we are going to select */
    show_idle = sel->idle;
    show_system = sel->system;
    show_uid = sel->uid != -1;
    show_command = sel->command != NULL;

    /* read all the proc structures in one fell swoop */

    nproc = getproc(procinfo, NPROCS, sizeof(struct procinfo));
    for (i=0; i<nproc; i++) {
	getuser(&procinfo[i], sizeof(struct procinfo),
	        &userinfo[i], sizeof(struct userinfo));
        cpu = (double)(userinfo[i].ui_ru.ru_utime.tv_sec +
                       userinfo[i].ui_ru.ru_stime.tv_sec) +
              (double)(userinfo[i].ui_ru.ru_utime.tv_usec +
                       userinfo[i].ui_ru.ru_stime.tv_usec)/1000000.0;
        if (procinfo[i].pi_pid == oldproc[i].pi_pid) {
	     oldproc[i].pct = (cpu - oldproc[i].cpu)/timediff;
	     oldproc[i].wcpu =  oldproc[i].wcpu*beta + oldproc[i].pct*alpha;
	} else {
	     oldproc[i].pi_pid = procinfo[i].pi_pid;
	     oldproc[i].pct = 0.0;
	     oldproc[i].wcpu = 0.0;
        }
	oldproc[i].cpu = cpu;
    }

    strcpy(userinfo[0].ui_comm, "swapper");

    /* count up process states and get pointers to interesting procs */

    total_procs = 0;
    active_procs = 0;
    bzero((char *)process_states, sizeof(process_states));
    prefp = pref;     
    for (pp = procinfo, i = 0; i < nproc; pp++, i++)
    {
        if (pp->pi_stat != 0)
        {
            total_procs++;
            process_states[pp->pi_stat]++;
            if ((pp->pi_stat != SZOMB) &&
		(show_idle || (pp->pi_stat == SRUN)) &&
                (!show_uid || pp->pi_uid == (uid_t)sel->uid))
            {
                *prefp++ = pp;
                active_procs++;
            }
	}
    }

    /* if requested, sort the "interesting" processes */
    if (compare != NULL)
    {
        qsort((char *)pref, active_procs, sizeof(struct procinfo *), compare);
    }

    si->p_total = pref_len = total_procs;
    si->p_active = active_procs;
    lasttime = thistime;

    /* pass back a handle */
    handle.next_proc = pref;
    handle.remaining = active_procs;
    return((caddr_t)&handle);
}

char fmt[128];		/* static area where result is built */


char *format_next_process(handle, get_userid)

caddr_t handle;
char *(*get_userid)();

{
    register struct procinfo *pp;
    register long cputime;
    struct user u;
    struct handle *hp;
    long rrsize = 0;
    long size = 0;
    int index = 0;

    /* find and remember the next proc structure */
    hp = (struct handle *)handle;
    pp = *(hp->next_proc++);
    hp->remaining--;
    
    index = pp - procinfo;
    size = 0;
    rrsize = 0;
    cputime = oldproc[index].cpu;
    /* format this entry */
    sprintf(fmt,
	    Proc_format,
	    pp->pi_pid,
	    (*get_userid)(pp->pi_uid),
	    pp->pi_pri,
	    (pp->pi_nice - NZERO),
	    format_k(pagetok(userinfo[index].ui_tsize/1024+
			     userinfo[index].ui_dvm*4)),
	    format_k(pagetok((userinfo[index].ui_drss+
			      userinfo[index].ui_trss)*4)),
	    state_abbrev[pp->pi_stat],
	    cputime / 60l,
	    cputime % 60l,
            oldproc[index].wcpu*100.0,
	    oldproc[index].pct*100.0,
	    printable(userinfo[index].ui_comm));

    /* return the result */
    return(fmt);
}


/*
 *  getkval(offset, ptr, size, refstr) - get a value out of the kernel.
 *	"offset" is the byte offset into the kernel for the desired value,
 *  	"ptr" points to a buffer into which the value is retrieved,
 *  	"size" is the size of the buffer (and the object to retrieve),
 *  	"refstr" is a reference string used when printing error meessages,
 *	    if "refstr" starts with a '!', then a failure on read will not
 *  	    be fatal (this may seem like a silly way to do things, but I
 *  	    really didn't want the overhead of another argument).
 *  	
 */

getkval(offset, ptr, size, refstr)

unsigned long offset;
int *ptr;
int size;
char *refstr;

{
    if (lseek(kmem, (long)offset, L_SET) == -1) {
        if (*refstr == '!')
            refstr++;
        (void) fprintf(stderr, "%s: lseek to %s: %s\n", KMEM,
                       refstr, strerror(errno));
        quit(23);
    }

    if (read(kmem, (char *)ptr, size) != size)
    {
	if (*refstr == '!')
	{
	    return(0);
	}
	else
	{
	    fprintf(stderr, "top: kvm_read for %s: %s\n",
		refstr, sys_errlist[errno]);
	    quit(23);
	}
    }
    return(1);
}
    
/* comparison routine for qsort */
/* NOTE: this is specific to the BSD proc structure, but it should
   give you a good place to start. */

/*
 *  proc_compare - comparison function for "qsort"
 *	Compares the resource consumption of two processes using five
 *  	distinct keys.  The keys (in descending order of importance) are:
 *  	percent cpu, cpu ticks, state, resident set size, total virtual
 *  	memory usage.  The process states are ordered as follows (from least
 *  	to most important):  WAIT, zombie, sleep, stop, start, run.  The
 *  	array declaration below maps a process state index into a number
 *  	that reflects this ordering.
 */

static unsigned char sorted_state[] =
{
    0,	/* not used		*/
    3,	/* sleep		*/
    1,	/* ABANDONED (WAIT)	*/
    6,	/* run			*/
    5,	/* start		*/
    2,	/* zombie		*/
    4	/* stop			*/
};
 
proc_compare(pp1, pp2)

struct procinfo **pp1;
struct procinfo **pp2;

{
    register struct procinfo *p1;
    register struct procinfo *p2;
    register int index1;
    register int index2;
    register int result;
    register double lresult;

    /* remove one level of indirection */
    p1 = *pp1;
    p2 = *pp2;
    index1 = p1 - procinfo;
    index2 = p2 - procinfo;

    /* compare percent cpu (pctcpu) */
    if ((lresult = oldproc[index2].pct - oldproc[index1].pct) == 0)
    {
        /* use process state to break the tie */
        if ((result = sorted_state[p2->pi_stat] -
 		      sorted_state[p1->pi_stat])  == 0)
	{
	    /* use priority to break the tie */
	    if ((result = p2->pi_pri - p1->pi_pri) == 0)
	    {
		    result = 0;
	    }
	}
    }
    else
    {
	result = lresult < 0 ? -1 : 1;
    }

    return(result);
}


int proc_owner(pid)
int pid;
{
    register int cnt;
    register struct procinfo **prefp;
    register struct procinfo *pp;

    prefp = pref;
    cnt =  pref_len;
    while (--cnt >= 0)
    {
	if ((pp = *prefp++)->pi_pid == (pid_t)pid)
	{
	    return((int)pp->pi_uid);
	}
    }
    return(-1);
}
