/*
 * top - a top users display for Unicos 6.1.X
 *
 * SYNOPSIS:  any YMP/XMP running Unicos 6.1.X
 *
 * DESCRIPTION:
 * Works for:
 *      Unicos 6.1.X
 *
 * MODE: 2711
 * UID: root
 * GID: sys
 * INSTALL: /usr/ucb/install
 *
 * AUTHOR: William L. Jones jones@chpc.utexas.edu
 */

#include <sys/types.h>
#include <sys/machd.h>
#include <sys/aoutdata.h>
#include <sys/sema.h>
#include <sys/param.h>
#include <sys/fcntl.h>
#include <sys/dir.h>
#include <sys/file.h>
#include <sys/table.h>
#include <sys/sysinfo.h>
#include <sys/time.h>
#include <sys/sysmacros.h>
#include <unistd.h>
#include <sys/proc.h>
#include <sys/pws.h>
#include <nlist.h>
#include <stdio.h>
#include <sys/category.h>

#include "top.h"
#include "machine.h"

/*
 * Defines.
 */
#define pagetok(size) ((size*4096)/(1024))

/*
 * Globals.
 */
static struct   proc    *proc;
static struct	proc    **pref;
static          int     pref_len = 0;
static int		proc_addr;
static int              nproc;
static long		hz;
static struct  sysinfo  Sysinfo;
static struct  pw       Pwold;

struct oldproc {
    pid_t  p_pid;
    double cpu;
    double pct;
    double wcpu;
} *oldproc = NULL;
	

/* get_process_info passes back a handle.  This is what it looks like: */

struct handle
{
    struct proc **next_proc;    /* points to next valid proc pointer */
    int remaining;              /* number of pointers remaining */
};



/*
 *  These definitions control the format of the per-process area
 */

static char header[] =
  "  PID X        PRI NICE    SIZE  STATE    TIME    WCPU     CPU  COMMAND";
/* 0123456   -- field to fill in starts at header+6 */
#define UNAME_START 6

#define Proc_format \
	"%5d %-8.8s %3d %4d %7s %-6s %5d:%02d %6.2f%% %6.2f%% %.14s"


int process_states[SSTOP+1];
char *procstatenames[] = {
    "", " sleeping, ", " waiting, ", " running, ", " idle, ",
    " zombie, ", " stopped, ",
    NULL
};

char *state_abbrev[] =
{
    "", "sleep", "wait", "run/xx\0\0", "idle", "zomb", "stop"
};


/* these are for detailing the cpu states */

int cpu_states[4];
char *cpustatenames[] = {
    "user", "idle", "system", NULL
};

/* these are for detailing the memory statistics */

int memory_stats[4];
char *memorynames[] = {
    "M used, ", "M locked, ", NULL
};

/* useful externals */
extern int errno;
extern char *sys_errlist[];

long lseek();
long time();
long percentages();

machine_init(statics)

struct statics *statics;

{
    static struct tbs pinfo;
    int i;
    struct timeval now;

    tabinfo(PROCTAB, &pinfo);
    nproc = pinfo.ent;
    proc_addr = (int)pinfo.addr;
    hz = sysconf(_SC_CLK_TCK);

    /*
     * Allocate storage.
     */
    proc = (struct proc *)malloc(nproc * sizeof(struct proc));
    oldproc = (struct oldproc *)malloc(nproc * sizeof(struct oldproc));
    pref = (struct proc **)malloc(nproc * sizeof(struct proc *));

    /* 
     * fill in the statics information 
     */
    statics->procstate_names = procstatenames;
    statics->cpustate_names = cpustatenames;
    statics->memory_names = memorynames;

    memset(oldproc, 0, sizeof(struct oldproc)*nproc);
 
    for (i=0; i<nproc; i++) oldproc[i].p_pid = -1;

    tabread(PWS, (char*)&Pwold, sizeof(struct pw), 0);
    return(0);
}

char *format_header(uname_field)

register char *uname_field;

{
    register char *ptr;

    ptr = header + UNAME_START;
    while (*uname_field != '\0')
    {
	*ptr++ = *uname_field++;
    }

    return(header);
}

get_system_info(si)

struct system_info *si;

{
    struct pw Pw;
    int i,j;
    double itime = 0.0;
    double utime = 0.0;
    double stime = 0.0;
    double wtime = 0.0;
    double etime = 0.0;

    tabread(SINFO, &Sysinfo, sizeof(struct sysinfo), 0);

    for (i=0; i<3; i++) {
        si->load_avg[i] = Sysinfo.avenrun[i];
    }

    tabread(PWS, (char*)&Pw, sizeof(struct pw), 0);
    for (i = 0; i <Pw.pw_ccpu; i++) {
        if (!(Pw.pws[i].pw_cpuflg&PW_STOP)) {
            utime += (double)(Pw.pws[i].pw_userc)/(double)hz ;
            itime += (double)(Pw.pws[i].pw_idlec)/(double)hz ;
            wtime += (double)(Pw.pws[i].pw_syswc)/(double)hz ;
            stime += (double)(Pw.pws[i].pw_unixc)/(double)hz ;
            utime -= (double)(Pwold.pws[i].pw_userc)/(double)hz ;
            itime -= (double)(Pwold.pws[i].pw_idlec)/(double)hz ;
            wtime -= (double)(Pwold.pws[i].pw_syswc)/(double)hz ;
            stime -= (double)(Pwold.pws[i].pw_unixc)/(double)hz ;
        }
    }
    etime = utime + itime + wtime + stime;
    if (etime == 0.0) etime = 1.0;
    cpu_states[0] = (1000.0*utime)/etime;
    cpu_states[1] = (1000.0*itime)/etime;
    cpu_states[2] = (1000.0*(stime+wtime))/etime;
    Pwold = Pw;
    si->cpustates = cpu_states;
    memory_stats[0] = ctob(Sysinfo.umemused)/(1024*1024);  
    memory_stats[1] = ctob(Sysinfo.memlock)/(1024*1024);;
    memory_stats[2] = 0;
    si->memory = memory_stats;
    si->procstates = process_states;
    si->p_total = 0;
    si->p_active = 0;
    si->last_pid = -1;

}

static struct handle handle;

caddr_t get_process_info(si, sel, compare)

struct system_info *si;
struct process_select *sel;
int (*compare)();

{
    register int i;
    register int total_procs;
    register int active_procs;
    register struct proc **prefp;
    register struct proc *pp;
    register struct pcomm *pc;
    struct   timeval now;
    double percent;
    int index;

    /* these are copied out of sel for speed */
    int show_idle;
    int show_system;
    int show_uid;
    int show_command;

    static struct timeval lasttime = {0, 0};
    struct timeval thistime;
    double timediff;
    double alpha, beta;
    double cpu;
 
    /* read all the proc structures in one fell swoop */

    tabread(PROCTAB, (char *)&proc[0], sizeof(struct proc)*nproc, 0);
    gettimeofday(&thistime, NULL);

    /*
     * To avoid divides, we keep times in nanoseconds.  This is
     * scaled by 1e7 rather than 1e9 so that when we divide we
     * get percent.
     */
    if (lasttime.tv_sec)
        timediff = (double)(thistime.tv_sec - lasttime.tv_sec) +
                   (double)(thistime.tv_usec - lasttime.tv_usec)/1000000.0;
    else
        timediff = 1.0;
    /*
     * constants for exponential average.  avg = alpha * new + beta * avg
     * The goal is 50% decay in 30 sec.  However if the sample period
     * is greater than 30 sec, there's not a lot we can do.
     */
    if (timediff < 30.0) {
        alpha = 0.5 * (timediff / 30.0);
        beta = 1.0 - alpha;
    } else {
        alpha = 0.5;
        beta = 0.5;
    }


    /* set up flags which define what we are going to select */
    show_idle = sel->idle;
    show_system = sel->system;
    show_uid = sel->uid != -1;
    show_command = sel->command != NULL;

    /* count up process states and get pointers to interesting procs */

    total_procs = 0;
    active_procs = 0;
    bzero((char *)process_states, sizeof(process_states));
    prefp = pref;     
    gettimeofday(&now, NULL);
    for (pp = proc, i = 0; i < nproc; pp++, i++)
    {
	cpu = (double)(pp->p_utime + pp->p_stime)/(double)hz;

        if (pp->p_pid == oldproc[i].p_pid) {
             oldproc[i].pct = (cpu - oldproc[i].cpu)/timediff;
             oldproc[i].wcpu =  oldproc[i].wcpu*beta + oldproc[i].pct*alpha;
        } else {
             oldproc[i].p_pid = pp->p_pid;
             oldproc[i].pct = 0.0;
             oldproc[i].wcpu = 0.0;
        }
        oldproc[i].cpu = cpu;
 
        index = (struct proc *)pp->p_pc - (struct proc *) proc_addr;
        if (pp->p_stat != 0 &&
            (show_system || ((proc[index].p_pcomm.pc_flag & PC_SYS) == 0)))
        {
            total_procs++;
            process_states[pp->p_stat]++;
            if ((pp->p_stat != SZOMB) && 
                (show_idle || (pp->p_stat == SRUN)) &&
                (!show_uid || proc[index].p_pcomm.pc_uid == (uid_t)sel->uid))
            {
                *prefp++ = pp;
                active_procs++;
            }
	}
    }

    /* if requested, sort the "interesting" processes */
    if (compare != NULL)
    {
        qsort((char *)pref, active_procs, sizeof(struct proc *), compare);
    }

    si->p_total = pref_len = total_procs;
    si->p_active = active_procs;
    lasttime = thistime;

    /* pass back a handle */
    handle.next_proc = pref;
    handle.remaining = active_procs;
    return((caddr_t)&handle);
}

char fmt[128];		/* static area where result is built */


char *format_next_process(handle, get_userid)

caddr_t handle;
char *(*get_userid)();

{
    register struct proc *pp;
    register long cputime;
    register double pct;
    struct handle *hp;
    long rrsize = 0;
    long size = 0;
    int index;
    time_t now;
    register struct pcomm *pc;

    time(&now);

    /* find and remember the next proc structure */
    hp = (struct handle *)handle;
    pp = *(hp->next_proc++);
    hp->remaining--;

    index = (struct proc *)pp - (struct proc *)proc;
    cputime = (pp->p_utime + pp->p_stime)/hz;

    index = (struct proc *)pp->p_pc - (struct proc *) proc_addr;

    /* set size */


    if (pp->p_stat == SRUN) {
	if (pp->p_pflag & P_CONN) {
		sprintf(state_abbrev[SRUN], "run/%02d", pp->p_cpun);
	} else {
		sprintf(state_abbrev[SRUN], "run   ");
	}
    }
   
    /* format this entry */
    sprintf(fmt,
	    Proc_format,
	    pp->p_pid,
	    (*get_userid)(proc[index].p_pcomm.pc_uid),
	    pp->p_pri,
	    pp->p_nice - NZERO,
	    format_k(ctob(proc[index].p_pcomm.pc_size)/1024),
	    state_abbrev[pp->p_stat],
	    cputime / 60l,
	    cputime % 60l,
	    100.0 * oldproc[index].wcpu,
	    100.0 * oldproc[index].pct,
	    printable(pp->p_comm));

    /* return the result */
    return(fmt);
}

    
/* comparison routine for qsort */
/* NOTE: this is specific to the BSD proc structure, but it should
   give you a good place to start. */

/*
 *  proc_compare - comparison function for "qsort"
 *	Compares the resource consumption of two processes using five
 *  	distinct keys.  The keys (in descending order of importance) are:
 *  	percent cpu, cpu ticks, state, resident set size, total virtual
 *  	memory usage.  The process states are ordered as follows (from least
 *  	to most important):  WAIT, zombie, sleep, stop, start, run.  The
 *  	array declaration below maps a process state index into a number
 *  	that reflects this ordering.
 */

static unsigned char sorted_state[] =
{
    0,	/* not used		*/
    3,	/* sleep		*/
    1,	/* ABANDONED (WAIT)	*/
    6,	/* run			*/
    5,	/* start		*/
    2,	/* zombie		*/
    4	/* stop			*/
};
 
proc_compare(pp1, pp2)

struct proc **pp1;
struct proc **pp2;

{
    register struct proc *p1;
    register struct proc *p2;
    register int result;
    register double lresult;
    int i1, i2;

    /* remove one level of indirection */
    p1 = *pp1;
    p2 = *pp2;

    i1 = p1 - proc;
    i2 = p2 - proc;

    /* compare percent cpu (pctcpu) */
    if ((lresult = oldproc[i2].pct - oldproc[i1].pct) == 0.0)
    {
        /* use process state to break the tie */
        if ((result = sorted_state[p2->p_stat] -
  	              sorted_state[p1->p_stat])  == 0)
     	{
	     /* use priority to break the tie */
	     if ((result = p2->p_pri - p1->p_pri) == 0)
	     {
		time_t t1, t2;
	        /* use cpu usage to break the tie */

		t2 = p2->p_utime + p2->p_stime;
		t1 = p1->p_utime + p1->p_stime;
	        result = t2 - t1;
	     }
	}
    }
    else
    {
	result = lresult < 0 ? -1 : 1;
    }

    if (result != 0) result = result < 0 ? -1 : 1;

    return(result);
}


int setpriority (dummy,  who, niceval)
int dummy;
int who;
int niceval;
{
    return nicem(C_PROC, who, niceval);
}



int proc_owner(pid)

int pid;
{
    register int cnt;
    register struct proc **prefp;
    register struct proc *pp;

    prefp = pref;
    cnt =  pref_len;
    while (--cnt >= 0)
    {
	if ((pp = *prefp++)->p_pid == (pid_t)pid)
	{
	    return((int)pp->p_pcomm.pc_uid);
	}
    }
    return(-1);
}


