#!/usr/bin/ruby
#
# x-math.rbѥ˥åȥƥ
# $Id: test_mathml.rb 430 2006-08-16 14:23:49Z  $
#
# Copyright (C) 2005, KURODA Hiraku <hiraku@hinet.mydns.jp>
# You can redistribute it and/or modify it under GPL2.
#
require 'test/unit'
require 'mathml.rb'
puts `clear`

module Misc4TC_LaTeX
	def strip(str)
		str.gsub(/>\s*</, "><")
	end

	def smml(str, display_style=false, p=MathML::LaTeX::Parser.new)
		mml = p.parse(str, display_style).to_s
		strip(mml)[/\A<math [^>]*>(.*)<\/math>\Z/m, 1]
	end

	def test_strip
		assert_equal("><", strip(">\n \t<"))
		assert_equal("", smml(""))
		assert_equal("<mn>1</mn>", smml("1"))
	end

	def check_chr(stag, etag, str, print=false)
		str.each_byte do |b|
			assert_equal("#{stag}#{b.chr}#{etag}", smml(b.chr))
			puts smml(b.chr) if print
		end
	end

	def check_hash(stag, etag, hash, print=false)
		hash.each do |k, v|
			e = "#{stag}#{v}#{etag}"
			s = smml(k)
			assert_equal(e, s)
			puts "#{k} => #{s}" if print
		end
	end

	def check_entity(stag, etag, hash, print=false)
		h = Hash.new
		hash.each do |k, v|
			h[k] = "&#{v};"
		end
		check_hash(stag, etag, h, print)
	end

	def parse_error(e)
		[e.message, e.correct, e.remain].flatten
	end
end

class TC_MathML_LaTeX_Misc < Test::Unit::TestCase
	include MathML::LaTeX
	include Misc
	include Misc4TC_LaTeX

	def test_slice_block
		src = "test"
		assert_equal("t", slice_block(src))
		assert_equal("test", src)
		assert_equal("t", slice_block!(src))
		assert_equal("est", src)

		e = " { {}{}{{}{}} }"
		src = "#{e} test"
		assert_equal(e, slice_block(src))
		assert_equal(e, slice_block!(src))
		assert_equal(" test", src)

		src = ' \command test'
		assert_equal(' \command', slice_block(src))
		assert_equal(' \command', slice_block!(src))
		assert_equal(" test", src)

		src = '\_test'
		assert_equal('\_', slice_block(src))
		assert_equal('\_', slice_block!(src))
		assert_equal("test", src)

		src = ""
		assert_equal("", slice_block(src))
		assert_equal("", slice_block!(src))
		assert_equal("", src)

		src = " test"
		assert_equal(" t", slice_block(src))
		assert_equal(" t", slice_block!(src))
		assert_equal("est", src)

		src = " "
		assert_equal(" ", slice_block(src))
		assert_equal(" ", slice_block!(src))
		assert_equal("", src)

		src = "{test"
		e = assert_raises(BraceNotClosed){slice_block(src)}

		src = ' \begin{env} test \end{env}'
		assert_equal(' \begin', slice_block(src))
		assert_equal(' \begin{env} test \end{env}', slice_block(src, true))
		s = src.dup
		assert_equal(' \begin', slice_block!(s))
		assert_equal('{env} test \end{env}', s)
		s = src.dup
		assert_equal(' \begin{env} test \end{env}', slice_block!(s, true))
		assert_equal("", s)
	end

	def test_slice_brace
		src = " { t { es } t } "
		e = " t { es } t "
		assert_equal(e, slice_brace!(src))
		assert_equal(e, src)
	end

	def test_slice_environment
		src = "a"
		assert_raises(NotEnvironment){slice_environment(src)}

		src = '\begin{env}'
		assert_raises(EnvironmentNotEnd){slice_environment(src)}

		src = ' \begin{env} test \end{emv} test'
		assert_equal(' \begin{env} test \end{emv}', slice_environment(src))
		assert_equal(' \begin{env} test \end{emv}', slice_environment!(src))
		assert_equal(" test", src)
	end
end

class TC_MathML_LaTeX_Parser < Test::Unit::TestCase
	include MathML::LaTeX
	include Misc4TC_LaTeX

	def test_nobody
		p = Parser.new
		assert_equal("<math xmlns='http://www.w3.org/1998/Math/MathML' display='inline'></math>", p.parse("").to_s)
		assert_equal("<math xmlns='http://www.w3.org/1998/Math/MathML' display='block'></math>", p.parse("", true).to_s)
		assert_equal("<math xmlns='http://www.w3.org/1998/Math/MathML' display='inline'></math>", p.parse("", false).to_s)
	end

	def test_ignore_space
		assert_equal("<mrow><mi>a</mi></mrow>", smml("{ a }"))
	end

	def test_parse_error
		src = 'a\hoge c'
		e = assert_raises(ParseError){smml(src)}
		assert_equal(["Undefined command.", "a", '\hoge c'], parse_error(e))

		src = '\sqrt\sqrt1'
		e = assert_raises(ParseError){smml(src)}
		assert_equal(["Syntax error.", '\sqrt\sqrt', "1"], parse_error(e))

		src = "a{b"
		e = assert_raises(ParseError){smml(src)}
		assert_equal(["Brace not closed.", "a", "{b"], parse_error(e))
	end

	def test_numerics
		assert_equal("<mn>1234567890</mn>", smml('1234567890'))
		assert_equal("<mn>1.2</mn>", smml('1.2'))
		assert_equal("<mn>1</mn><mo>.</mo>", smml('1.'))
		assert_equal("<mn>.2</mn>", smml('.2'))
		assert_equal("<mn>1.2</mn><mn>.3</mn>", smml('1.2.3'))
	end

	def test_alphabets
		check_chr("<mi>", "</mi>", "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ")
	end

	def test_non_alphabet_command
		assert_equal("<mspace width='0.167em'/>", smml('\,'))
		assert_equal("<mo>&Vert;</mo>", smml('\|'))
	end

	def test_operators
		check_chr("<mo>", "</mo>", ",.+-*=/()[]'|;:!\"")
		check_entity("<mo>", "</mo>", {"<"=>"lt", ">"=>"gt"})
		check_hash("<mo>", "</mo>", {'\backslash'=>'\\', '\%'=>'%', '\{'=>'{', '\}'=>'}', '\$'=>'$', '\#'=>'#'})
	end

	def test_sqrt
		assert_equal("<msqrt><mi>a</mi></msqrt>", smml('\sqrt a'))
	end

	def test_subsup
		assert_equal("<msubsup><mi>a</mi><mi>b</mi><mi>c</mi></msubsup>", smml("a_b^c"))
		assert_equal("<msub><mi>a</mi><mi>b</mi></msub>", smml("a_b"))
		assert_equal("<msup><mi>a</mi><mi>b</mi></msup>", smml("a^b"))

		e = assert_raises(ParseError){smml("a_b_c")}
		assert_equal(["Double subscript.", "a_b", "_c"], parse_error(e))
		e = assert_raises(ParseError){smml("a^b^c")}
		assert_equal(["Double superscript.", "a^b", "^c"], parse_error(e))
		e = assert_raises(ParseError){smml("a_")}
		assert_equal(["Subscript not exist.", "a_", ""], parse_error(e))
		e = assert_raises(ParseError){smml("a^")}
		assert_equal(["Superscript not exist.", "a^", ""], parse_error(e))
	end

	def test_underover
		assert_equal("<munderover><mo>&sum;</mo><mi>a</mi><mi>b</mi></munderover>", smml('\sum_a^b', true))
		assert_equal("<munderover><mo>&sum;</mo><mi>a</mi><mi>b</mi></munderover>", smml('\sum_a^b', false))
		assert_equal("<munder><mo>&sum;</mo><mi>a</mi></munder>", smml('\sum_a'))
		assert_equal("<mover><mo>&sum;</mo><mi>a</mi></mover>", smml('\sum^a'))
		e = assert_raises(ParseError){smml('\sum_b_c')}
		assert_equal(["Double subscript.", '\sum_b', "_c"], parse_error(e))
		e = assert_raises(ParseError){smml('\sum^b^c')}
		assert_equal(["Double superscript.", '\sum^b', "^c"], parse_error(e))
		e = assert_raises(ParseError){smml('\sum_')}
		assert_equal(["Subscript not exist.", '\sum_', ""], parse_error(e))
		e = assert_raises(ParseError){smml('\sum^')}
		assert_equal(["Superscript not exist.", '\sum^', ""], parse_error(e))
	end

	def test_fonts
		assert_equal("<mi>a</mi><mrow><mi mathvariant='bold'>b</mi><mi mathvariant='bold'>c</mi></mrow><mi>d</mi>", smml('a{\bf b c}d'))
		assert_equal("<mi mathvariant='bold'>a</mi><mrow><mi>b</mi><mi>c</mi></mrow><mi mathvariant='bold'>d</mi>", smml('\bf a{\it b c}d'))
		assert_equal("<mi>a</mi><mrow><mi mathvariant='normal'>b</mi><mi mathvariant='normal'>c</mi></mrow><mi>d</mi>", smml('a{\rm b c}d'))

		assert_equal("<mi>a</mi><mrow><mrow><mi mathvariant='bold'>b</mi><mi mathvariant='bold'>c</mi></mrow></mrow><mi>d</mi>", smml('a \mathbf{bc}d'))
		assert_equal("<mi mathvariant='bold'>a</mi><mrow><mrow><mi>b</mi><mi>c</mi></mrow></mrow><mi mathvariant='bold'>d</mi>", smml('\bf a \mathit{bc} d'))
		assert_equal("<mi>a</mi><mrow><mrow><mi mathvariant='normal'>b</mi><mi mathvariant='normal'>c</mi></mrow></mrow><mi>d</mi>", smml('a\mathrm{bc}d'))

		assert_equal("<mi>a</mi><mrow><mrow><mi>&bopf;</mi><mi>&copf;</mi></mrow></mrow><mi>d</mi>", smml('a \mathbb{b c} d'))
		assert_equal("<mi>a</mi><mrow><mrow><mi>&bscr;</mi><mi>&cscr;</mi></mrow></mrow><mi>d</mi>", smml('a \mathscr{b c} d'))
		assert_equal("<mi>a</mi><mrow><mrow><mi>&bfr;</mi><mi>&cfr;</mi></mrow></mrow><mi>d</mi>", smml('a \mathfrak{b c} d'))
		e = assert_raises(ParseError){smml('\mathit')}
		assert_equal(["Syntax error.", '\mathit', ""], parse_error(e))
		e = assert_raises(ParseError){smml('\mathrm')}
		assert_equal(["Syntax error.", '\mathrm', ""], parse_error(e))
		e = assert_raises(ParseError){smml('\mathbf')}
		assert_equal(["Syntax error.", '\mathbf', ""], parse_error(e))
		e = assert_raises(ParseError){smml('\mathbb')}
		assert_equal(["Syntax error.", '\mathbb', ""], parse_error(e))
		e = assert_raises(ParseError){smml('\mathscr')}
		assert_equal(["Syntax error.", '\mathscr', ""], parse_error(e))
		e = assert_raises(ParseError){smml('\mathfrak')}
		assert_equal(["Syntax error.", '\mathfrak', ""], parse_error(e))
	end

	def test_mbox
		assert_equal("<mi>a</mi><mtext>b c</mtext><mi>d</mi>", smml('a\mbox{b c}d'))
	end

	def test_frac
		assert_equal("<mfrac><mi>a</mi><mi>b</mi></mfrac>", smml('\frac ab'))
		assert_equal("<mfrac><mn>1</mn><mn>2</mn></mfrac>", smml('\frac12'))

		e = assert_raises(ParseError){smml('\frac a')}
		assert_equal(["Syntax error.", '\frac a', ""], parse_error(e))
	end

	def test_environment
		e = assert_raises(ParseError){smml('{\begin}remain')}
		assert_equal(["Environment name not exist.", '{\begin', '}remain'], parse_error(e))

		e = assert_raises(ParseError){smml('{\begin{array}{c}dummy}remain')}
		assert_equal(['Matching \end not exist.', '{\begin{array}{c}dummy', '}remain'], parse_error(e))

		e = assert_raises(ParseError){smml('\begin{array}c dummy\end{test}')}
		assert_equal(["Environment mismatched.", '\begin{array}c dummy\end', "{test}"], parse_error(e))
	end

	def test_array_env
		assert_equal("<mtable columnalign='left right center'><mtr><mtd><mi>a</mi></mtd><mtd><mi>b</mi></mtd><mtd><mi>c</mi></mtd></mtr><mtr><mtd><mi>d</mi></mtd><mtd><mi>e</mi></mtd><mtd><mi>f</mi></mtd></mtr></mtable>", smml('\begin{array}{lrc} a & b & c \\\\ d & e & f \\\\ \end{array}'))

		assert_equal("<mtable columnalign='left right center'><mtr><mtd><mi>a</mi></mtd><mtd><mi>b</mi></mtd><mtd><mi>c</mi></mtd></mtr><mtr><mtd><mi>d</mi></mtd><mtd><mi>e</mi></mtd><mtd><mi>f</mi></mtd></mtr></mtable>", smml('\begin{array}{lrc}a&b&c\\\\d&e&f \end{array}'))

		assert_equal("<mtable></mtable>", smml('\begin{array}{c}\end{array}'))

		e = assert_raises(ParseError){smml('\begin{array}\end{array}')}
		assert_equal(['Syntax error.', '\begin{array}', '\end{array}'], parse_error(e))

		e = assert_raises(ParseError){smml('\begin{array}{a}\end{array}')}
		assert_equal(["Syntax error.", '\begin{array}{', 'a}\end{array}'], parse_error(e))

		e = assert_raises(ParseError){smml('\begin{array}{cc}a\\\\b&c\end{array}')}
		assert_equal(["Need more column.", '\begin{array}{cc}a', '\\\\b&c\end{array}'], parse_error(e))

		e = assert_raises(ParseError){smml('\begin{array}{cc}a\end{array}')}
		assert_equal(["Need more column.", '\begin{array}{cc}a', '\end{array}'], parse_error(e))

		e = assert_raises(ParseError){smml('\begin{array}{c}a&\end{array}')}
		assert_equal(["Too many column.", '\begin{array}{c}a', '&\end{array}'], parse_error(e))

		assert_equal("<mtable><mtr><mtd></mtd><mtd></mtd></mtr></mtable>", smml('\begin{array}{cc}&\end{array}'))

		assert_equal("<mfenced open='{' close='}'><mrow><mtable><mtr><mtd><msub><mi>a</mi><mi>b</mi></msub></mtd></mtr></mtable></mrow></mfenced>", smml('\left\{\begin{array}ca_b\end{array}\right\}'))

		assert_equal("<mtable columnalign='center left center center center right center'><mtr><mtd><mrow><msub><mi>a</mi><mn>1</mn></msub></mrow></mtd><mtd><mi>A</mi></mtd><mtd><mi>b</mi></mtd><mtd><mi>B</mi></mtd><mtd><mi>c</mi></mtd><mtd><mi>C</mi></mtd><mtd><mi>d</mi></mtd></mtr></mtable>", smml('\begin{array}{@{a_1}l@bc@cr@d}A&B&C\end{array}'))

		assert_equal("<mfenced open='{' close='}'><mrow><mtable><mtr><mtd><msub><mi>a</mi><mi>b</mi></msub></mtd></mtr></mtable></mrow></mfenced>", smml('\left\{\begin{array}ca_b\end{array}\right\}'))

		assert_equal("<mtable columnlines='solid'><mtr><mtd><mi>a</mi></mtd><mtd><mi>b</mi></mtd></mtr><mtr><mtd><mi>c</mi></mtd><mtd><mi>d</mi></mtd></mtr></mtable>", smml('\begin{array}{c|c}a&b\\\\c&d\end{array}'))
		assert_equal("<mtable columnlines='solid solid'><mtr><mtd></mtd><mtd><mi>a</mi></mtd><mtd></mtd></mtr><mtr><mtd></mtd><mtd><mi>c</mi></mtd><mtd></mtd></mtr></mtable>", smml('\begin{array}{|c|}a\\\\c\end{array}'))
		assert_equal("<mtable rowlines='solid'><mtr></mtr><mtr><mtd><mi>c</mi></mtd></mtr></mtable>", smml('\begin{array}{c}\hline c\end{array}'))
		assert_equal("<mtable rowlines='solid'><mtr><mtd><mi>c</mi></mtd><mtd><mi>a</mi></mtd><mtd><mi>c</mi></mtd><mtd><mi>c</mi></mtd></mtr><mtr><mtd></mtd><mtd></mtd><mtd></mtd><mtd></mtd></mtr></mtable>", smml('\begin{array}{c@acc}c&c&c\\\\\hline\end{array}'))
	end

	def test_leftright
		assert_equal("<mfenced open='(' close=')'><mrow><mfrac><mn>1</mn><mn>2</mn></mfrac></mrow></mfenced>", smml('\left(\frac12\right)'))

		assert_equal("<mfenced open='&lfloor;' close='&rfloor;'><mrow><mi>a</mi></mrow></mfenced>", smml('\left \lfloor a\right \rfloor'))

		assert_equal("<mfenced open='{' close='}'><mrow><mi>a</mi></mrow></mfenced>", smml('\left \{ a \right \}'))

		assert_equal("<mfenced open='{' close='}'><mrow><mtable><mtr><mtd><mtable><mtr><mtd><mi>a</mi></mtd></mtr></mtable></mtd></mtr></mtable></mrow></mfenced>", smml('\left\{\begin{array}c\begin{array}ca\end{array}\end{array}\right\}'))

		assert_equal("<mfenced open='(' close=')'><mrow><munder><mo>&sum;</mo><mi>a</mi></munder></mrow></mfenced>", smml('\left(\sum_a\right)'))
	end

	def test_over
		assert_equal("<mover><mi>a</mi><mo>&circ;</mo></mover>", smml('\hat a'))
		e = assert_raises(ParseError){smml('{\hat}a')}
		assert_equal(["Syntax error.", '{\hat', '}a'], parse_error(e))
	end

	def test_under
		assert_equal("<munder><mi>a</mi><mo>&macr;</mo></munder>", smml('\underline a'))
		e = assert_raises(ParseError){smml('{\underline}a')}
		assert_equal(["Syntax error.", '{\underline', '}a'], parse_error(e))
	end

	def test_stackrel
		assert_equal("<mover><mo>=</mo><mo>&rightarrow;</mo></mover>", smml('\stackrel\to='))
	end

	def test_comment
		assert_equal("<mi>a</mi>", smml('a%b'))
	end

	def test_entity
		p = Parser.new
		e = assert_raises(ParseError){smml('\entity{therefore}', false, p)}
		assert_equal(["Unregistered entity.", '\entity{', "therefore}"], parse_error(e))

		p.unsecure_entity = true
		assert_equal("<mo>&therefore;</mo>", smml('\entity{therefore}', false, p))

		p.unsecure_entity = false
		e = assert_raises(ParseError){smml('\entity{therefore}', false, p)}
		assert_equal(["Unregistered entity.", '\entity{', "therefore}"], parse_error(e))

		p.add_entity_list(['therefore'])
		assert_equal("<mo>&therefore;</mo>", smml('\entity{therefore}', false, p))
	end

	def test_backslash
		assert_equal('<br xmlns="http://www.w3.org/1999/xhtml"/>', smml('\\\\'))
	end
end
