
/*LICENSE_START*/
/*
 *  Copyright (C) 2024 Washington University School of Medicine
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
/*LICENSE_END*/

#include <algorithm>
#define __ZARR_DATA_TYPE_ENUM_DECLARE__
#include "ZarrDataTypeEnum.h"
#undef __ZARR_DATA_TYPE_ENUM_DECLARE__

#include "CaretAssert.h"

using namespace caret;

    
/**
 * \class caret::ZarrDataTypeEnum 
 * \brief Data type for ZARR data
 *
 * https://zarr-specs.readthedocs.io/en/latest/v2/v2.0.html
 *
 * Using this enumerated type in the GUI with an EnumComboBoxTemplate
 * 
 * Header File (.h)
 *     Forward declare the data type:
 *         class EnumComboBoxTemplate;
 * 
 *     Declare the member:
 *         EnumComboBoxTemplate* m_zarrDataTypeEnumComboBox;
 * 
 *     Declare a slot that is called when user changes selection
 *         private slots:
 *             void zarrDataTypeEnumComboBoxItemActivated();
 * 
 * Implementation File (.cxx)
 *     Include the header files
 *         #include "EnumComboBoxTemplate.h"
 *         #include "ZarrDataTypeEnum.h"
 * 
 *     Instatiate:
 *         m_zarrDataTypeEnumComboBox = new EnumComboBoxTemplate(this);
 *         m_zarrDataTypeEnumComboBox->setup<ZarrDataTypeEnum,ZarrDataTypeEnum::Enum>();
 * 
 *     Get notified when the user changes the selection: 
 *         QObject::connect(m_zarrDataTypeEnumComboBox, SIGNAL(itemActivated()),
 *                          this, SLOT(zarrDataTypeEnumComboBoxItemActivated()));
 * 
 *     Update the selection:
 *         m_zarrDataTypeEnumComboBox->setSelectedItem<ZarrDataTypeEnum,ZarrDataTypeEnum::Enum>(NEW_VALUE);
 * 
 *     Read the selection:
 *         const ZarrDataTypeEnum::Enum VARIABLE = m_zarrDataTypeEnumComboBox->getSelectedItem<ZarrDataTypeEnum,ZarrDataTypeEnum::Enum>();
 * 
 */

/*
switch (value) {
    case ZarrDataTypeEnum::UNKNOWN:
        break;
    case ZarrDataTypeEnum::INT_8:
        break;
    case ZarrDataTypeEnum::UINT_8:
        break;
    case ZarrDataTypeEnum::INT_16:
        break;
    case ZarrDataTypeEnum::UINT_16:
        break;
    case ZarrDataTypeEnum::INT_32:
        break;
    case ZarrDataTypeEnum::UINT_32:
        break;
    case ZarrDataTypeEnum::INT_64:
        break;
    case ZarrDataTypeEnum::UINT_64:
        break;
    case ZarrDataTypeEnum::FLOAT_32:
        break;
    case ZarrDataTypeEnum::FLOAT_64:
        break;
}
*/

/**
 * Constructor.
 *
 * @param enumValue
 *    An enumerated value.
 * @param name
 *    Name of enumerated value.
 *
 * @param guiName
 *    User-friendly name for use in user-interface.
 */
ZarrDataTypeEnum::ZarrDataTypeEnum(const Enum enumValue,
                           const AString& name,
                           const AString& guiName)
{
    this->enumValue = enumValue;
    this->integerCode = integerCodeCounter++;
    this->name = name;
    this->guiName = guiName;
}

/**
 * Destructor.
 */
ZarrDataTypeEnum::~ZarrDataTypeEnum()
{
}

/**
 * Initialize the enumerated metadata.
 */
void
ZarrDataTypeEnum::initialize()
{
    if (initializedFlag) {
        return;
    }
    initializedFlag = true;

    enumData.push_back(ZarrDataTypeEnum(UNKNOWN, 
                                    "UNKNOWN", 
                                    "Unknown"));
    
    enumData.push_back(ZarrDataTypeEnum(INT_8, 
                                    "INT_8", 
                                    "Int 8"));
    
    enumData.push_back(ZarrDataTypeEnum(UINT_8, 
                                    "UINT_8", 
                                    "Uint 8"));
    
    enumData.push_back(ZarrDataTypeEnum(INT_16, 
                                    "INT_16", 
                                    "Int 16"));
    
    enumData.push_back(ZarrDataTypeEnum(UINT_16, 
                                    "UINT_16", 
                                    "Uint 16"));
    
    enumData.push_back(ZarrDataTypeEnum(INT_32, 
                                    "INT_32", 
                                    "Int 32"));
    
    enumData.push_back(ZarrDataTypeEnum(UINT_32, 
                                    "UINT_32", 
                                    "Uint 32"));
    
    enumData.push_back(ZarrDataTypeEnum(INT_64, 
                                    "INT_64", 
                                    "Int 64"));
    
    enumData.push_back(ZarrDataTypeEnum(UINT_64, 
                                    "UINT_64", 
                                    "Uint 64"));
    
    enumData.push_back(ZarrDataTypeEnum(FLOAT_32, 
                                    "FLOAT_32", 
                                    "Float 32"));
    
    enumData.push_back(ZarrDataTypeEnum(FLOAT_64, 
                                    "FLOAT_64", 
                                    "Float 64"));
    
}

/**
 * Find the data for and enumerated value.
 * @param enumValue
 *     The enumerated value.
 * @return Pointer to data for this enumerated type
 * or NULL if no data for type or if type is invalid.
 */
const ZarrDataTypeEnum*
ZarrDataTypeEnum::findData(const Enum enumValue)
{
    if (initializedFlag == false) initialize();

    size_t num = enumData.size();
    for (size_t i = 0; i < num; i++) {
        const ZarrDataTypeEnum* d = &enumData[i];
        if (d->enumValue == enumValue) {
            return d;
        }
    }

    return NULL;
}

/**
 * Get a string representation of the enumerated type.
 * @param enumValue 
 *     Enumerated value.
 * @return 
 *     String representing enumerated value.
 */
AString 
ZarrDataTypeEnum::toName(Enum enumValue) {
    if (initializedFlag == false) initialize();
    
    const ZarrDataTypeEnum* enumInstance = findData(enumValue);
    return enumInstance->name;
}

/**
 * Get an enumerated value corresponding to its name.
 * @param name 
 *     Name of enumerated value.
 * @param isValidOut 
 *     If not NULL, it is set indicating that a
 *     enum value exists for the input name.
 * @return 
 *     Enumerated value.
 */
ZarrDataTypeEnum::Enum 
ZarrDataTypeEnum::fromName(const AString& name, bool* isValidOut)
{
    if (initializedFlag == false) initialize();
    
    bool validFlag = false;
    Enum enumValue = ZarrDataTypeEnum::enumData[0].enumValue;
    
    for (std::vector<ZarrDataTypeEnum>::iterator iter = enumData.begin();
         iter != enumData.end();
         iter++) {
        const ZarrDataTypeEnum& d = *iter;
        if (d.name == name) {
            enumValue = d.enumValue;
            validFlag = true;
            break;
        }
    }
    
    if (isValidOut != 0) {
        *isValidOut = validFlag;
    }
    else if (validFlag == false) {
        CaretAssertMessage(0, AString("Name " + name + " failed to match enumerated value for type ZarrDataTypeEnum"));
    }
    return enumValue;
}

/**
 * Get a GUI string representation of the enumerated type.
 * @param enumValue 
 *     Enumerated value.
 * @return 
 *     String representing enumerated value.
 */
AString 
ZarrDataTypeEnum::toGuiName(Enum enumValue) {
    if (initializedFlag == false) initialize();
    
    const ZarrDataTypeEnum* enumInstance = findData(enumValue);
    return enumInstance->guiName;
}

/**
 * Get an enumerated value corresponding to its GUI name.
 * @param s 
 *     Name of enumerated value.
 * @param isValidOut 
 *     If not NULL, it is set indicating that a
 *     enum value exists for the input name.
 * @return 
 *     Enumerated value.
 */
ZarrDataTypeEnum::Enum 
ZarrDataTypeEnum::fromGuiName(const AString& guiName, bool* isValidOut)
{
    if (initializedFlag == false) initialize();
    
    bool validFlag = false;
    Enum enumValue = ZarrDataTypeEnum::enumData[0].enumValue;
    
    for (std::vector<ZarrDataTypeEnum>::iterator iter = enumData.begin();
         iter != enumData.end();
         iter++) {
        const ZarrDataTypeEnum& d = *iter;
        if (d.guiName == guiName) {
            enumValue = d.enumValue;
            validFlag = true;
            break;
        }
    }
    
    if (isValidOut != 0) {
        *isValidOut = validFlag;
    }
    else if (validFlag == false) {
        CaretAssertMessage(0, AString("guiName " + guiName + " failed to match enumerated value for type ZarrDataTypeEnum"));
    }
    return enumValue;
}

/**
 * Get the integer code for a data type.
 *
 * @return
 *    Integer code for data type.
 */
int32_t
ZarrDataTypeEnum::toIntegerCode(Enum enumValue)
{
    if (initializedFlag == false) initialize();
    const ZarrDataTypeEnum* enumInstance = findData(enumValue);
    return enumInstance->integerCode;
}

/**
 * Find the data type corresponding to an integer code.
 *
 * @param integerCode
 *     Integer code for enum.
 * @param isValidOut
 *     If not NULL, on exit isValidOut will indicate if
 *     integer code is valid.
 * @return
 *     Enum for integer code.
 */
ZarrDataTypeEnum::Enum
ZarrDataTypeEnum::fromIntegerCode(const int32_t integerCode, bool* isValidOut)
{
    if (initializedFlag == false) initialize();
    
    bool validFlag = false;
    Enum enumValue = ZarrDataTypeEnum::enumData[0].enumValue;
    
    for (std::vector<ZarrDataTypeEnum>::iterator iter = enumData.begin();
         iter != enumData.end();
         iter++) {
        const ZarrDataTypeEnum& enumInstance = *iter;
        if (enumInstance.integerCode == integerCode) {
            enumValue = enumInstance.enumValue;
            validFlag = true;
            break;
        }
    }
    
    if (isValidOut != 0) {
        *isValidOut = validFlag;
    }
    else if (validFlag == false) {
        CaretAssertMessage(0, AString("Integer code " + AString::number(integerCode) + " failed to match enumerated value for type ZarrDataTypeEnum"));
    }
    return enumValue;
}

/**
 * Get all of the enumerated type values.  The values can be used
 * as parameters to toXXX() methods to get associated metadata.
 *
 * @param allEnums
 *     A vector that is OUTPUT containing all of the enumerated values.
 */
void
ZarrDataTypeEnum::getAllEnums(std::vector<ZarrDataTypeEnum::Enum>& allEnums)
{
    if (initializedFlag == false) initialize();
    
    allEnums.clear();
    
    for (std::vector<ZarrDataTypeEnum>::iterator iter = enumData.begin();
         iter != enumData.end();
         iter++) {
        allEnums.push_back(iter->enumValue);
    }
}

/**
 * Get all of the names of the enumerated type values.
 *
 * @param allNames
 *     A vector that is OUTPUT containing all of the names of the enumerated values.
 * @param isSorted
 *     If true, the names are sorted in alphabetical order.
 */
void
ZarrDataTypeEnum::getAllNames(std::vector<AString>& allNames, const bool isSorted)
{
    if (initializedFlag == false) initialize();
    
    allNames.clear();
    
    for (std::vector<ZarrDataTypeEnum>::iterator iter = enumData.begin();
         iter != enumData.end();
         iter++) {
        allNames.push_back(ZarrDataTypeEnum::toName(iter->enumValue));
    }
    
    if (isSorted) {
        std::sort(allNames.begin(), allNames.end());
    }
}

/**
 * Get all of the GUI names of the enumerated type values.
 *
 * @param allNames
 *     A vector that is OUTPUT containing all of the GUI names of the enumerated values.
 * @param isSorted
 *     If true, the names are sorted in alphabetical order.
 */
void
ZarrDataTypeEnum::getAllGuiNames(std::vector<AString>& allGuiNames, const bool isSorted)
{
    if (initializedFlag == false) initialize();
    
    allGuiNames.clear();
    
    for (std::vector<ZarrDataTypeEnum>::iterator iter = enumData.begin();
         iter != enumData.end();
         iter++) {
        allGuiNames.push_back(ZarrDataTypeEnum::toGuiName(iter->enumValue));
    }
    
    if (isSorted) {
        std::sort(allGuiNames.begin(), allGuiNames.end());
    }
}

