=begin
=module NumRu::Met::SkewT

Draw Skew-T log-p diagram by using RubyDCL

((<Sample image|URL:skewT-sample.jpg>))

==Example

   require "numru/met"
   include NumRu
   include NumRu::Met
   DCL.gropn(1)
   DCL.grfrm
   DCL.grsvpt(0.25,0.75,0.2,0.8)
   SkewT.frame([-10.0,40.0],[1000.0,250.0])
   prs = [1000,850,700,500,300]
   temp = [20,10,0,-20,-45]
   r = [0.012,0.008,0.004,0.0005,0.0001]
   SkewT.plot_tempC(temp,prs)
   SkewT.plot_mixing_ratio(r,prs)
   DCL.grcls

==Module functions
---frame(t_range=nil, p_range=nil, title=nil)
    Defines viewport, draws axes, and isopleths needed for a skew-T diagram
    (temperature, mixing ratio, potential temperature, saturation
    equivalent potential temperature (pseudo-adiabatic))

    ARGUMENTS
    * t_range [Array of two numerics, or nil] min & max temperature 
      at the bottom of the figure (Celcius)
    * p_range [Array of two numerics, or nil] max & min pressure
      of the figure (hPa)
    * title [String, or nil] title of the figure

---plot_tempC(tempC, prs, line_type=1, line_index=5)
    Plots temperarue on the skew-T diagram. You have to call ((<frame>)) in advance.

    ARGUMENTS
    * tempC [Array of Numeric or 1D NArray]: temperature [Celcius]
    * prs [Array of Numeric or 1D NArray]: pressure [hPa]

---plot_mixing_ratio(r, prs, line_type=3, line_index=5)
    Plots mixing ratio -- dew(frost) point temperature on the skew-T diagram.
    You have to call ((<frame>)) in advance.

    ARGUMENTS
    * r [Array of Numeric or 1D NArray]: mixing ratio [kg/kg]
    * prs [Array of Numeric or 1D NArray]: pressure [hPa]

---plot_specific_humidity(q, prs, line_type=3, line_index=5)
    Same as plot_mixing_ratio, but takes specific humidity as an argument
    (internally converted to mixing ratio)
    Plots mixing ratio -- dew(frost) point temperature on the skew-T diagram.
    You have to call ((<frame>)) in advance.

    ARGUMENTS
    * q [Array of Numeric or 1D NArray]: specific humidity [kg/kg]
    * prs [Array of Numeric or 1D NArray]: pressure [hPa]

=end

require 'numru/dcl'
require 'numru/misc'

module NumRu
  module Met

    if !defined?(Thermo)
      require './thermo'     # for test before installation
    end

    module SkewT

      extend Misc::EMath      #  in NumRu::Misc ('numru/misc')

      module_function

      def plot_tempC(tempC, prs, line_type=1, line_index=5)
	tempC = NArray.to_na(tempC).to_f if tempC.is_a?(Array)
	prs = NArray.to_na(prs).to_f if prs.is_a?(Array)
	tempC = tempC + prs2dux(prs)
	DCL.sgplzu(tempC, prs, line_type, line_index)
      end
      def plot_mixing_ratio(r, prs, line_type=3, line_index=5)
	prs = NArray.to_na(prs).to_f if prs.is_a?(Array)
	tempC = Thermo::t_dewpoint(r,prs) + prs2dux(prs)
	DCL.sgplzu(tempC, prs, line_type, line_index)
      end
      def plot_specific_humidity(q, prs, line_type=3, line_index=5)
	r = Thermo.q2r(q)
	plot_mixing_ratio(r, prs, line_type=3, line_index=5)
      end

      def frame(t_range=nil, p_range=nil, title=nil)
	if !t_range
	  t_range=[-30.0, 40.0]
	else
	  t_range=[t_range.min.to_f, t_range.max.to_f]
	end
        if !p_range
	  p_range=[1000.0, 100.0]
	else
	  p_range=[p_range.max.to_f, p_range.min.to_f]
	end
	p_ticks=NArray[1000,925,850,700,600,500,400,300,250,200,150,100,70]
	p_ticks = p_ticks[ (p_range[0]>=p_ticks) & (p_ticks>=p_range[1]) ]
	p p_ticks
	DCL.grswnd(t_range[0],t_range[1],p_range[0],p_range[1])
	DCL.grstrn(2)
	DCL.grstrf
        DCL.uziset('inner',-1)
	DCL.uxaxdv('t',5,10)
	DCL.uxaxdv('b',5,10)
	DCL.uyaxnm('l',p_ticks,p_ticks)
	DCL.uyaxnm('r',p_ticks,p_ticks)
        DCL.sgpset('lchar',true)
	DCL.uxsttl('t',title,0.0) if title

	r_levs=[0.1e-5, 0.2e-5,0.5e-5, 1.0e-5, 2.0e-5, 5.0e-5, 
	        1.0e-4, 2.0e-4, 5.0e-4,
	        1.0e-3, 2.0e-3, 3.0e-3, 5.0e-3, 
	        0.7e-2, 1.0e-2, 1.5e-2, 2.0e-2, 2.5e-2, 3.0e-2, 4.0e-2, 5.0e-2]

	th_es_levs = 250 + 10*NArray[0.0..10.0,11.5,13,15,17]
	
	theta_levs = 253 + 10*NArray[0.0..20.0]

	nz = 51   # no of points to descretize log(p)
        z = log(p_range[0]) + 
	    log(p_range[1]/p_range[0]) * NArray.sfloat(nz).indgen!/(nz-1)
	p_levs = exp(z)

        #< temperature isopleths >

        tmax = (t_range[1]/10).floor * 10.0
        tmin = t_range[0] - prs2dux(p_range[1])
        tempC = tmax
        DCL.sgspls(0.85*DCL.uzpget('rsizel1'))
        while (tempC>=tmin)
          DCL.sgpset('ffct',0.5)
          p1 = [p_range[1], dux2prs(t_range[1]-tempC)].max
          p0 = [p_range[0], dux2prs(t_range[0]-tempC)].min
          DCL.sgsplc(tempC.to_i.to_s)
          DCL.sgplzu([tempC+prs2dux(p0),tempC+prs2dux(p1)],[p0,p1],1,3)
          tempC -= 10
        end
      
        #< mixing ratio isopleths >

        r_levs.each{ |r|
	  if r > 0.02
	    DCL.sgpset('ffct',[0.8+(r-0.02)*10, 0.97].min)
	  else
	    DCL.sgpset('ffct',0.8)
	  end
	  r_g_kg = r*1000
	  r_g_kg = r_g_kg.to_i if r_g_kg == r_g_kg.to_i
	  DCL.sgsplc(r_g_kg.to_s)
	  t_dew = Thermo::t_dewpoint(r,p_levs)+prs2dux(p_levs)
	  mask = (t_dew > t_range[0]) & (t_dew < t_range[1])
	  if mask.count_true >= 2
	    DCL.sgplzu(t_dew[mask],p_levs[mask],3,1)
	  end
        }

        #< potential temperature isopleths >

        theta_levs.each{ |theta|
          DCL.sgpset('ffct',0.6)
	  tempC = theta*(p_levs/1000.0)**Thermo::Kappa - Thermo::T0 +
	         prs2dux(p_levs)
	  mask = (tempC > t_range[0]) & (tempC < t_range[1])
	  DCL.sgsplc(theta.to_i.to_s)
	  if mask.count_true >= 2
	    DCL.sgplzu(tempC[mask],p_levs[mask],1,1)
	  end
        }

        #< saturation equivalent potential temperature isopleths >

        th_es_levs.each{ |th_es|
          DCL.sgpset('ffct',0.4)
	  tempC = Thermo::theta_es_prs2tempC(th_es,p_levs)+prs2dux(p_levs)
	  mask = (tempC > t_range[0]) & (tempC < t_range[1])
	  DCL.sgsplc(th_es.to_i.to_s)
	  if mask.count_true >= 2
	    DCL.sgplzu(tempC[mask],p_levs[mask],2,3)
	  end
        }

        # < finish > 
        DCL.sgpset('lchar',false)
        nil
      end

      def dux2prs(dux)
         vxmin,vxmax,vymin,vymax = DCL.sgqvpt
         uxmin,uxmax,uymin,uymax = DCL.sgqwnd
         dvx = dux*(vxmax-vxmin)/(uxmax-uxmin)
         dvy = dvx                         # 45 degree in the V coordinate
         prs = uymin * exp(dvy*log(uymax/uymin)/(vymax-vymin))
      end

      def prs2dux(prs)
         vxmin,vxmax,vymin,vymax = DCL.sgqvpt
         uxmin,uxmax,uymin,uymax = DCL.sgqwnd
         dvy = log(prs/uymin)/log(uymax/uymin) * (vymax-vymin)
         dvx = dvy                         # 45 degree in the V coordinate
         dux = (uxmax-uxmin)*dvx/(vxmax-vxmin)
         dux
      end

    end
  end
end

if $0 == __FILE__
  include NumRu
  include NumRu::Met
  DCL.gropn(1)
  DCL.grfrm
  DCL.grsvpt(0.2,0.8,0.15,0.85)
  SkewT.frame
  DCL.grfrm
  DCL.grsvpt(0.25,0.75,0.2,0.8)
  SkewT.frame([-10.0,40.0],[1000.0,250.0])
  SkewT.plot_tempC([20,10,0,-20,-45],[1000,850,700,500,300])
  SkewT.plot_mixing_ratio([0.012,0.008,0.004,0.0005,0.0001],[1000,850,700,500,300])
  DCL.grcls
end
