// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.openapi.diff.impl.incrementalMerge.ui;

import com.intellij.idea.ActionsBundle;
import com.intellij.openapi.actionSystem.*;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.diff.DocumentContent;
import com.intellij.openapi.diff.SimpleDiffRequest;
import com.intellij.openapi.diff.impl.external.DiffManagerImpl;
import com.intellij.openapi.editor.Editor;
import com.intellij.openapi.fileTypes.FileType;
import com.intellij.openapi.project.DumbAware;
import com.intellij.openapi.project.Project;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

class OpenPartialDiffAction extends AnAction implements DumbAware {
  private static final Logger LOG = Logger.getInstance("#com.intellij.openapi.diff.impl.incrementalMerge.ui.OpenPartialDiffAction");
  private final int myLeftIndex;
  private final int myRightIndex;

  OpenPartialDiffAction(int leftIndex, int rightIndex, Icon icon) {
    super("", null, icon);
    myLeftIndex = leftIndex;
    myRightIndex = rightIndex;
  }

  @Override
  public void actionPerformed(@NotNull AnActionEvent e) {
    DataContext dataContext = e.getDataContext();
    MergePanel2 mergePanel = MergePanel2.fromDataContext(dataContext);
    Project project = projectFromDataContext(dataContext);
    Editor leftEditor = mergePanel.getEditor(myLeftIndex);
    Editor rightEditor = mergePanel.getEditor(myRightIndex);
    FileType type = mergePanel.getContentType();
    SimpleDiffRequest diffData = new SimpleDiffRequest(project, composeName());
    diffData.setContents(new DocumentContent(project, leftEditor.getDocument(), type), new DocumentContent(project, rightEditor.getDocument(), type));
    diffData.setContentTitles(mergePanel.getVersionTitle(myLeftIndex), mergePanel.getVersionTitle(myRightIndex));
    LOG.assertTrue(DiffManagerImpl.INTERNAL_DIFF.canShow(diffData));
    DiffManagerImpl.INTERNAL_DIFF.show(diffData);
  }

  @Override
  public void update(@NotNull AnActionEvent e) {
    DataContext dataContext = e.getDataContext();
    MergePanel2 mergePanel = MergePanel2.fromDataContext(dataContext);
    Project project = projectFromDataContext(dataContext);
    Presentation presentation = e.getPresentation();
    if (mergePanel == null || project == null) {
      presentation.setEnabled(false);
      presentation.setVisible(false);
      return;
    }
    presentation.setVisible(true);
    Editor leftEditor = mergePanel.getEditor(myLeftIndex);
    Editor rightEditor = mergePanel.getEditor(myRightIndex);
    if (leftEditor == null || rightEditor == null) {
      presentation.setEnabled(false);
      return;
    }
    presentation.setText(composeName());
    presentation.setEnabled(true);
  }

  private String composeName() {
    if (myLeftIndex == 0 && myRightIndex == 1) {
      return ActionsBundle.actionText("Diff.ComparePartial.Base.Left");
    }
    if (myLeftIndex == 1 && myRightIndex == 2) {
      return ActionsBundle.actionText("Diff.ComparePartial.Base.Right");
    }

    return ActionsBundle.actionText("Diff.ComparePartial.Left.Right");
  }

  @Nullable
  private static Project projectFromDataContext(DataContext dataContext) {
    return CommonDataKeys.PROJECT.getData(dataContext);
  }
}
