/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_utils.h
    \brief  API for the utilities for the LCAS
    \author Martijn Steenbakkers for the EU DataGrid.

    This header contains the declarations of the LCAS utility functions:
    -# lcas_get_dn():
    -# lcas_genfilename(): 
    -# lcas_getfexist():
    -# lcas_findfile():
    \ingroup APIforLcasPlugins
*/

#ifndef LCAS_UTILS_H
#define LCAS_UTILS_H

/******************************************************************************
                             Include header files
******************************************************************************/

#ifndef NOGLOBUS
    #include <gssapi.h>
#endif /* NOGLOBUS */
#include "lcas_types.h"

/******************************************************************************
 *                 Module definition
 *****************************************************************************/

/*! \name CREDENTIAL FUNCTIONS */
/*@{*/ /* Beginning of credential functions */

/******************************************************************************
Function:       lcas_get_gss_cred()
Description:    returns globus gss credential
Parameters:
                lcas_credential: lcas_credential
Returns:        globus gss credential
******************************************************************************/
/*!
    \fn lcas_get_gss_cred(
        lcas_cred_id_t lcas_credential
        )
    \brief Retrieve globus gss credential from (LCAS) credential

    This function takes an LCAS credential as input and returns the corresponding
    globus gss credential.
    
    \param lcas_credential the LCAS credential
    \return globus gss credential
*/
extern gss_cred_id_t lcas_get_gss_cred(
        lcas_cred_id_t lcas_credential
);

/******************************************************************************
Function:       lcas_get_dn()
Description:    returns user dn
Parameters:
                lcas_credential: lcas_credential
Returns:        user dn
******************************************************************************/
/*!
    \fn lcas_get_dn(
        lcas_cred_id_t lcas_credential
        )
    \brief Retrieve user DN from (LCAS) credential

    This function takes an LCAS credential as input and returns the corresponding
    user distinguished name (DN).
    
    (Internal:) If the GLOBUS credential part of the LCAS credential is empty
    the user DN is already included in the LCAS credential.

    \param lcas_credential the LCAS credential
    \return a string containing the user DN
*/
extern char * lcas_get_dn(
        lcas_cred_id_t lcas_credential
);


/* This define extends the internal struct, but it requires a rebuild of the elements to actually get something going here */
#ifdef LCAS_INTERFACE_EXTENDED

/******************************************************************************
Function:       lcas_get_cert_stack()
Description:    returns user's certificate stack
Parameters:
                lcas_credential: lcas_credential
Returns:        STACK_OF(X509) *
******************************************************************************/
/*!
    \fn lcas_get_cert_stack(
        lcas_cred_id_t lcas_credential
        )
    \brief Retrieve user certificate stack from (LCAS) credential

    This function takes an LCAS credential as input and returns the corresponding
    stack of X509 certificates.
    

    \param lcas_credential the LCAS credential
    \return Stack of certificates as STACK_OF (X509) *
*/
extern STACK_OF (X509) * lcas_get_cert_stack(
        lcas_cred_id_t lcas_credential
);

/******************************************************************************
Function:       lcas_get_user_cert()
Description:    returns user's certificate
Parameters:
                lcas_credential: lcas_credential
Returns:        X509 *
******************************************************************************/
/*!
    \fn lcas_get_user_cert()
        lcas_cred_id_t lcas_credential
        )
    \brief Retrieve user certificate from the (LCAS) credential

    This function takes an LCAS credential as input and returns the corresponding
    user certificate as X509 *.
    

    \param lcas_credential the LCAS credential
    \return User certificate as X509 *.
*/
extern X509 * lcas_get_user_cert (
        lcas_cred_id_t lcas_credential
);

/******************************************************************************
Function:       lcas_get_first_delegation()
Description:    returns user's first delegated certificate (first proxy)
Parameters:
                lcas_credential: lcas_credential
Returns:        X509 *
******************************************************************************/
/*!
    \fn lcas_get_first_delegation(
        lcas_cred_id_t lcas_credential
        )
    \brief Retrieve first delegated user cretificate from the (LCAS) credential

    This function takes an LCAS credential as input and returns the corresponding
    first delegated certificate as X509 *.
    

    \param lcas_credential the LCAS credential
    \return First delegated proxy certificate as X509 *.
*/
extern X509 * lcas_get_first_delegation (
        lcas_cred_id_t lcas_credential
);


/******************************************************************************
Function:       lcas_get_leaf_proxy()
Description:    returns user's final (leaf) proxy certificate
Parameters:
                lcas_credential: lcas_credential
Returns:        X509 *
******************************************************************************/
/*!
    \fn lcas_get_leaf_proxy(
        lcas_cred_id_t lcas_credential
        )
    \brief Retrieve leaf proxy (final delegated proxy) certificate from the (LCAS) credential

    This function takes an LCAS credential as input and returns the corresponding
    leaf proxy (final delegated proxy) certificate as X509 *.
    

    \param lcas_credential the LCAS credential
    \return Leaf proxy certificate as X509 *.
*/
extern X509 * lcas_get_leaf_proxy (
        lcas_cred_id_t lcas_credential
);

#endif /* LCAS_INTERFACE_EXTENDED */



/*@}*/ /* End of credential functions */

/*! \name FILENAME FUNCTIONS */
/*@{*/ /* Beginning of filename functions */
/******************************************************************************
Function:       lcas_genfilename() (copied from GLOBUS gatekeeper.c)
Description:    generate an absolute file name given a starting prefix,
                a relative or absolute path, and a suffix
                Only use prefix if path is relative.
Parameters:
Returns:        a pointer to a string which could be freeded.
******************************************************************************/
/*!
    \fn lcas_genfilename(
        char * prefixp,
        char * pathp,
        char * suffixp
        )
    \brief Generate an absolute file name.
    
    Given a starting prefix, a relative or absolute path, and a suffix an
    absolute file name is generated. Uses the prefix only if
    the path is relative. (Copied (and modified) from GLOBUS gatekeeper.c)
    
    \param prefixp string containing the prefix to be prepended.
    \param pathp   relative/absolute path to file name.
    \param suffixp string containing the suffix to be appended.

    \return pointer to a string containing the absolute path to the file, which has to
    be freed.
*/
extern char * lcas_genfilename(
        char * prefixp,
        char * pathp,
        char * suffixp
);

/******************************************************************************
Function:       lcas_getfexist()
Description:    picks the first existing file in argument list
Parameters:     n   : number of paths,
                ... : list of paths
Returns:        returns filename found or NULL
******************************************************************************/
/*!
    \fn lcas_getfexist(
        int n,
        ...
        )
    \brief Picks the first existing file in argument list.

    \param n   the number of paths presented in the following argument list.
    \param ... variable argument list of paths.

    \return filename found or NULL
*/
extern char * lcas_getfexist(
        int n,
        ...
);

/******************************************************************************
Function:       lcas_findplugin()
Description:    Checks for plugin in standard directories
Parameters:     name
Returns:        returns filename found (should be freeed) or NULL
******************************************************************************/
/*!
    \fn lcas_findplugin(
        char * name
        )
    \brief Checks for plugin in standard directories

    The directories that are checked are:
    - absolute location when starting with /
    - LCAS_ETC_HOME

    \param name string containing the file name
    \return pointer to a string containing the absolute path to the file, which has to
    be freed or NULL.
*/
extern char * lcas_finddbfile(
        char * name
);
extern char * lcas_findfile(
        char * name
);
/******************************************************************************
Function:       lcas_finddbfile() or lcas_findfile()
Description:    Checks for file in standard directories
Parameters:     name
Returns:        returns filename found (should be freeed) or NULL
******************************************************************************/
/*!
    \fn lcas_finddbfile(
        char * name
        )
    \fn lcas_findfile(
        char * name
        )
    \brief Checks for config-type files in standard directories

    The directories that are checked are
    - LCAS_MODULES_DIR env variable
    - ${libdir}/lcas (i.e. LCAS_MOD_HOME, when relative)

    \param name string containing the file name
    \return pointer to a string containing the absolute path to the file, which has to
    be freed or NULL.
*/
extern char * lcas_findplugin(
        char * name
);
/*@}*/ /* End of filename functions */

#endif /* LCAS_UTILS_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/interface/lcas_utils.h,v $
    $Date: 2010-05-03 10:43:18 $
    $Revision: 1.7 $
    $Author: okoeroo $
******************************************************************************/
