#include <stdio.h>
#include <stdlib.h>
#include <check.h>

#include "../src/signal_protocol.h"
#include "key_helper.h"
#include "test_common.h"

/*
 * Since the expected output from all these operations is random, these tests
 * use a special fake random number generator.
 * The expected output for these tests was generated by running the equivalent
 * functions in the Java version with a similarly fake random number generator.
 */

signal_context *global_context;
uint8_t test_next_random;

int fake_random_generator(uint8_t *data, size_t len, void *user_data)
{
    unsigned int i = 0;
    for(i = 0; i < len; i++) {
        data[i] = test_next_random++;
    }
    return 0;
}

void test_setup()
{
    int result;
    result = signal_context_create(&global_context, 0);
    ck_assert_int_eq(result, 0);
    signal_context_set_log_function(global_context, test_log);

    signal_crypto_provider provider = {
            .random_func = fake_random_generator,
            .hmac_sha256_init_func = test_hmac_sha256_init,
            .hmac_sha256_update_func = test_hmac_sha256_update,
            .hmac_sha256_final_func = test_hmac_sha256_final,
            .hmac_sha256_cleanup_func = test_hmac_sha256_cleanup,
            .user_data = 0
    };

    signal_context_set_crypto_provider(global_context, &provider);
    test_next_random = 0;
}

void test_teardown()
{
    signal_context_destroy(global_context);
}

START_TEST(test_generate_identity_key_pair)
{
    int result = 0;

    uint8_t identityKeyPair[] = {
            0x0a, 0x21, 0x05, 0x8f, 0x40, 0xc5, 0xad, 0xb6,
            0x8f, 0x25, 0x62, 0x4a, 0xe5, 0xb2, 0x14, 0xea,
            0x76, 0x7a, 0x6e, 0xc9, 0x4d, 0x82, 0x9d, 0x3d,
            0x7b, 0x5e, 0x1a, 0xd1, 0xba, 0x6f, 0x3e, 0x21,
            0x38, 0x28, 0x5f, 0x12, 0x20, 0x00, 0x01, 0x02,
            0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
            0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12,
            0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a,
            0x1b, 0x1c, 0x1d, 0x1e, 0x5f
    };

    ratchet_identity_key_pair *key_pair = 0;
    signal_buffer *buffer = 0;

    result = signal_protocol_key_helper_generate_identity_key_pair(&key_pair, global_context);
    ck_assert_int_eq(result, 0);

    result = ratchet_identity_key_pair_serialize(&buffer, key_pair);
    ck_assert_int_ge(result, 0);

    uint8_t *data = signal_buffer_data(buffer);
    size_t len = signal_buffer_len(buffer);

    ck_assert_int_eq(len, sizeof(identityKeyPair));
    ck_assert_int_eq(memcmp(identityKeyPair, data, len), 0);

    /* Cleanup */
    SIGNAL_UNREF(key_pair);
    signal_buffer_free(buffer);
}
END_TEST

START_TEST(test_generate_pre_keys)
{
    uint8_t preKey1[] = {
            0x08, 0x01, 0x12, 0x21, 0x05, 0x8f, 0x40, 0xc5,
            0xad, 0xb6, 0x8f, 0x25, 0x62, 0x4a, 0xe5, 0xb2,
            0x14, 0xea, 0x76, 0x7a, 0x6e, 0xc9, 0x4d, 0x82,
            0x9d, 0x3d, 0x7b, 0x5e, 0x1a, 0xd1, 0xba, 0x6f,
            0x3e, 0x21, 0x38, 0x28, 0x5f, 0x1a, 0x20, 0x00,
            0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
            0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
            0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
            0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x5f
    };
    uint8_t preKey2[] = {
            0x08, 0x02, 0x12, 0x21, 0x05, 0x35, 0x80, 0x72,
            0xd6, 0x36, 0x58, 0x80, 0xd1, 0xae, 0xea, 0x32,
            0x9a, 0xdf, 0x91, 0x21, 0x38, 0x38, 0x51, 0xed,
            0x21, 0xa2, 0x8e, 0x3b, 0x75, 0xe9, 0x65, 0xd0,
            0xd2, 0xcd, 0x16, 0x62, 0x54, 0x1a, 0x20, 0x20,
            0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
            0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
            0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
            0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x7f
    };
    uint8_t preKey3[] = {
            0x08, 0x03, 0x12, 0x21, 0x05, 0x79, 0xa6, 0x31,
            0xee, 0xde, 0x1b, 0xf9, 0xc9, 0x8f, 0x12, 0x03,
            0x2c, 0xde, 0xad, 0xd0, 0xe7, 0xa0, 0x79, 0x39,
            0x8f, 0xc7, 0x86, 0xb8, 0x8c, 0xc8, 0x46, 0xec,
            0x89, 0xaf, 0x85, 0xa5, 0x1a, 0x1a, 0x20, 0x40,
            0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
            0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
            0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
            0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f
    };
    uint8_t preKey4[] = {
            0x08, 0x04, 0x12, 0x21, 0x05, 0x67, 0x5d, 0xd5,
            0x74, 0xed, 0x77, 0x89, 0x31, 0x0b, 0x3d, 0x2e,
            0x76, 0x81, 0xf3, 0x79, 0x0b, 0x46, 0x6c, 0x77,
            0x3b, 0x15, 0x21, 0xfe, 0xcf, 0x36, 0x57, 0x79,
            0x58, 0x37, 0x1e, 0xa5, 0x2f, 0x1a, 0x20, 0x60,
            0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
            0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70,
            0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
            0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f
    };

    int result = 0;
    signal_protocol_key_helper_pre_key_list_node *head = 0;
    signal_protocol_key_helper_pre_key_list_node *cur_node = 0;
    session_pre_key *pre_key1 = 0;
    session_pre_key *pre_key2 = 0;
    session_pre_key *pre_key3 = 0;
    session_pre_key *pre_key4 = 0;
    signal_buffer *pre_key_buf1 = 0;
    signal_buffer *pre_key_buf2 = 0;
    signal_buffer *pre_key_buf3 = 0;
    signal_buffer *pre_key_buf4 = 0;

    /* Generate a list of 4 pre-keys */
    result = signal_protocol_key_helper_generate_pre_keys(&head,
            1, 4, global_context);
    ck_assert_int_eq(result, 0);

    ck_assert_ptr_ne(head, 0);
    cur_node = head;

    /* Explicitly iterate the list to get the 4 generated keys */
    pre_key1 = signal_protocol_key_helper_key_list_element(cur_node);
    cur_node = signal_protocol_key_helper_key_list_next(cur_node);
    ck_assert_ptr_ne(cur_node, 0);

    pre_key2 = signal_protocol_key_helper_key_list_element(cur_node);
    cur_node = signal_protocol_key_helper_key_list_next(cur_node);
    ck_assert_ptr_ne(cur_node, 0);

    pre_key3 = signal_protocol_key_helper_key_list_element(cur_node);
    cur_node = signal_protocol_key_helper_key_list_next(cur_node);
    ck_assert_ptr_ne(cur_node, 0);

    pre_key4 = signal_protocol_key_helper_key_list_element(cur_node);
    cur_node = signal_protocol_key_helper_key_list_next(cur_node);
    ck_assert_ptr_eq(cur_node, 0);

    /* Get the serialized data for the 4 generated keys */
    result = session_pre_key_serialize(&pre_key_buf1, pre_key1);
    ck_assert_int_ge(result, 0);
    result = session_pre_key_serialize(&pre_key_buf2, pre_key2);
    ck_assert_int_ge(result, 0);
    result = session_pre_key_serialize(&pre_key_buf3, pre_key3);
    ck_assert_int_ge(result, 0);
    result = session_pre_key_serialize(&pre_key_buf4, pre_key4);
    ck_assert_int_ge(result, 0);

    /* Compare to the expected values */
    ck_assert_int_eq(signal_buffer_len(pre_key_buf1), sizeof(preKey1));
    ck_assert_int_eq(memcmp(preKey1, signal_buffer_data(pre_key_buf1), sizeof(preKey1)), 0);
    ck_assert_int_eq(signal_buffer_len(pre_key_buf2), sizeof(preKey2));
    ck_assert_int_eq(memcmp(preKey2, signal_buffer_data(pre_key_buf2), sizeof(preKey2)), 0);
    ck_assert_int_eq(signal_buffer_len(pre_key_buf3), sizeof(preKey3));
    ck_assert_int_eq(memcmp(preKey3, signal_buffer_data(pre_key_buf3), sizeof(preKey3)), 0);
    ck_assert_int_eq(signal_buffer_len(pre_key_buf4), sizeof(preKey4));
    ck_assert_int_eq(memcmp(preKey4, signal_buffer_data(pre_key_buf4), sizeof(preKey4)), 0);

    /* Cleanup */
    signal_protocol_key_helper_key_list_free(head);
    signal_buffer_free(pre_key_buf1);
    signal_buffer_free(pre_key_buf2);
    signal_buffer_free(pre_key_buf3);
    signal_buffer_free(pre_key_buf4);
}
END_TEST

START_TEST(test_generate_signed_pre_key)
{
    int64_t timestamp = 1411152577000LL;

    uint8_t signedPreKey[] = {
            0x08, 0xd2, 0x09, 0x12, 0x21, 0x05, 0x35, 0x80,
            0x72, 0xd6, 0x36, 0x58, 0x80, 0xd1, 0xae, 0xea,
            0x32, 0x9a, 0xdf, 0x91, 0x21, 0x38, 0x38, 0x51,
            0xed, 0x21, 0xa2, 0x8e, 0x3b, 0x75, 0xe9, 0x65,
            0xd0, 0xd2, 0xcd, 0x16, 0x62, 0x54, 0x1a, 0x20,
            0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
            0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
            0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
            0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x7f,
            0x22, 0x40, 0xd8, 0x12, 0x88, 0xf2, 0x77, 0x38,
            0x08, 0x86, 0xac, 0xa4, 0x06, 0x2f, 0x06, 0xd8,
            0x30, 0xe6, 0xab, 0x73, 0x39, 0x4c, 0x85, 0xa0,
            0xc0, 0x5a, 0x81, 0x16, 0x3d, 0x21, 0x9c, 0x77,
            0xed, 0x41, 0xc1, 0x2d, 0x72, 0x61, 0x25, 0x4f,
            0xf4, 0x11, 0x64, 0xba, 0x6d, 0x89, 0x5c, 0x09,
            0x6c, 0x5e, 0x1f, 0xa6, 0xaa, 0x42, 0x53, 0x8d,
            0xb9, 0xe2, 0x6b, 0xbb, 0xb0, 0xb3, 0x6c, 0x99,
            0x74, 0x04, 0x29, 0xe8, 0x81, 0x3f, 0x8f, 0x48,
            0x01, 0x00, 0x00, 0x32, 0x40, 0xa9, 0x48, 0x21,
            0x7a, 0xf1, 0xa3, 0x60, 0xb9, 0x85, 0x44, 0x91,
            0x73, 0x2a, 0x29, 0x52, 0xb3, 0x3e, 0xb3, 0xe7,
            0x54, 0x8d, 0xd2, 0x06, 0xe1, 0x99, 0xec, 0xff,
            0xc2, 0x6b, 0x00, 0x95, 0x56, 0x30, 0xcd, 0x48,
            0xc0, 0xf0, 0x23, 0xef, 0x43, 0xa9, 0x1e, 0x97,
            0x80, 0x61, 0x63, 0x96, 0x60, 0x58, 0x6e, 0xa9,
            0x6f, 0x6d, 0x95, 0xb1, 0x12, 0x36, 0x47, 0x7d,
            0x16, 0xf5, 0x98, 0x79, 0x0f
    };

    int result = 0;
    ratchet_identity_key_pair *identity_key_pair = 0;
    session_signed_pre_key *signed_pre_key = 0;
    signal_buffer *buffer = 0;

    result = signal_protocol_key_helper_generate_identity_key_pair(&identity_key_pair, global_context);
    ck_assert_int_eq(result, 0);

    result = signal_protocol_key_helper_generate_signed_pre_key(&signed_pre_key,
            identity_key_pair, 1234, timestamp, global_context);
    ck_assert_int_eq(result, 0);

    result = session_signed_pre_key_serialize(&buffer, signed_pre_key);
    ck_assert_int_ge(result, 0);

    uint8_t *data = signal_buffer_data(buffer);
    size_t len = signal_buffer_len(buffer);

    ck_assert_int_eq(len, sizeof(signedPreKey));
    ck_assert_int_eq(memcmp(signedPreKey, data, len), 0);

    /* Cleanup */
    SIGNAL_UNREF(identity_key_pair);
    SIGNAL_UNREF(signed_pre_key);
    signal_buffer_free(buffer);
}
END_TEST

Suite *key_helper_suite(void)
{
    Suite *suite = suite_create("key_helper");

    TCase *tcase = tcase_create("case");
    tcase_add_checked_fixture(tcase, test_setup, test_teardown);
    tcase_add_test(tcase, test_generate_identity_key_pair);
    tcase_add_test(tcase, test_generate_pre_keys);
    tcase_add_test(tcase, test_generate_signed_pre_key);
    suite_add_tcase(suite, tcase);

    return suite;
}

int main(void)
{
    int number_failed;
    Suite *suite;
    SRunner *runner;

    suite = key_helper_suite();
    runner = srunner_create(suite);

    srunner_run_all(runner, CK_VERBOSE);
    number_failed = srunner_ntests_failed(runner);
    srunner_free(runner);
    return (number_failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
