/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Portable runtime library
 *
 * Copyright (C) 2007-2014 Stefano D'Angelo
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: Mutexes

   Mutexes with deadlock checking.
 */

#ifndef _NASPRO_CORE_MUTEX_H
#define _NASPRO_CORE_MUTEX_H

#ifndef _NASPRO_CORE_LIB_H
# error Only <NASPRO/core/lib.h> can be included directly.
#endif

NACORE_BEGIN_C_DECLS

/*
   Type: nacore_mutex

   Mutex.
 */
typedef struct _nacore_mutex *nacore_mutex;

/*
   Function: nacore_mutex_new()

   Creates a new mutex.

   Returns:

     The newly creted mutex or NULL if some error occurred, in which case errno
     is set to EAGAIN if the system lacked the necessary resources (other than
     memory), ENOMEM if there was not enough memory, EPERM if the caller does
     not have the priviledge to perform the operation or <NACORE_EUNKNOWN> if
     another kind of error happened.
 */
_NACORE_DEF nacore_mutex
nacore_mutex_new();

/*
   Function: nacore_mutex_free()

   Destroys a mutex.

   Once this function is called, referring to mutex will cause undefined
   behavior.

   Parameters:

     mutex - The mutex to be destroyed.
 */
_NACORE_DEF void
nacore_mutex_free(nacore_mutex mutex);

/*
   Function: nacore_mutex_lock()

   Acquires the lock on a mutex.

   If the mutex is locked by some other thread, the calling thread shall block
   until the mutex becomes available.

   Parameters:

     mutex - The mutex to acquire the lock on.

   Returns:

     0 on success or EDEADLK if the current thread already owns the mutex.
 */
_NACORE_DEF int
nacore_mutex_lock(nacore_mutex mutex);

/*
   Function: nacore_mutex_trylock()

   Attempts to acquire the lock on a mutex.

   If the mutex is already locked by any thread, including the current thread,
   the call shall return immediately.

   Parameters:

     mutex	 - The mutex to acquire the lock on.

   Returns:

     0 on success or EBUSY if the lock couldn't be acquired because it was
     already locked.
 */
_NACORE_DEF int
nacore_mutex_trylock(nacore_mutex mutex);

/*
   Function: nacore_mutex_unlock()

   Releases the lock on a mutex.

   Parameters:

     mutex - The mutex on which the lock is to be released.
 */
_NACORE_DEF void
nacore_mutex_unlock(nacore_mutex mutex);

NACORE_END_C_DECLS

#endif
