(*
 * BatPrintexc - Extended Printexc module
 * Copyright (C) 1996 Xavier Leroy
 *               2008 David Teller, LIFO, Universite d'Orleans
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version,
 * with the special exception on linking described in file LICENSE.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)


(** Facilities for printing exceptions.

    @author Xavier Leroy (Base module)
    @author David Teller
*)

val pass : ('a -> 'b) -> 'a -> 'b
(** [Printexc.pass fn x] applies [fn] to [x] and returns the result.
    If the evaluation of [fn x] raises any exception, the name of the
    exception is printed on standard error output, and the exception
    is raised again.  The typical use is to catch and report
    exceptions that escape a function application. This function is a
    renamed version of [Printexc.print] from stdlib.*)

val catch: ('a -> 'b) -> 'a -> 'b
(** [Printexc.catch fn x] is similar to {!Printexc.print}, but aborts
    the program with exit code 2 after printing the uncaught exception.
    This function is deprecated: the runtime system is now able to
    print uncaught exceptions as precisely as [Printexc.catch] does.
    Moreover, calling [Printexc.catch] makes it harder to track the
    location of the exception using the debugger or the stack backtrace
    facility.  So, do not use [Printexc.catch] in new code.  *)

val to_string: exn -> string
(** [Printexc.to_string e] returns a string representation of the
    exception [e]. *)

val print_backtrace: _ BatInnerIO.output -> unit
(** [print_backtrace oc] Prints the an exception backtrace on the
    output channel [oc].  The backtrace lists the program locations
    where the most-recently raised exception was raised and where it
    was propagated through function calls.

    @since 1.4.0
*)

val get_backtrace: unit -> string
(** [Printexc.get_backtrace ()] returns a string containing the same
    exception backtrace that [Printexc.print_backtrace] would print.
*)

val record_backtrace: bool -> unit
(** [Printexc.record_backtrace b] turns recording of exception
    backtraces on (if [b = true]) or off (if [b = false]).  Initially,
    backtraces are not recorded, unless the [b] flag is given to the
    program through the [OCAMLRUNPARAM] variable.
*)

val backtrace_status: unit -> bool
(** [Printexc.backtrace_status()] returns [true] if exception
    backtraces are currently recorded, [false] if not.
*)

val register_printer: (exn -> string option) -> unit
(** [Printexc.register_printer fn] registers [fn] as an exception
    printer.  The printer should return [None] or raise an exception
    if it does not know how to convert the passed exception, and [Some
    s] with [s] the resulting string if it can convert the passed
    exception. Exceptions raised by the printer are ignored.

    When converting an exception into a string, the printers will be invoked
    in the reverse order of their registrations, until a printer returns
    a [Some s] value (if no such printer exists, the runtime will use a
    generic printer).
*)

val print : _ BatInnerIO.output -> exn -> unit
  (** Print an exception.  The stdlib [print] function is now named [!pass].*)


##V=4.1##(** {1 Raw backtraces} *)
##V=4.1##
##V=4.1##type raw_backtrace = Printexc.raw_backtrace
##V=4.1##
##V=4.1##(** The abstract type [backtrace] stores exception backtraces in
##V=4.1##    a low-level format, instead of directly exposing them as string as
##V=4.1##    the [get_backtrace()] function does.
##V=4.1##
##V=4.1##    This allows to pay the performance overhead of representation
##V=4.1##    conversion and formatting only at printing time, which is useful
##V=4.1##    if you want to record more backtrace than you actually print.
##V=4.1##*)
##V=4.1##
##V=4.1##val get_raw_backtrace: unit -> raw_backtrace
##V=4.1##val print_raw_backtrace: out_channel -> raw_backtrace -> unit
##V=4.1##val raw_backtrace_to_string: raw_backtrace -> string

##V>=4.5##external raise_with_backtrace: exn -> Printexc.raw_backtrace -> 'a
##V>=4.5##  = "%raise_with_backtrace"
##V>=4.5##(** Reraise the exception using the given raw_backtrace for the
##V>=4.5##    origin of the exception
##V>=4.5##
##V>=4.5##    @since 2.7.0 and OCaml 4.05.0
##V>=4.5##*)

##V=4.1##(** {1 Current call stack} *)
##V=4.1##
##V=4.1##val get_callstack: int -> raw_backtrace
##V=4.1##
##V=4.1##(** [Printexc.get_callstack n] returns a description of the top of the
##V=4.1##    call stack on the current program point (for the current thread),
##V=4.1##    with at most [n] entries.  (Note: this function is not related to
##V=4.1##    exceptions at all, despite being part of the [Printexc] module.)
##V=4.1##
##V=4.1##    @since 2.2.0 and OCaml 4.01.0
##V=4.1##*)

##V>=4.2##(** {1 Raw backtraces} *)
##V>=4.2##
##V>=4.2##type raw_backtrace = Printexc.raw_backtrace
##V>=4.2##(** The abstract type [raw_backtrace] stores a backtrace in
##V>=4.2##    a low-level format, instead of directly exposing them as string as
##V>=4.2##    the [get_backtrace()] function does.
##V>=4.2##
##V>=4.2##    This allows delaying the formatting of backtraces to when they are
##V>=4.2##    actually printed, which may be useful if you record more
##V>=4.2##    backtraces than you print.
##V>=4.2##
##V>=4.2##    Raw backtraces cannot be marshalled. If you need marshalling, you
##V>=4.2##    should use the array returned by the [backtrace_slots] function of
##V>=4.2##    the next section.
##V>=4.2##
##V>=4.2##    @since 2.2.0 and OCaml 4.01.0
##V>=4.2##*)
##V>=4.2##
##V>=4.2##val get_raw_backtrace: unit -> raw_backtrace
##V>=4.2##(** [Printexc.get_raw_backtrace ()] returns the same exception
##V>=4.2##    backtrace that [Printexc.print_backtrace] would print, but in
##V>=4.2##    a raw format.
##V>=4.2##
##V>=4.2##    @since 2.2.0 and OCaml 4.01.0
##V>=4.2##*)
##V>=4.2##
##V>=4.2##val print_raw_backtrace: out_channel -> raw_backtrace -> unit
##V>=4.2##(** Print a raw backtrace in the same format
##V>=4.2##    [Printexc.print_backtrace] uses.
##V>=4.2##
##V>=4.2##    @since 2.2.0 and OCaml 4.01.0
##V>=4.2##*)
##V>=4.2##
##V>=4.2##val raw_backtrace_to_string: raw_backtrace -> string
##V>=4.2##(** Return a string from a raw backtrace, in the same format
##V>=4.2##    [Printexc.get_backtrace] uses.
##V>=4.2##
##V>=4.2##    @since 2.2.0 and OCaml 4.01.0
##V>=4.2##*)
##V>=4.2##
##V>=4.2##(** {1 Current call stack} *)
##V>=4.2##
##V>=4.2##val get_callstack: int -> raw_backtrace
##V>=4.10##external get_callstack : int -> raw_backtrace = "caml_get_current_callstack"
##V>=4.2##(** [Printexc.get_callstack n] returns a description of the top of the
##V>=4.2##    call stack on the current program point (for the current thread),
##V>=4.2##    with at most [n] entries.  (Note: this function is not related to
##V>=4.2##    exceptions at all, despite being part of the [Printexc] module.)
##V>=4.2##
##V>=4.2##    @since 2.2.0 and OCaml 4.01.0
##V>=4.2##*)
##V>=4.2##
##V>=4.2##(** {1 Uncaught exceptions} *)
##V>=4.2##
##V>=4.11##val default_uncaught_exception_handler: exn -> raw_backtrace -> unit
##V>=4.11##(** [Printexc.default_uncaught_exception_handler] prints the exception and
##V>=4.11##    backtrace on standard error output.
##V>=4.11##
##V>=4.11##    @since 4.11
##V>=4.11##*)
##V>=4.11##
##V>=4.2##val set_uncaught_exception_handler: (exn -> raw_backtrace -> unit) -> unit
##V>=4.2##(** [Printexc.set_uncaught_exception_handler fn] registers [fn] as the handler
##V>=4.2##    for uncaught exceptions. The default handler prints the exception and
##V>=4.2##    backtrace on standard error output.
##V>=4.2##
##V>=4.2##    Note that when [fn] is called all the functions registered with
##V>=4.2##    {!Pervasives.at_exit} have already been called. Because of this you must
##V>=4.2##    make sure any output channel [fn] writes on is flushed.
##V>=4.2##
##V>=4.2##    If [fn] raises an exception, it is ignored.
##V>=4.2##
##V>=4.2##    @since 2.3.0 and OCaml 4.02.0
##V>=4.2##*)
##V>=4.2##
##V>=4.2##
##V>=4.2##(** {1 Manipulation of backtrace information}
##V>=4.2##
##V>=4.2##    Those function allow to traverse the slots of a raw backtrace,
##V>=4.2##    extract information from them in a programmer-friendly format.
##V>=4.2##*)
##V>=4.2##
##V>=4.2##type backtrace_slot = Printexc.backtrace_slot
##V>=4.2##(** The abstract type [backtrace_slot] represents a single slot of
##V>=4.2##    a backtrace.
##V>=4.2##
##V>=4.2##    @since 2.3.0 and OCaml 4.02
##V>=4.2##*)
##V>=4.2##
##V>=4.2##val backtrace_slots : raw_backtrace -> backtrace_slot array option
##V>=4.2##(** Returns the slots of a raw backtrace, or [None] if none of them
##V>=4.2##    contain useful information.
##V>=4.2##
##V>=4.2##    In the return array, the slot at index [0] corresponds to the most
##V>=4.2##    recent function call, raise, or primitive [get_backtrace] call in
##V>=4.2##    the trace.
##V>=4.2##
##V>=4.2##    Some possible reasons for returning [None] are as follow:
##V>=4.2##    - none of the slots in the trace come from modules compiled with
##V>=4.2##    debug information ([-g])
##V>=4.2##    - the program is a bytecode program that has not been linked with
##V>=4.2##    debug information enabled ([ocamlc -g])
##V>=4.2##
##V>=4.2##   @since 2.3.0 and OCaml 4.02.0
##V>=4.2##*)
##V>=4.2##
##V>=4.2##type location = Printexc.location = {
##V>=4.2##  filename : string;
##V>=4.2##  line_number : int;
##V>=4.2##  start_char : int;
##V>=4.2##  end_char : int;
##V>=5.2##  end_line : int;
##V>=5.2##  end_col : int;
##V>=4.2##}
##V>=4.2##(** The type of location information found in backtraces. [start_char]
##V>=4.2##    and [end_char] are positions relative to the beginning of the
##V>=4.2##    line.
##V>=4.2##
##V>=4.2##    @since 2.3.0 and OCaml 4.02
##V>=4.2##*)
##V>=4.2##
##V>=4.2##module Slot : sig
##V>=4.2##  type t = backtrace_slot
##V>=4.2##
##V>=4.2##  val is_raise : t -> bool
##V>=4.2##  (** [is_raise slot] is [true] when [slot] refers to a raising
##V>=4.2##      point in the code, and [false] when it comes from a simple
##V>=4.2##      function call.
##V>=4.2##
##V>=4.2##      @since 2.3.0 and OCaml 4.02
##V>=4.2##  *)
##V>=4.2##
##V>=4.4##
##V>=4.4##  val is_inline : t -> bool
##V>=4.4##  (** [is_inline slot] is [true] when [slot] refers to a call
##V>=4.4##      that got inlined by the compiler, and [false] when it comes from
##V>=4.4##      any other context.
##V>=4.4##
##V>=4.4##      @since 4.04.0
##V>=4.4##  *)
##V>=4.4##
##V>=4.2##  val location : t -> location option
##V>=4.2##  (** [location slot] returns the location information of the slot,
##V>=4.2##      if available, and [None] otherwise.
##V>=4.2##
##V>=4.2##      Some possible reasons for failing to return a location are as follow:
##V>=4.2##      - the slot corresponds to a compiler-inserted raise
##V>=4.2##      - the slot corresponds to a part of the program that has not been
##V>=4.2##      compiled with debug information ([-g])
##V>=4.2##
##V>=4.2##      @since 2.3.0 and OCaml 4.02
##V>=4.2##  *)
##V>=4.2##
##V>=4.11##  val name : t -> string option
##V>=4.11##  (** [name slot] returns the name of the function or definition
##V>=4.11##      enclosing the location referred to by the slot.
##V>=4.11##
##V>=4.11##      [name slot] returns None if the name is unavailable, which
##V>=4.11##      may happen for the same reasons as [location] returning None.
##V>=4.11##
##V>=4.11##      @since 4.11
##V>=4.11##  *)
##V>=4.11##
##V>=4.2##  val format : int -> t -> string option
##V>=4.2##  (** [format pos slot] returns the string representation of [slot] as
##V>=4.2##      [raw_backtrace_to_string] would format it, assuming it is the
##V>=4.2##      [pos]-th element of the backtrace: the [0]-th element is
##V>=4.2##      pretty-printed differently than the others.
##V>=4.2##
##V>=4.2##      Whole-backtrace printing functions also skip some uninformative
##V>=4.2##      slots; in that case, [format pos slot] returns [None].
##V>=4.2##
##V>=4.2##      @since 2.3.0 and OCaml 4.02
##V>=4.2##  *)
##V>=4.2##end
##V>=4.2##
##V>=4.2##
##V>=4.2##(** {1 Raw backtrace slots} *)
##V>=4.2##
##V>=4.2##type raw_backtrace_slot = Printexc.raw_backtrace_slot
##V>=4.2##(** This type allows direct access to raw backtrace slots, without any
##V>=4.2##    conversion in an OCaml-usable data-structure. Being
##V>=4.2##    process-specific, they must absolutely not be marshalled, and are
##V>=4.2##    unsafe to use for this reason (marshalling them may not fail, but
##V>=4.2##    un-marshalling and using the result will result in
##V>=4.2##    undefined behavior).
##V>=4.2##
##V>=4.2##    Elements of this type can still be compared and hashed: when two
##V>=4.2##    elements are equal, then they represent the same source location
##V>=4.2##    (the converse is not necessarily true in presence of inlining,
##V>=4.2##    for example).
##V>=4.2##*)
##V>=4.2##
##V>=4.2##val raw_backtrace_length : raw_backtrace -> int
##V>=4.2##(** [raw_backtrace_length bckt] returns the number of slots in the
##V>=4.2##    backtrace [bckt].
##V>=4.2##
##V>=4.2##    @since 2.3.0 and OCaml 4.02
##V>=4.2##*)
##V>=4.2##
##V>=4.2##val get_raw_backtrace_slot : raw_backtrace -> int -> raw_backtrace_slot
##V>=4.2##(** [get_raw_backtrace_slot bckt pos] returns the slot in position [pos] in the
##V>=4.2##    backtrace [bckt].
##V>=4.2##
##V>=4.2##    @since 2.3.0 and OCaml 4.02
##V>=4.2##*)
##V>=4.2##
##V>=4.2##val convert_raw_backtrace_slot : raw_backtrace_slot -> backtrace_slot
##V>=4.2##(** Extracts the user-friendly [backtrace_slot] from a low-level
##V>=4.2##    [raw_backtrace_slot].
##V>=4.2##
##V>=4.2##    @since 2.3.0 and OCaml 4.02
##V>=4.2##*)

##V>=4.4##val get_raw_backtrace_next_slot :
##V>=4.4##    raw_backtrace_slot -> raw_backtrace_slot option
##V>=4.4##(** [get_raw_backtrace_next_slot slot] returns the next slot inlined, if any.
##V>=4.4##
##V>=4.4##    @since 2.11.0 and OCaml 4.04
##V>=4.4##*)


##V>=4.2##(** {1 Exception slots} *)
##V>=4.2##
##V>=4.2##val exn_slot_id: exn -> int
##V>=4.2##(** [Printexc.exn_slot_id] returns an integer which uniquely identifies
##V>=4.2##    the constructor used to create the exception value [exn]
##V>=4.2##    (in the current runtime).
##V>=4.2##
##V>=4.2##    @since 2.3.0 and OCaml 4.02.0
##V>=4.2##*)
##V>=4.2##
##V>=4.2##val exn_slot_name: exn -> string
##V>=4.2##(** [Printexc.exn_slot_name exn] returns the internal name of the constructor
##V>=4.2##    used to create the exception value [exn].
##V>=4.2##
##V>=4.2##    @since 2.3.0 and OCaml 4.02.0
##V>=4.2##*)

##V>=4.08##type t = exn = ..
##V>=4.08##(** The type of exception values. *)

##V>=4.09##val use_printers: exn -> string option
##V>=4.09##(** [Printexc.use_printers e] returns [None] if there are no registered
##V>=4.09##    printers and [Some s] with else as the resulting string otherwise.
##V>=4.09##
##V>=4.09##    @since 2.11.0 and OCaml 4.09
##V>=4.09##*)

##V>=4.09##val to_string_default: exn -> string
##V>=4.09##(** [Printexc.to_string_default e] returns a string representation of the
##V>=4.09##    exception [e], ignoring all registered exception printers.
##V>=4.09##
##V>=4.09##    @since 2.11.0 and OCaml 4.09
##V>=4.09##*)

##V>=4.12##type raw_backtrace_entry = Printexc.raw_backtrace_entry
##V>=4.12##(** A [raw_backtrace_entry] is an element of a [raw_backtrace].
##V>=4.12##
##V>=4.12##    Each [raw_backtrace_entry] is an opaque integer, whose value is not stable
##V>=4.12##    between different programs, or even between different runs of the same
##V>=4.12##    binary.
##V>=4.12##
##V>=4.12##    A [raw_backtrace_entry] can be converted to a usable form using
##V>=4.12##    [backtrace_slots_of_raw_entry] below. Note that, due to inlining, a
##V>=4.12##    single [raw_backtrace_entry] may convert to several [backtrace_slot]s.
##V>=4.12##    Since the values of a [raw_backtrace_entry] are not stable, they cannot
##V>=4.12##    be marshalled. If they are to be converted, the conversion must be done
##V>=4.12##    by the process that generated them.
##V>=4.12##
##V>=4.12##    Again due to inlining, there may be multiple distinct raw_backtrace_entry
##V>=4.12##    values that convert to equal [backtrace_slot]s. However, if two
##V>=4.12##    [raw_backtrace_entry]s are equal as integers, then they represent the same
##V>=4.12##    [backtrace_slot]s.
##V>=4.12##
##V>=4.12##    @since 3.3.0 and 4.12.0 *)
##V>=4.12##
##V>=4.12##val raw_backtrace_entries : raw_backtrace -> raw_backtrace_entry array
##V>=4.12##(** @since 3.3.0 and 4.12.0 *)
##V>=4.12##
##V>=4.12##val backtrace_slots_of_raw_entry :
##V>=4.12##  raw_backtrace_entry -> backtrace_slot array option
##V>=4.12##(** Returns the slots of a single raw backtrace entry, or [None] if this
##V>=4.12##    entry lacks debug information.
##V>=4.12##
##V>=4.12##    Slots are returned in the same order as [backtrace_slots]: the slot
##V>=4.12##    at index [0] is the most recent call, raise, or primitive, and
##V>=4.12##    subsequent slots represent callers.
##V>=4.12##
##V>=4.12##    @since 3.3.0 and 4.12
##V>=4.12##*)

(**/**)

(**  {1 Obj printer}
  Unsafe printer used in this module and exposed to the other standard library
   module
*)

##V>=5.0##val string_of_extension_constructor: Obj.t -> string
