/*
 * Copyright (c) 2015, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

// This test case relies on static security property, no way to re-use
// security property in samevm/agentvm mode.

/**
 * @test
 *
 * @bug 8141287
 * @summary Add MD5 to jdk.certpath.disabledAlgorithms security property
 *
 * @run main/othervm CPBuilderWithMD5 trustAnchor_SHA1withRSA_1024 0 true
 * @run main/othervm CPBuilderWithMD5 trustAnchor_SHA1withRSA_512  0 true
 * @run main/othervm CPBuilderWithMD5 intermediate_SHA1withRSA_1024_1024 1 true
 * @run main/othervm CPBuilderWithMD5 intermediate_SHA1withRSA_1024_512  1 false
 * @run main/othervm CPBuilderWithMD5 intermediate_SHA1withRSA_512_1024  1 false
 * @run main/othervm CPBuilderWithMD5 intermediate_SHA1withRSA_512_512  1 false
 * @run main/othervm CPBuilderWithMD5 intermediate_MD5withRSA_1024_1024  1 false
 * @run main/othervm CPBuilderWithMD5 intermediate_MD5withRSA_1024_512  1 false
 * @run main/othervm CPBuilderWithMD5 endentiry_SHA1withRSA_1024_1024  2 true
 * @run main/othervm CPBuilderWithMD5 endentiry_SHA1withRSA_1024_512  2 false
 * @run main/othervm CPBuilderWithMD5 endentiry_SHA1withRSA_512_1024  2 false
 * @run main/othervm CPBuilderWithMD5 endentiry_SHA1withRSA_512_512  2 false
 * @run main/othervm CPBuilderWithMD5 endentiry_MD5withRSA_1024_1024  2 false
 * @run main/othervm CPBuilderWithMD5 endentiry_MD5withRSA_1024_512  2 false
 *
 * @author Xuelei Fan
 */

/*
 * The generate.sh was designed to generate MD2 signed certificates.  The
 * certificates used in this test are generated by an updated generate.sh that
 * replacing MD2 with MD5 algorithm.
 */
import java.io.*;
import java.net.SocketException;
import java.util.*;
import java.security.Security;
import java.security.cert.*;
import sun.security.util.DerInputStream;

public class CPBuilderWithMD5 {

    // SHA1withRSA 1024
    static String trustAnchor_SHA1withRSA_1024 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICPjCCAaegAwIBAgIBADANBgkqhkiG9w0BAQUFADAfMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTAeFw0xMzEyMjgxMTA4NTFaFw0zNDEyMDgxMTA4NTFa\n" +
        "MB8xCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlMIGfMA0GCSqGSIb3DQEB\n" +
        "AQUAA4GNADCBiQKBgQDn3JSHACqi/bcohVy7eFqDs3L5ehnXmF9Jrg4rMRUeNrxA\n" +
        "61F8bJ9JXx4j8WyqmT0TtokgXuqGxbsXRQVVw4AdXLF2PwCs/y+Y+AwU59uDHA3J\n" +
        "AMk4VvjV9MB2Ea6YzuLnbbj/TNrfxB6LZ7KBvh0fYGt2T40yMvOvilU/f6e3zQID\n" +
        "AQABo4GJMIGGMB0GA1UdDgQWBBSIxINDFVm8GpUz3v+BbWNmDEKP7TBHBgNVHSME\n" +
        "QDA+gBSIxINDFVm8GpUz3v+BbWNmDEKP7aEjpCEwHzELMAkGA1UEBhMCVVMxEDAO\n" +
        "BgNVBAoTB0V4YW1wbGWCAQAwDwYDVR0TAQH/BAUwAwEB/zALBgNVHQ8EBAMCAgQw\n" +
        "DQYJKoZIhvcNAQEFBQADgYEAt0mYDXTpInrGvEOylIL2gx65A/bpdz9iDQsSs5sZ\n" +
        "r3m0v9zJnzR8lRqN4GbaD1vrFdkUrIoObcvXjXitnf5QqDzmc9BbIYj83Ft8QSUj\n" +
        "jCMy04EGT/7ATss4SiFEu6sJpmOBjsgH6wYuobR27wl/01XOu2CXUo3OOjgAoPBs\n" +
        "QoQ=\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 512
    static String trustAnchor_SHA1withRSA_512 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIIBuTCCAWOgAwIBAgIBADANBgkqhkiG9w0BAQUFADAfMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTAeFw0xMzEyMjgxMTA4NTFaFw0zNDEyMDgxMTA4NTFa\n" +
        "MB8xCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlMFwwDQYJKoZIhvcNAQEB\n" +
        "BQADSwAwSAJBANLqQkOpH7rBTo/a2ccYjJxvNib/Lxm6UXO1uAd/0AUzPWzJsOpB\n" +
        "u2zyD26UYc0GNyXCkWMZ44FrtSQ8VI146j8CAwEAAaOBiTCBhjAdBgNVHQ4EFgQU\n" +
        "5PVLxBY//smN31jHb/MAmCEz5NIwRwYDVR0jBEAwPoAU5PVLxBY//smN31jHb/MA\n" +
        "mCEz5NKhI6QhMB8xCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlggEAMA8G\n" +
        "A1UdEwEB/wQFMAMBAf8wCwYDVR0PBAQDAgIEMA0GCSqGSIb3DQEBBQUAA0EAzF9E\n" +
        "dXYPLqziCRY45IHCUtxaLjLQmwsjEu91TV4xyuuozGEumcqH7m6Hg6Ohnd1FGfsN\n" +
        "X+vt1tdaDIu9+OzGjQ==\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 1024 signed with RSA 1024
    static String intermediate_SHA1withRSA_1024_1024 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICUDCCAbmgAwIBAgIBAzANBgkqhkiG9w0BAQUFADAfMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTAeFw0xMzEyMjgxMTA5MDlaFw0zMzA5MTQxMTA5MDla\n" +
        "MDExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlMRAwDgYDVQQLEwdDbGFz\n" +
        "cy0xMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQC/m9wdJT0HR+exquh2Q2Yq\n" +
        "XvL9HtEsCabCikd0Vjuoi3sZJ/5SBbbHTvh7z7enW0NEpLHQee0ry5FW8mLxDtrR\n" +
        "38NjE9W7zutucBG5WztwGuvcts13aEw+vH+EwhokJW9PXz9Do+y4PTJo3vdsk7Zs\n" +
        "bGVY9+YjvlgEaozWXZ1JhQIDAQABo4GJMIGGMB0GA1UdDgQWBBQIsaDZL94kLug/\n" +
        "A1N4EkNOA4z47DBHBgNVHSMEQDA+gBSIxINDFVm8GpUz3v+BbWNmDEKP7aEjpCEw\n" +
        "HzELMAkGA1UEBhMCVVMxEDAOBgNVBAoTB0V4YW1wbGWCAQAwDwYDVR0TAQH/BAUw\n" +
        "AwEB/zALBgNVHQ8EBAMCAgQwDQYJKoZIhvcNAQEFBQADgYEA0673aIUF2k20jkpR\n" +
        "4USN9UkbfX57Uazhl4n765EeAiteWnTzKztISeH1GTrCw7bSl1r07aaflsnbKOHC\n" +
        "RrL2RxbxNwQARvuuCxr664vXnsGrt86xA5F2iNF22uDM/5HA5sIfBmEk5xXSLrgH\n" +
        "I7jOaYqAA1b8C+4DU2Z5ZgO4LOA=\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 1024 signed with RSA 512
    static String intermediate_SHA1withRSA_1024_512 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICDzCCAbmgAwIBAgIBBDANBgkqhkiG9w0BAQUFADAfMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTAeFw0xMzEyMjgxMTA5MDlaFw0zMzA5MTQxMTA5MDla\n" +
        "MDExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlMRAwDgYDVQQLEwdDbGFz\n" +
        "cy0xMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQC/m9wdJT0HR+exquh2Q2Yq\n" +
        "XvL9HtEsCabCikd0Vjuoi3sZJ/5SBbbHTvh7z7enW0NEpLHQee0ry5FW8mLxDtrR\n" +
        "38NjE9W7zutucBG5WztwGuvcts13aEw+vH+EwhokJW9PXz9Do+y4PTJo3vdsk7Zs\n" +
        "bGVY9+YjvlgEaozWXZ1JhQIDAQABo4GJMIGGMB0GA1UdDgQWBBQIsaDZL94kLug/\n" +
        "A1N4EkNOA4z47DBHBgNVHSMEQDA+gBTk9UvEFj/+yY3fWMdv8wCYITPk0qEjpCEw\n" +
        "HzELMAkGA1UEBhMCVVMxEDAOBgNVBAoTB0V4YW1wbGWCAQAwDwYDVR0TAQH/BAUw\n" +
        "AwEB/zALBgNVHQ8EBAMCAgQwDQYJKoZIhvcNAQEFBQADQQAihshnF7RWZ13tUGsH\n" +
        "iM4i8HmBjw2+pwW/cs0E8BcycYEy3beWMcL1Np2yfOa/7K5ZvGPhe/piwzTel+Kt\n" +
        "5VLm\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 512 signed with RSA 1024
    static String intermediate_SHA1withRSA_512_1024 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICDDCCAXWgAwIBAgIBBTANBgkqhkiG9w0BAQUFADAfMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTAeFw0xMzEyMjgxMTA5MDlaFw0zMzA5MTQxMTA5MDla\n" +
        "MDExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlMRAwDgYDVQQLEwdDbGFz\n" +
        "cy0xMFwwDQYJKoZIhvcNAQEBBQADSwAwSAJBAK4h/iW3wt+ugR5ObWiFSl394UU/\n" +
        "lWNm+N6UAgsBRhSzZz/Iof7xZTQI+usNXzOrTnU3+uZsMgokpjkrko1osxUCAwEA\n" +
        "AaOBiTCBhjAdBgNVHQ4EFgQU88OD48Osuh7lJiLnhfMhrySqW8QwRwYDVR0jBEAw\n" +
        "PoAUiMSDQxVZvBqVM97/gW1jZgxCj+2hI6QhMB8xCzAJBgNVBAYTAlVTMRAwDgYD\n" +
        "VQQKEwdFeGFtcGxlggEAMA8GA1UdEwEB/wQFMAMBAf8wCwYDVR0PBAQDAgIEMA0G\n" +
        "CSqGSIb3DQEBBQUAA4GBAAHN8XUTT6asa1MvpfqAvKTH6tNrMOmzoFsUamPxSrUB\n" +
        "tnBv/fa/E9+1QvQwl3g6luVXBkQf2/nVD0195IdkEuD/C6psuGKerXmiaRMv5Wcs\n" +
        "B+8bTzhNxMzHKPZDJ8Tf/RD3XpPvtxw0T+I5xud68FH/WDhJtu7TiXPAhs7srtHt\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 512 signed with RSA 512
    static String intermediate_SHA1withRSA_512_512 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIIByzCCAXWgAwIBAgIBBjANBgkqhkiG9w0BAQUFADAfMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTAeFw0xMzEyMjgxMTA5MDlaFw0zMzA5MTQxMTA5MDla\n" +
        "MDExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlMRAwDgYDVQQLEwdDbGFz\n" +
        "cy0xMFwwDQYJKoZIhvcNAQEBBQADSwAwSAJBAK4h/iW3wt+ugR5ObWiFSl394UU/\n" +
        "lWNm+N6UAgsBRhSzZz/Iof7xZTQI+usNXzOrTnU3+uZsMgokpjkrko1osxUCAwEA\n" +
        "AaOBiTCBhjAdBgNVHQ4EFgQU88OD48Osuh7lJiLnhfMhrySqW8QwRwYDVR0jBEAw\n" +
        "PoAU5PVLxBY//smN31jHb/MAmCEz5NKhI6QhMB8xCzAJBgNVBAYTAlVTMRAwDgYD\n" +
        "VQQKEwdFeGFtcGxlggEAMA8GA1UdEwEB/wQFMAMBAf8wCwYDVR0PBAQDAgIEMA0G\n" +
        "CSqGSIb3DQEBBQUAA0EASLN+1/pfo+9ty5EaYkoPu4QeYGr+5wmXyDceiaED/Lok\n" +
        "RdV0ZH0qwD4kiarlJssNOgMCk+2EzgvXcIhEMDa5hA==\n" +
        "-----END CERTIFICATE-----";

    // MD5withRSA 1024 signed with RSA 1024
    static String intermediate_MD5withRSA_1024_1024 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICUDCCAbmgAwIBAgIBBzANBgkqhkiG9w0BAQQFADAfMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTAeFw0xMzEyMjgxMTA5MDlaFw0zMzA5MTQxMTA5MDla\n" +
        "MDExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlMRAwDgYDVQQLEwdDbGFz\n" +
        "cy0xMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQC/m9wdJT0HR+exquh2Q2Yq\n" +
        "XvL9HtEsCabCikd0Vjuoi3sZJ/5SBbbHTvh7z7enW0NEpLHQee0ry5FW8mLxDtrR\n" +
        "38NjE9W7zutucBG5WztwGuvcts13aEw+vH+EwhokJW9PXz9Do+y4PTJo3vdsk7Zs\n" +
        "bGVY9+YjvlgEaozWXZ1JhQIDAQABo4GJMIGGMB0GA1UdDgQWBBQIsaDZL94kLug/\n" +
        "A1N4EkNOA4z47DBHBgNVHSMEQDA+gBSIxINDFVm8GpUz3v+BbWNmDEKP7aEjpCEw\n" +
        "HzELMAkGA1UEBhMCVVMxEDAOBgNVBAoTB0V4YW1wbGWCAQAwDwYDVR0TAQH/BAUw\n" +
        "AwEB/zALBgNVHQ8EBAMCAgQwDQYJKoZIhvcNAQEEBQADgYEAerx2je3FBVn2eoPs\n" +
        "nTzLKILezqCTCO7mXWiyBidRhh4RGdM8JggMN5SRmuwRurxfYFgPfqmAenWtEFqO\n" +
        "xZrTXQUvIrrEgpzqkfppFnkCh4kDsX4roD5Nho3J4MTBQkqE0r676Yq6Rp6cywCq\n" +
        "CHQQztRGY7n/ZYRNJ3uzvuoT1tk=\n" +
        "-----END CERTIFICATE-----";

    // MD5withRSA 1024 signed with RSA 512
    static String intermediate_MD5withRSA_1024_512 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICDzCCAbmgAwIBAgIBCDANBgkqhkiG9w0BAQQFADAfMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTAeFw0xMzEyMjgxMTA5MTBaFw0zMzA5MTQxMTA5MTBa\n" +
        "MDExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFtcGxlMRAwDgYDVQQLEwdDbGFz\n" +
        "cy0xMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQC/m9wdJT0HR+exquh2Q2Yq\n" +
        "XvL9HtEsCabCikd0Vjuoi3sZJ/5SBbbHTvh7z7enW0NEpLHQee0ry5FW8mLxDtrR\n" +
        "38NjE9W7zutucBG5WztwGuvcts13aEw+vH+EwhokJW9PXz9Do+y4PTJo3vdsk7Zs\n" +
        "bGVY9+YjvlgEaozWXZ1JhQIDAQABo4GJMIGGMB0GA1UdDgQWBBQIsaDZL94kLug/\n" +
        "A1N4EkNOA4z47DBHBgNVHSMEQDA+gBTk9UvEFj/+yY3fWMdv8wCYITPk0qEjpCEw\n" +
        "HzELMAkGA1UEBhMCVVMxEDAOBgNVBAoTB0V4YW1wbGWCAQAwDwYDVR0TAQH/BAUw\n" +
        "AwEB/zALBgNVHQ8EBAMCAgQwDQYJKoZIhvcNAQEEBQADQQA3XGQPNin8cDIsJ4vx\n" +
        "tTxUO6XVJoWOdTsjwzlMrPmLvjJNZeXLtQe3pQu0vjgyUpQ59VYLW3qKN/LF3UH0\n" +
        "Ep7V\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 1024 signed with RSA 1024
    static String endentiry_SHA1withRSA_1024_1024 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICNzCCAaCgAwIBAgIBAjANBgkqhkiG9w0BAQUFADAxMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTEQMA4GA1UECxMHQ2xhc3MtMTAeFw0xMzEyMjgxMTA5\n" +
        "MTJaFw0zMzA5MTQxMTA5MTJaMEExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFt\n" +
        "cGxlMRAwDgYDVQQLEwdDbGFzcy0xMQ4wDAYDVQQDEwVBbGljZTCBnzANBgkqhkiG\n" +
        "9w0BAQEFAAOBjQAwgYkCgYEAvktJqK4/SvQrTyGgV8tM6zP/K5xQP1pFRipRKS8i\n" +
        "2yaXdlW4jQBZWVXdfEsm8YwGwtXFKIlleALmgJcLldPwNm0qaKixL4mRJVMm4bXM\n" +
        "UXCfmr+Im1SpA4Yum4VFCfIJ1kkeQkXqc57sCSfS+rFnC+1kSNa9wj+Mc4+5FR4k\n" +
        "zqUCAwEAAaNPME0wCwYDVR0PBAQDAgPoMB0GA1UdDgQWBBRDzLh/sWyTsdq1KKnG\n" +
        "8e7JW1tPUDAfBgNVHSMEGDAWgBQIsaDZL94kLug/A1N4EkNOA4z47DANBgkqhkiG\n" +
        "9w0BAQUFAAOBgQB2RkWHOaL4WTOGoeTS4J4o9FW+4UXihbocdI/64rMExERjDkE/\n" +
        "Jh31TEmatnP1gPrF1AfmqQPubqVSbRtCHrZF+Ilk6L6YeyRNzKvsLiMUtgrLYLas\n" +
        "Vop0DFZxR02xHgaJdoJkcWBjNadb9zG7eZtt8OOOJ4lRwg02aLTy+WDqPA==\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 1024 signed with RSA 512
    static String endentiry_SHA1withRSA_1024_512 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIIB9jCCAaCgAwIBAgIBAzANBgkqhkiG9w0BAQUFADAxMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTEQMA4GA1UECxMHQ2xhc3MtMTAeFw0xMzEyMjgxMTA5\n" +
        "MTJaFw0zMzA5MTQxMTA5MTJaMEExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFt\n" +
        "cGxlMRAwDgYDVQQLEwdDbGFzcy0xMQ4wDAYDVQQDEwVBbGljZTCBnzANBgkqhkiG\n" +
        "9w0BAQEFAAOBjQAwgYkCgYEAvktJqK4/SvQrTyGgV8tM6zP/K5xQP1pFRipRKS8i\n" +
        "2yaXdlW4jQBZWVXdfEsm8YwGwtXFKIlleALmgJcLldPwNm0qaKixL4mRJVMm4bXM\n" +
        "UXCfmr+Im1SpA4Yum4VFCfIJ1kkeQkXqc57sCSfS+rFnC+1kSNa9wj+Mc4+5FR4k\n" +
        "zqUCAwEAAaNPME0wCwYDVR0PBAQDAgPoMB0GA1UdDgQWBBRDzLh/sWyTsdq1KKnG\n" +
        "8e7JW1tPUDAfBgNVHSMEGDAWgBTzw4Pjw6y6HuUmIueF8yGvJKpbxDANBgkqhkiG\n" +
        "9w0BAQUFAANBAIapvjECUm4YD4O99G0v2SM17cKQzjZtSWkScS7FSk4sxS+dP3hM\n" +
        "Qb2UpoRl6CGynhOVVy2G/VJN8BEqOfywj8k=\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 512 signed with RSA 1024
    static String endentiry_SHA1withRSA_512_1024 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIIB8zCCAVygAwIBAgIBBDANBgkqhkiG9w0BAQUFADAxMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTEQMA4GA1UECxMHQ2xhc3MtMTAeFw0xMzEyMjgxMTA5\n" +
        "MTJaFw0zMzA5MTQxMTA5MTJaMEExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFt\n" +
        "cGxlMRAwDgYDVQQLEwdDbGFzcy0xMQ4wDAYDVQQDEwVBbGljZTBcMA0GCSqGSIb3\n" +
        "DQEBAQUAA0sAMEgCQQCngiNTE+qngHcfj2jUpdc82gCw+TFRjR7oMSdp7b/3NwpD\n" +
        "E+11z9WspoXTDzvbKcGUH9svFl691NyY0ZUmf+4RAgMBAAGjTzBNMAsGA1UdDwQE\n" +
        "AwID6DAdBgNVHQ4EFgQUK+oVsFTQbz08evgQZ5Sd82c2y4UwHwYDVR0jBBgwFoAU\n" +
        "CLGg2S/eJC7oPwNTeBJDTgOM+OwwDQYJKoZIhvcNAQEFBQADgYEAMd/8XnjRz5jK\n" +
        "nbss9DDQQC2mUuCbV/tGdke7eQ1DtBVZLBU6wDgisGr52sUXmyZIPmSVKpQqwCG5\n" +
        "8cY5uQhaNwPtPmMMKXzX32zN9NhVkiDNceL+zHs3vdjD1i/QiUTST+NKfLYVb6dF\n" +
        "YMG65lxe3gMVxMweiHSZSukmk1k3gUA=\n" +
        "-----END CERTIFICATE-----";

    // SHA1withRSA 512 signed with RSA 512
    static String endentiry_SHA1withRSA_512_512 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIIBsjCCAVygAwIBAgIBBTANBgkqhkiG9w0BAQUFADAxMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTEQMA4GA1UECxMHQ2xhc3MtMTAeFw0xMzEyMjgxMTA5\n" +
        "MTJaFw0zMzA5MTQxMTA5MTJaMEExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFt\n" +
        "cGxlMRAwDgYDVQQLEwdDbGFzcy0xMQ4wDAYDVQQDEwVBbGljZTBcMA0GCSqGSIb3\n" +
        "DQEBAQUAA0sAMEgCQQCngiNTE+qngHcfj2jUpdc82gCw+TFRjR7oMSdp7b/3NwpD\n" +
        "E+11z9WspoXTDzvbKcGUH9svFl691NyY0ZUmf+4RAgMBAAGjTzBNMAsGA1UdDwQE\n" +
        "AwID6DAdBgNVHQ4EFgQUK+oVsFTQbz08evgQZ5Sd82c2y4UwHwYDVR0jBBgwFoAU\n" +
        "88OD48Osuh7lJiLnhfMhrySqW8QwDQYJKoZIhvcNAQEFBQADQQB4xFWtC6ijDBIe\n" +
        "/Gkf3B9+ycmP52pTPNiPwMS6u1a5vTRXMn5xRDexWfxJKJVZ2s9UR1jheZvWgPC8\n" +
        "VUWO8bbG\n" +
        "-----END CERTIFICATE-----";

    // MD5withRSA 1024 signed with RSA 1024
    static String endentiry_MD5withRSA_1024_1024 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIICNzCCAaCgAwIBAgIBBjANBgkqhkiG9w0BAQQFADAxMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTEQMA4GA1UECxMHQ2xhc3MtMTAeFw0xMzEyMjgxMTA5\n" +
        "MTJaFw0zMzA5MTQxMTA5MTJaMEExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFt\n" +
        "cGxlMRAwDgYDVQQLEwdDbGFzcy0xMQ4wDAYDVQQDEwVBbGljZTCBnzANBgkqhkiG\n" +
        "9w0BAQEFAAOBjQAwgYkCgYEAvktJqK4/SvQrTyGgV8tM6zP/K5xQP1pFRipRKS8i\n" +
        "2yaXdlW4jQBZWVXdfEsm8YwGwtXFKIlleALmgJcLldPwNm0qaKixL4mRJVMm4bXM\n" +
        "UXCfmr+Im1SpA4Yum4VFCfIJ1kkeQkXqc57sCSfS+rFnC+1kSNa9wj+Mc4+5FR4k\n" +
        "zqUCAwEAAaNPME0wCwYDVR0PBAQDAgPoMB0GA1UdDgQWBBRDzLh/sWyTsdq1KKnG\n" +
        "8e7JW1tPUDAfBgNVHSMEGDAWgBQIsaDZL94kLug/A1N4EkNOA4z47DANBgkqhkiG\n" +
        "9w0BAQQFAAOBgQBCVn9/JQxsRTaKIKSrgB+KtEreat+33k3SXuJICuRxcmvjOXIx\n" +
        "wSdq+mRCA+DpIPSNtgnDAUyipnyxRxpdmRRUHuRYpkALq4a5QtTJK0Y/CEMfsd2J\n" +
        "Yd2zKcfynDLW6LVeNdtjlY7fTemJnbA/WImNhwyW55V9vbnk3J04EZN8jw==\n" +
        "-----END CERTIFICATE-----";

    // MD5withRSA 1024 signed with RSA 512
    static String endentiry_MD5withRSA_1024_512 =
        "-----BEGIN CERTIFICATE-----\n" +
        "MIIB9jCCAaCgAwIBAgIBBzANBgkqhkiG9w0BAQQFADAxMQswCQYDVQQGEwJVUzEQ\n" +
        "MA4GA1UEChMHRXhhbXBsZTEQMA4GA1UECxMHQ2xhc3MtMTAeFw0xMzEyMjgxMTA5\n" +
        "MTNaFw0zMzA5MTQxMTA5MTNaMEExCzAJBgNVBAYTAlVTMRAwDgYDVQQKEwdFeGFt\n" +
        "cGxlMRAwDgYDVQQLEwdDbGFzcy0xMQ4wDAYDVQQDEwVBbGljZTCBnzANBgkqhkiG\n" +
        "9w0BAQEFAAOBjQAwgYkCgYEAvktJqK4/SvQrTyGgV8tM6zP/K5xQP1pFRipRKS8i\n" +
        "2yaXdlW4jQBZWVXdfEsm8YwGwtXFKIlleALmgJcLldPwNm0qaKixL4mRJVMm4bXM\n" +
        "UXCfmr+Im1SpA4Yum4VFCfIJ1kkeQkXqc57sCSfS+rFnC+1kSNa9wj+Mc4+5FR4k\n" +
        "zqUCAwEAAaNPME0wCwYDVR0PBAQDAgPoMB0GA1UdDgQWBBRDzLh/sWyTsdq1KKnG\n" +
        "8e7JW1tPUDAfBgNVHSMEGDAWgBTzw4Pjw6y6HuUmIueF8yGvJKpbxDANBgkqhkiG\n" +
        "9w0BAQQFAANBAAbZwmkqb6sfiiIxuLnj6PjhJsXGfvPomkkbLu5CapAMhen/p6ZG\n" +
        "6vh69TbIsBR9UHu7qDyTl5Xax7bmYeW+sDQ=\n" +
        "-----END CERTIFICATE-----";

    static HashMap<String, String> certmap = new HashMap<String, String>();
    static {
        certmap.put("trustAnchor_SHA1withRSA_1024",
                                    trustAnchor_SHA1withRSA_1024);
        certmap.put("trustAnchor_SHA1withRSA_512",
                                    trustAnchor_SHA1withRSA_512);
        certmap.put("intermediate_SHA1withRSA_1024_1024",
                                    intermediate_SHA1withRSA_1024_1024);
        certmap.put("intermediate_SHA1withRSA_1024_512",
                                    intermediate_SHA1withRSA_1024_512);
        certmap.put("intermediate_SHA1withRSA_512_1024",
                                    intermediate_SHA1withRSA_512_1024);
        certmap.put("intermediate_SHA1withRSA_512_512",
                                    intermediate_SHA1withRSA_512_512);
        certmap.put("intermediate_MD5withRSA_1024_1024",
                                    intermediate_MD5withRSA_1024_1024);
        certmap.put("intermediate_MD5withRSA_1024_512",
                                    intermediate_MD5withRSA_1024_512);
        certmap.put("endentiry_SHA1withRSA_1024_1024",
                                    endentiry_SHA1withRSA_1024_1024);
        certmap.put("endentiry_SHA1withRSA_1024_512",
                                    endentiry_SHA1withRSA_1024_512);
        certmap.put("endentiry_SHA1withRSA_512_1024",
                                    endentiry_SHA1withRSA_512_1024);
        certmap.put("endentiry_SHA1withRSA_512_512",
                                    endentiry_SHA1withRSA_512_512);
        certmap.put("endentiry_MD5withRSA_1024_1024",
                                    endentiry_MD5withRSA_1024_1024);
        certmap.put("endentiry_MD5withRSA_1024_512",
                                    endentiry_MD5withRSA_1024_512);
    }

    private static Set<TrustAnchor> generateTrustAnchors()
            throws CertificateException {
        // generate certificate from cert string
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        HashSet<TrustAnchor> anchors = new HashSet<TrustAnchor>();

        ByteArrayInputStream is =
            new ByteArrayInputStream(trustAnchor_SHA1withRSA_1024.getBytes());
        Certificate cert = cf.generateCertificate(is);
        TrustAnchor anchor = new TrustAnchor((X509Certificate)cert, null);
        anchors.add(anchor);

        is = new ByteArrayInputStream(trustAnchor_SHA1withRSA_512.getBytes());
        cert = cf.generateCertificate(is);
        anchor = new TrustAnchor((X509Certificate)cert, null);
        anchors.add(anchor);

        return anchors;
    }

    private static CertStore generateCertificateStore() throws Exception {
        Collection entries = new HashSet();

        // generate certificate from certificate string
        CertificateFactory cf = CertificateFactory.getInstance("X.509");

        for (String key : certmap.keySet()) {
            String certStr = certmap.get(key);
            ByteArrayInputStream is =
                        new ByteArrayInputStream(certStr.getBytes());;
            Certificate cert = cf.generateCertificate(is);
            entries.add(cert);
        }

        return CertStore.getInstance("Collection",
                            new CollectionCertStoreParameters(entries));
    }

    private static X509CertSelector generateSelector(String name)
                throws Exception {
        X509CertSelector selector = new X509CertSelector();

        String certStr = certmap.get(name);
        if (certStr == null) {
            return null;
        }

        // generate certificate from certificate string
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        ByteArrayInputStream is = new ByteArrayInputStream(certStr.getBytes());
        X509Certificate target = (X509Certificate)cf.generateCertificate(is);

        selector.setCertificate(target);

        return selector;
    }

    private static boolean match(String name, Certificate cert)
                throws Exception {
        X509CertSelector selector = new X509CertSelector();

        String certStr = certmap.get(name);
        if (certStr == null) {
            return false;
        }

        // generate certificate from certificate string
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        ByteArrayInputStream is = new ByteArrayInputStream(certStr.getBytes());
        X509Certificate target = (X509Certificate)cf.generateCertificate(is);

        return target.equals(cert);
    }

    public static void main(String args[]) throws Exception {
        CertPathBuilder builder = CertPathBuilder.getInstance("PKIX");

        X509CertSelector selector = generateSelector(args[0]);
        if (selector == null) {
            // no target certificate, ignore it
            return;
        }

        Set<TrustAnchor> anchors = generateTrustAnchors();
        CertStore certs = generateCertificateStore();

        PKIXBuilderParameters params =
                new PKIXBuilderParameters(anchors, selector);
        params.addCertStore(certs);
        params.setRevocationEnabled(false);
        params.setDate(new Date(114, 9, 1));   // 2014-09-01

        boolean success = Boolean.valueOf(args[2]);
        try {
            PKIXCertPathBuilderResult result =
                        (PKIXCertPathBuilderResult)builder.build(params);
            if (!success) {
                throw new Exception("expected algorithm disabled exception");
            }

            int length = Integer.parseInt(args[1]);
            List<? extends Certificate> path =
                                    result.getCertPath().getCertificates();
            if (length != path.size()) {
                throw new Exception("unexpected certification path length");
            }

            if (!path.isEmpty()) {    // the target is not a trust anchor
                if (!match(args[0], path.get(0))) {
                    throw new Exception("unexpected certificate");
                }
            }
        } catch (CertPathBuilderException cpbe) {
            if (success) {
                throw new Exception("unexpected exception", cpbe);
            } else {
                System.out.println("Get the expected exception " + cpbe);
            }
        }
    }

}
