% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kNN.R
\name{kNN}
\alias{kNN}
\alias{knn}
\alias{sort.kNN}
\alias{adjacencylist.kNN}
\alias{print.kNN}
\title{Find the k Nearest Neighbors}
\usage{
kNN(
  x,
  k,
  query = NULL,
  sort = TRUE,
  search = "kdtree",
  bucketSize = 10,
  splitRule = "suggest",
  approx = 0
)

\method{sort}{kNN}(x, decreasing = FALSE, ...)

\method{adjacencylist}{kNN}(x, ...)

\method{print}{kNN}(x, ...)
}
\arguments{
\item{x}{a data matrix, a \link{dist} object or a \link{kNN} object.}

\item{k}{number of neighbors to find.}

\item{query}{a data matrix with the points to query. If query is not
specified, the NN for all the points in \code{x} is returned. If query is
specified then \code{x} needs to be a data matrix.}

\item{sort}{sort the neighbors by distance? Note that some search methods
already sort the results. Sorting is expensive and \code{sort = FALSE} may
be much faster for some search methods. kNN objects can be sorted using
\code{sort()}.}

\item{search}{nearest neighbor search strategy (one of \code{"kdtree"}, \code{"linear"} or
\code{"dist"}).}

\item{bucketSize}{max size of the kd-tree leafs.}

\item{splitRule}{rule to split the kd-tree. One of \code{"STD"}, \code{"MIDPT"}, \code{"FAIR"},
\code{"SL_MIDPT"}, \code{"SL_FAIR"} or \code{"SUGGEST"} (SL stands for sliding). \code{"SUGGEST"} uses
ANNs best guess.}

\item{approx}{use approximate nearest neighbors. All NN up to a distance of
a factor of \code{1 + approx} eps may be used. Some actual NN may be omitted
leading to spurious clusters and noise points.  However, the algorithm will
enjoy a significant speedup.}

\item{decreasing}{sort in decreasing order?}

\item{...}{further arguments}
}
\value{
An object of class \code{kNN} (subclass of \link{NN}) containing a
list with the following components:
\item{dist }{a matrix with distances. }
\item{id }{a matrix with \code{ids}. }
\item{k }{number \code{k} used. }
\item{metric }{ used distance metric. }
}
\description{
This function uses a kd-tree to find all k nearest neighbors in a data
matrix (including distances) fast.
}
\details{
\strong{Ties:} If the kth and the (k+1)th nearest neighbor are tied, then the
neighbor found first is returned and the other one is ignored.

\strong{Self-matches:} If no query is specified, then self-matches are
removed.

Details on the search parameters:
\itemize{
\item \code{search} controls if
a kd-tree or linear search (both implemented in the ANN library; see Mount
and Arya, 2010). Note, that these implementations cannot handle NAs.
\code{search = "dist"} precomputes Euclidean distances first using R. NAs are
handled, but the resulting distance matrix cannot contain NAs. To use other
distance measures, a precomputed distance matrix can be provided as \code{x}
(\code{search} is ignored).
\item \code{bucketSize} and \code{splitRule} influence how the kd-tree is
built. \code{approx} uses the approximate nearest neighbor search
implemented in ANN. All nearest neighbors up to a distance of
\code{eps / (1 + approx)} will be considered and all with a distance
greater than \code{eps} will not be considered. The other points might be
considered. Note that this results in some actual nearest neighbors being
omitted leading to spurious clusters and noise points. However, the
algorithm will enjoy a significant speedup. For more details see Mount and
Arya (2010).
}
}
\examples{
data(iris)
x <- iris[, -5]

# Example 1: finding kNN for all points in a data matrix (using a kd-tree)
nn <- kNN(x, k = 5)
nn

# explore neighborhood of point 10
i <- 10
nn$id[i,]
plot(x, col = ifelse(seq_len(nrow(iris)) \%in\% nn$id[i,], "red", "black"))

# visualize the 5 nearest neighbors
plot(nn, x)

# visualize a reduced 2-NN graph
plot(kNN(nn, k = 2), x)

# Example 2: find kNN for query points
q <- x[c(1,100),]
nn <- kNN(x, k = 10, query = q)

plot(nn, x, col = "grey")
points(q, pch = 3, lwd = 2)

# Example 3: find kNN using distances
d <- dist(x, method = "manhattan")
nn <- kNN(d, k = 1)
plot(nn, x)
}
\references{
David M. Mount and Sunil Arya (2010). ANN: A Library for
Approximate Nearest Neighbor Searching,
\url{http://www.cs.umd.edu/~mount/ANN/}.
}
\seealso{
Other NN functions: 
\code{\link{NN}},
\code{\link{comps}()},
\code{\link{frNN}()},
\code{\link{kNNdist}()},
\code{\link{sNN}()}
}
\author{
Michael Hahsler
}
\concept{NN functions}
\keyword{model}
