/*
 * Copyright (C) Research In Motion Limited 2010. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include "SVGResources.h"

#include "FilterOperation.h"
#include "LegacyRenderSVGRoot.h"
#include "PathOperation.h"
#include "RenderSVGResourceClipperInlines.h"
#include "RenderSVGResourceFilterInlines.h"
#include "RenderSVGResourceMarkerInlines.h"
#include "RenderSVGResourceMaskerInlines.h"
#include "SVGElementTypeHelpers.h"
#include "SVGFilterElement.h"
#include "SVGGradientElement.h"
#include "SVGMarkerElement.h"
#include "SVGNames.h"
#include "SVGPatternElement.h"
#include "SVGRenderStyle.h"
#include "SVGURIReference.h"
#include "StyleCachedImage.h"
#include <wtf/RobinHoodHashSet.h>

#if ENABLE(TREE_DEBUGGING)
#include <stdio.h>
#endif

namespace WebCore {

SVGResources::SVGResources()
{
}

static MemoryCompactLookupOnlyRobinHoodHashSet<AtomString> tagSet(std::span<decltype(SVGNames::aTag)* const> tags)
{
    MemoryCompactLookupOnlyRobinHoodHashSet<AtomString> set;
    set.reserveInitialCapacity(tags.size());
    for (auto& tag : tags)
        set.add(tag->get().localName());
    return set;
}

static const MemoryCompactLookupOnlyRobinHoodHashSet<AtomString>& clipperFilterMaskerTags()
{
    static constexpr std::array tags {
        // "container elements": http://www.w3.org/TR/SVG11/intro.html#TermContainerElement
        // "graphics elements" : http://www.w3.org/TR/SVG11/intro.html#TermGraphicsElement
        &SVGNames::aTag,
        &SVGNames::circleTag,
        &SVGNames::ellipseTag,
        &SVGNames::glyphTag,
        &SVGNames::gTag,
        &SVGNames::imageTag,
        &SVGNames::lineTag,
        &SVGNames::markerTag,
        &SVGNames::maskTag,
        &SVGNames::missing_glyphTag,
        &SVGNames::pathTag,
        &SVGNames::polygonTag,
        &SVGNames::polylineTag,
        &SVGNames::rectTag,
        &SVGNames::svgTag,
        &SVGNames::textTag,
        &SVGNames::useTag,

        // Not listed in the definitions is the clipPath element, the SVG spec says though:
        // The "clipPath" element or any of its children can specify property "clip-path".
        // So we have to add clipPathTag here, otherwhise clip-path on clipPath will fail.
        // (Already mailed SVG WG, waiting for a solution)
        &SVGNames::clipPathTag,

        // Not listed in the definitions are the text content elements, though filter/clipper/masker on tspan/text/.. is allowed.
        // (Already mailed SVG WG, waiting for a solution)
        &SVGNames::altGlyphTag,
        &SVGNames::textPathTag,
        &SVGNames::trefTag,
        &SVGNames::tspanTag,

        // Not listed in the definitions is the foreignObject element, but clip-path
        // is a supported attribute.
        &SVGNames::foreignObjectTag,

        // Elements that we ignore, as it doesn't make any sense.
        // defs, pattern, switch (FIXME: Mail SVG WG about these)
        // symbol (is converted to a svg element, when referenced by use, we can safely ignore it.)
    };
    static NeverDestroyed set = tagSet(tags);
    return set;
}

static const MemoryCompactLookupOnlyRobinHoodHashSet<AtomString>& markerTags()
{
    static constexpr std::array tags {
        &SVGNames::lineTag,
        &SVGNames::pathTag,
        &SVGNames::polygonTag,
        &SVGNames::polylineTag,
    };
    static NeverDestroyed set = tagSet(tags);
    return set;
}

static const MemoryCompactLookupOnlyRobinHoodHashSet<AtomString>& fillAndStrokeTags()
{
    static constexpr std::array tags {
        &SVGNames::altGlyphTag,
        &SVGNames::circleTag,
        &SVGNames::ellipseTag,
        &SVGNames::lineTag,
        &SVGNames::pathTag,
        &SVGNames::polygonTag,
        &SVGNames::polylineTag,
        &SVGNames::rectTag,
        &SVGNames::textTag,
        &SVGNames::textPathTag,
        &SVGNames::trefTag,
        &SVGNames::tspanTag,
    };
    static NeverDestroyed set = tagSet(tags);
    return set;
}

static const MemoryCompactLookupOnlyRobinHoodHashSet<AtomString>& chainableResourceTags()
{
    static constexpr std::array tags {
        &SVGNames::linearGradientTag,
        &SVGNames::filterTag,
        &SVGNames::patternTag,
        &SVGNames::radialGradientTag,
    };
    static NeverDestroyed set = tagSet(tags);
    return set;
}

static inline String targetReferenceFromResource(SVGElement& element)
{
    String target;
    if (is<SVGPatternElement>(element))
        target = downcast<SVGPatternElement>(element).href();
    else if (is<SVGGradientElement>(element))
        target = downcast<SVGGradientElement>(element).href();
    else if (is<SVGFilterElement>(element))
        target = downcast<SVGFilterElement>(element).href();
    else
        ASSERT_NOT_REACHED();

    return SVGURIReference::fragmentIdentifierFromIRIString(target, element.document());
}

static inline bool isChainableResource(const SVGElement& element, const SVGElement& linkedResource)
{
    if (is<SVGPatternElement>(element))
        return is<SVGPatternElement>(linkedResource);

    if (is<SVGGradientElement>(element))
        return is<SVGGradientElement>(linkedResource);
    
    if (is<SVGFilterElement>(element))
        return is<SVGFilterElement>(linkedResource);

    ASSERT_NOT_REACHED();
    return false;
}

static inline RenderSVGResourceContainer* paintingResourceFromSVGPaint(TreeScope& treeScope, const SVGPaintType& paintType, const String& paintUri, AtomString& id, bool& hasPendingResource)
{
    if (paintType != SVGPaintType::URI && paintType != SVGPaintType::URIRGBColor && paintType != SVGPaintType::URICurrentColor)
        return nullptr;

    id = SVGURIReference::fragmentIdentifierFromIRIString(paintUri, treeScope.documentScope());
    RenderSVGResourceContainer* container = getRenderSVGResourceContainerById(treeScope, id);
    if (!container) {
        hasPendingResource = true;
        return nullptr;
    }

    RenderSVGResourceType resourceType = container->resourceType();
    if (resourceType != PatternResourceType && resourceType != LinearGradientResourceType && resourceType != RadialGradientResourceType)
        return nullptr;

    return container;
}

bool SVGResources::buildCachedResources(const RenderElement& renderer, const RenderStyle& style)
{
    ASSERT(renderer.element());
    ASSERT_WITH_SECURITY_IMPLICATION(renderer.element()->isSVGElement());

    if (!renderer.element())
        return false;

    auto& element = downcast<SVGElement>(*renderer.element());

    auto& treeScope = element.treeScopeForSVGReferences();
    Document& document = treeScope.documentScope();

    const AtomString& tagName = element.localName();
    if (tagName.isNull())
        return false;

    const SVGRenderStyle& svgStyle = style.svgStyle();

    bool foundResources = false;
    if (clipperFilterMaskerTags().contains(tagName)) {
        if (is<ReferencePathOperation>(style.clipPath())) {
            // FIXME: -webkit-clip-path should support external resources
            // https://bugs.webkit.org/show_bug.cgi?id=127032
            auto& clipPath = downcast<ReferencePathOperation>(*style.clipPath());
            AtomString id(clipPath.fragment());
            if (setClipper(getRenderSVGResourceById<RenderSVGResourceClipper>(treeScope, id)))
                foundResources = true;
            else
                treeScope.addPendingSVGResource(id, element);
        }

        if (style.hasFilter()) {
            const FilterOperations& filterOperations = style.filter();
            if (filterOperations.size() == 1) {
                const FilterOperation& filterOperation = *filterOperations.at(0);
                if (filterOperation.type() == FilterOperation::Type::Reference) {
                    const auto& referenceFilterOperation = downcast<ReferenceFilterOperation>(filterOperation);
                    AtomString id = SVGURIReference::fragmentIdentifierFromIRIString(referenceFilterOperation.url(), element.document());
                    if (setFilter(getRenderSVGResourceById<RenderSVGResourceFilter>(treeScope, id)))
                        foundResources = true;
                    else
                        treeScope.addPendingSVGResource(id, element);
                }
            }
        }

        if (style.hasPositionedMask()) {
            // FIXME: We should support all the values in the CSS mask property, but for now just use the first mask-image if it's a reference.
            auto* maskImage = style.maskImage();
            if (is<StyleCachedImage>(maskImage)) {
                auto resourceID = SVGURIReference::fragmentIdentifierFromIRIString(downcast<StyleCachedImage>(*maskImage).reresolvedURL(document).string(), document);
                if (setMasker(getRenderSVGResourceById<RenderSVGResourceMasker>(treeScope, resourceID)))
                    foundResources = true;
                else
                    treeScope.addPendingSVGResource(resourceID, element);
            }
        }
    }

    if (markerTags().contains(tagName) && svgStyle.hasMarkers()) {
        auto buildCachedMarkerResource = [&](const String& markerResource, bool (SVGResources::*setMarker)(RenderSVGResourceMarker*)) {
            auto markerId = SVGURIReference::fragmentIdentifierFromIRIString(markerResource, document);
            if ((this->*setMarker)(getRenderSVGResourceById<RenderSVGResourceMarker>(treeScope, markerId)))
                foundResources = true;
            else
                treeScope.addPendingSVGResource(markerId, element);
        };
        buildCachedMarkerResource(svgStyle.markerStartResource(), &SVGResources::setMarkerStart);
        buildCachedMarkerResource(svgStyle.markerMidResource(), &SVGResources::setMarkerMid);
        buildCachedMarkerResource(svgStyle.markerEndResource(), &SVGResources::setMarkerEnd);
    }

    if (fillAndStrokeTags().contains(tagName)) {
        if (svgStyle.hasFill()) {
            bool hasPendingResource = false;
            AtomString id;
            if (setFill(paintingResourceFromSVGPaint(treeScope, svgStyle.fillPaintType(), svgStyle.fillPaintUri(), id, hasPendingResource)))
                foundResources = true;
            else if (hasPendingResource)
                treeScope.addPendingSVGResource(id, element);
        }

        if (svgStyle.hasStroke()) {
            bool hasPendingResource = false;
            AtomString id;
            if (setStroke(paintingResourceFromSVGPaint(treeScope, svgStyle.strokePaintType(), svgStyle.strokePaintUri(), id, hasPendingResource)))
                foundResources = true;
            else if (hasPendingResource)
                treeScope.addPendingSVGResource(id, element);
        }
    }

    if (chainableResourceTags().contains(tagName)) {
        AtomString id(targetReferenceFromResource(element));
        auto* linkedResource = getRenderSVGResourceContainerById(document, id);
        if (!linkedResource)
            treeScope.addPendingSVGResource(id, element);
        else if (isChainableResource(element, linkedResource->element())) {
            setLinkedResource(linkedResource);
            foundResources = true;
        }
    }

    return foundResources;
}

void SVGResources::layoutDifferentRootIfNeeded(const LegacyRenderSVGRoot* svgRoot)
{
    auto layoutDifferentRootIfNeeded = [&](RenderElement* container) {
        if (!container)
            return;
        auto* root = SVGRenderSupport::findTreeRootObject(*container);
        if (svgRoot == root || root->isInLayout())
            return;
        container->layoutIfNeeded();
    };
    layoutDifferentRootIfNeeded(clipper());
    layoutDifferentRootIfNeeded(masker());
    layoutDifferentRootIfNeeded(filter());
    layoutDifferentRootIfNeeded(markerStart());
    layoutDifferentRootIfNeeded(markerMid());
    layoutDifferentRootIfNeeded(markerEnd());
}

bool SVGResources::markerReverseStart() const
{
    return m_markerData
        && m_markerData->markerStart
        && m_markerData->markerStart->markerElement().orientType() == SVGMarkerOrientAutoStartReverse;
}

void SVGResources::removeClientFromCache(RenderElement& renderer, bool markForInvalidation) const
{
    if (isEmpty())
        return;

    if (m_linkedResource) {
        ASSERT(!m_clipperFilterMaskerData);
        ASSERT(!m_markerData);
        ASSERT(!m_fillStrokeData);
        m_linkedResource->removeClientFromCache(renderer, markForInvalidation);
        return;
    }

    if (m_clipperFilterMaskerData) {
        if (auto* clipper = m_clipperFilterMaskerData->clipper.get())
            clipper->removeClientFromCache(renderer, markForInvalidation);
        if (auto* filter = m_clipperFilterMaskerData->filter.get())
            filter->removeClientFromCache(renderer, markForInvalidation);
        if (auto* masker = m_clipperFilterMaskerData->masker.get())
            masker->removeClientFromCache(renderer, markForInvalidation);
    }

    if (m_markerData) {
        if (auto* markerStart = m_markerData->markerStart.get())
            markerStart->removeClientFromCache(renderer, markForInvalidation);
        if (auto* markerMid = m_markerData->markerMid.get())
            markerMid->removeClientFromCache(renderer, markForInvalidation);
        if (auto* markerEnd = m_markerData->markerEnd.get())
            markerEnd->removeClientFromCache(renderer, markForInvalidation);
    }

    if (m_fillStrokeData) {
        if (auto* fill = m_fillStrokeData->fill.get())
            fill->removeClientFromCache(renderer, markForInvalidation);
        if (auto* stroke = m_fillStrokeData->stroke.get())
            stroke->removeClientFromCache(renderer, markForInvalidation);
    }
}

bool SVGResources::resourceDestroyed(RenderSVGResourceContainer& resource)
{
    if (isEmpty())
        return false;

    if (m_linkedResource == &resource) {
        ASSERT(!m_clipperFilterMaskerData);
        ASSERT(!m_markerData);
        ASSERT(!m_fillStrokeData);
        m_linkedResource->removeAllClientsFromCache();
        m_linkedResource = nullptr;
        return true;
    }

    bool foundResources = false;
    switch (resource.resourceType()) {
    case MaskerResourceType:
        if (!m_clipperFilterMaskerData)
            break;
        if (m_clipperFilterMaskerData->masker == &resource) {
            resource.removeAllClientsFromCache();
            m_clipperFilterMaskerData->masker = nullptr;
            foundResources = true;
        }
        break;
    case MarkerResourceType:
        if (!m_markerData)
            break;
        if (m_markerData->markerStart == &resource) {
            resource.removeAllClientsFromCache();
            m_markerData->markerStart = nullptr;
            foundResources = true;
        }
        if (m_markerData->markerMid == &resource) {
            resource.removeAllClientsFromCache();
            m_markerData->markerMid = nullptr;
            foundResources = true;
        }
        if (m_markerData->markerEnd == &resource) {
            resource.removeAllClientsFromCache();
            m_markerData->markerEnd = nullptr;
            foundResources = true;
        }
        break;
    case PatternResourceType:
    case LinearGradientResourceType:
    case RadialGradientResourceType:
        if (!m_fillStrokeData)
            break;
        if (m_fillStrokeData->fill == &resource) {
            resource.removeAllClientsFromCache();
            m_fillStrokeData->fill = nullptr;
            foundResources = true;
        }
        if (m_fillStrokeData->stroke == &resource) {
            resource.removeAllClientsFromCache();
            m_fillStrokeData->stroke = nullptr;
            foundResources = true;
        }
        break;
    case FilterResourceType:
        if (!m_clipperFilterMaskerData)
            break;
        if (m_clipperFilterMaskerData->filter == &resource) {
            resource.removeAllClientsFromCache();
            m_clipperFilterMaskerData->filter = nullptr;
            foundResources = true;
        }
        break;
    case ClipperResourceType:
        if (!m_clipperFilterMaskerData)
            break; 
        if (m_clipperFilterMaskerData->clipper == &resource) {
            resource.removeAllClientsFromCache();
            m_clipperFilterMaskerData->clipper = nullptr;
            foundResources = true;
        }
        break;
    case SolidColorResourceType:
        ASSERT_NOT_REACHED();
    }
    return foundResources;
}

void SVGResources::buildSetOfResources(WeakHashSet<RenderSVGResourceContainer>& set)
{
    if (isEmpty())
        return;

    if (auto* linkedResource = m_linkedResource.get()) {
        ASSERT(!m_clipperFilterMaskerData);
        ASSERT(!m_markerData);
        ASSERT(!m_fillStrokeData);
        set.add(*linkedResource);
        return;
    }

    if (m_clipperFilterMaskerData) {
        if (auto* clipper = m_clipperFilterMaskerData->clipper.get())
            set.add(*clipper);
        if (auto* filter = m_clipperFilterMaskerData->filter.get())
            set.add(*filter);
        if (auto* masker = m_clipperFilterMaskerData->masker.get())
            set.add(*masker);
    }

    if (m_markerData) {
        if (auto* markerStart = m_markerData->markerStart.get())
            set.add(*markerStart);
        if (auto* markerMid = m_markerData->markerMid.get())
            set.add(*markerMid);
        if (auto* markerEnd = m_markerData->markerEnd.get())
            set.add(*markerEnd);
    }

    if (m_fillStrokeData) {
        if (auto* fill = m_fillStrokeData->fill.get())
            set.add(*fill);
        if (auto* stroke = m_fillStrokeData->stroke.get())
            set.add(*stroke);
    }
}

bool SVGResources::setClipper(RenderSVGResourceClipper* clipper)
{
    if (!clipper)
        return false;

    ASSERT(clipper->resourceType() == ClipperResourceType);

    if (!m_clipperFilterMaskerData)
        m_clipperFilterMaskerData = makeUnique<ClipperFilterMaskerData>();

    m_clipperFilterMaskerData->clipper = clipper;
    return true;
}

void SVGResources::resetClipper()
{
    ASSERT(m_clipperFilterMaskerData);
    ASSERT(m_clipperFilterMaskerData->clipper);
    m_clipperFilterMaskerData->clipper = nullptr;
}

bool SVGResources::setFilter(RenderSVGResourceFilter* filter)
{
    if (!filter)
        return false;

    ASSERT(filter->resourceType() == FilterResourceType);

    if (!m_clipperFilterMaskerData)
        m_clipperFilterMaskerData = makeUnique<ClipperFilterMaskerData>();

    m_clipperFilterMaskerData->filter = filter;
    return true;
}

void SVGResources::resetFilter()
{
    ASSERT(m_clipperFilterMaskerData);
    ASSERT(m_clipperFilterMaskerData->filter);
    m_clipperFilterMaskerData->filter = nullptr;
}

bool SVGResources::setMarkerStart(RenderSVGResourceMarker* markerStart)
{
    if (!markerStart)
        return false;

    ASSERT(markerStart->resourceType() == MarkerResourceType);

    if (!m_markerData)
        m_markerData = makeUnique<MarkerData>();

    m_markerData->markerStart = markerStart;
    return true;
}

void SVGResources::resetMarkerStart()
{
    ASSERT(m_markerData);
    ASSERT(m_markerData->markerStart);
    m_markerData->markerStart = nullptr;
}

bool SVGResources::setMarkerMid(RenderSVGResourceMarker* markerMid)
{
    if (!markerMid)
        return false;

    ASSERT(markerMid->resourceType() == MarkerResourceType);

    if (!m_markerData)
        m_markerData = makeUnique<MarkerData>();

    m_markerData->markerMid = markerMid;
    return true;
}

void SVGResources::resetMarkerMid()
{
    ASSERT(m_markerData);
    ASSERT(m_markerData->markerMid);
    m_markerData->markerMid = nullptr;
}

bool SVGResources::setMarkerEnd(RenderSVGResourceMarker* markerEnd)
{
    if (!markerEnd)
        return false;

    ASSERT(markerEnd->resourceType() == MarkerResourceType);

    if (!m_markerData)
        m_markerData = makeUnique<MarkerData>();

    m_markerData->markerEnd = markerEnd;
    return true;
}

void SVGResources::resetMarkerEnd()
{
    ASSERT(m_markerData);
    ASSERT(m_markerData->markerEnd);
    m_markerData->markerEnd = nullptr;
}

bool SVGResources::setMasker(RenderSVGResourceMasker* masker)
{
    if (!masker)
        return false;

    ASSERT(masker->resourceType() == MaskerResourceType);

    if (!m_clipperFilterMaskerData)
        m_clipperFilterMaskerData = makeUnique<ClipperFilterMaskerData>();

    m_clipperFilterMaskerData->masker = masker;
    return true;
}

void SVGResources::resetMasker()
{
    ASSERT(m_clipperFilterMaskerData);
    ASSERT(m_clipperFilterMaskerData->masker);
    m_clipperFilterMaskerData->masker = nullptr;
}

bool SVGResources::setFill(RenderSVGResourceContainer* fill)
{
    if (!fill)
        return false;

    ASSERT(fill->resourceType() == PatternResourceType
           || fill->resourceType() == LinearGradientResourceType
           || fill->resourceType() == RadialGradientResourceType);

    if (!m_fillStrokeData)
        m_fillStrokeData = makeUnique<FillStrokeData>();

    m_fillStrokeData->fill = fill;
    return true;
}

void SVGResources::resetFill()
{
    ASSERT(m_fillStrokeData);
    ASSERT(m_fillStrokeData->fill);
    m_fillStrokeData->fill = nullptr;
}

bool SVGResources::setStroke(RenderSVGResourceContainer* stroke)
{
    if (!stroke)
        return false;

    ASSERT(stroke->resourceType() == PatternResourceType
           || stroke->resourceType() == LinearGradientResourceType
           || stroke->resourceType() == RadialGradientResourceType);

    if (!m_fillStrokeData)
        m_fillStrokeData = makeUnique<FillStrokeData>();

    m_fillStrokeData->stroke = stroke;
    return true;
}

void SVGResources::resetStroke()
{
    ASSERT(m_fillStrokeData);
    ASSERT(m_fillStrokeData->stroke);
    m_fillStrokeData->stroke = nullptr;
}

bool SVGResources::setLinkedResource(RenderSVGResourceContainer* linkedResource)
{
    if (!linkedResource)
        return false;

    m_linkedResource = linkedResource;
    return true;
}

void SVGResources::resetLinkedResource()
{
    ASSERT(m_linkedResource);
    m_linkedResource = nullptr;
}

#if ENABLE(TREE_DEBUGGING)
void SVGResources::dump(const RenderObject* object)
{
    ASSERT(object);
    ASSERT(object->node());

    fprintf(stderr, "-> this=%p, SVGResources(renderer=%p, node=%p)\n", this, object, object->node());
    fprintf(stderr, " | DOM Tree:\n");
    object->node()->showTreeForThis();

    fprintf(stderr, "\n | List of resources:\n");
    if (m_clipperFilterMaskerData) {
        if (auto* clipper = m_clipperFilterMaskerData->clipper.get())
            fprintf(stderr, " |-> Clipper    : %p (node=%p)\n", clipper, &clipper->clipPathElement());
        if (auto* filter = m_clipperFilterMaskerData->filter.get())
            fprintf(stderr, " |-> Filter     : %p (node=%p)\n", filter, &filter->filterElement());
        if (auto* masker = m_clipperFilterMaskerData->masker.get())
            fprintf(stderr, " |-> Masker     : %p (node=%p)\n", masker, &masker->maskElement());
    }

    if (m_markerData) {
        if (auto* markerStart = m_markerData->markerStart.get())
            fprintf(stderr, " |-> MarkerStart: %p (node=%p)\n", markerStart, &markerStart->markerElement());
        if (auto* markerMid = m_markerData->markerMid.get())
            fprintf(stderr, " |-> MarkerMid  : %p (node=%p)\n", markerMid, &markerMid->markerElement());
        if (auto* markerEnd = m_markerData->markerEnd.get())
            fprintf(stderr, " |-> MarkerEnd  : %p (node=%p)\n", markerEnd, &markerEnd->markerElement());
    }

    if (m_fillStrokeData) {
        if (auto* fill = m_fillStrokeData->fill.get())
            fprintf(stderr, " |-> Fill       : %p (node=%p)\n", fill, &fill->element());
        if (auto* stroke = m_fillStrokeData->stroke.get())
            fprintf(stderr, " |-> Stroke     : %p (node=%p)\n", stroke, &stroke->element());
    }

    if (m_linkedResource)
        fprintf(stderr, " |-> xlink:href : %p (node=%p)\n", m_linkedResource.get(), &m_linkedResource->element());
}
#endif

}
