/* NVTV Brooktree TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_bt.c,v 1.20 2004/01/27 17:05:25 dthierbach Exp $
 *
 * Contents:
 *
 * Routines to access the Brooktree chip registers via the I2C bus.
 *
 */

#include "local.h" /* before everything else */
#include "xf86_ansic.h" 

#include "bitmask.h"
#include "xf86i2c.h"
#include "tv_i2c.h"
#include "tv_bt.h"

/* -------- Brooktree -------- */

/* I have no idea what registers 0xda-0xfc do, but mode 0 is supposed
 * to disable Macrovision. However, on my card Macrovision does not
 * seem to work anyway, so it improves just image quality :-) */

static I2CByte tvBtNtscMacro0 [] = {
  0x0f,0xfc,0x20,0xd0,0x6f,0x0f,0x00,0x00,0x0c,0xf3,0x09,
  0xbd,0x67,0xb5,0x90,0xb2,0x7d,0x00,0x00};
static I2CByte tvBtNtscMacro1 [] = {
  0x0f,0xfc,0x20,0xd0,0x6f,0x0f,0x00,0x00,0x0c,0xf3,0x09,
  0xbd,0x67,0xb5,0x90,0xb2,0x7d,0x63,0x00};
static I2CByte tvBtNtscMacro2 [] = {
  0x0f,0xfc,0x20,0xd0,0x6f,0x0f,0x00,0x00,0x0c,0xf3,0x09,
  0xbd,0x6c,0x31,0x92,0x32,0xdd,0xe3,0x00};
static I2CByte tvBtNtscMacro3 [] = {
  0x0f,0xfc,0x20,0xd0,0x6f,0x0f,0x00,0x00,0x0c,0xf3,0x09,
  0xbd,0x66,0xb5,0x90,0xb2,0x7d,0xe3,0x00};

static I2CByte tvBtPalMacro0 [] = {
  0x05,0x57,0x20,0x40,0x6e,0x7e,0xf4,0x51,0x0f,0xf1,0x05,
  0xd3,0x78,0xa2,0x25,0x54,0xa5,0x00,0x00};
static I2CByte tvBtPalMacro1 [] = {
  0x05,0x57,0x20,0x40,0x6e,0x7e,0xf4,0x51,0x0f,0xf1,0x05,
  0xd3,0x78,0xa2,0x25,0x54,0xa5,0x63,0x00};

void TVBtMacroMode (TVEncoderObj *this, TVEncoderRegs *r)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  RAISE (MSG_DEBUG, "tv macro bt %i", r->bt.macro); 
  switch (r->bt.macro) {
    case 0: /* NTSC APS 0 */
      TVWriteSeqBus (dev, 0xda, tvBtNtscMacro0, sizeof (tvBtNtscMacro0));
      break;
    case 1: /* NTSC APS 1 */
      TVWriteSeqBus (dev, 0xda, tvBtNtscMacro1, sizeof (tvBtNtscMacro1));
      break;
    case 2: /* NTSC APS 2 */
      TVWriteSeqBus (dev, 0xda, tvBtNtscMacro2, sizeof (tvBtNtscMacro2));
      break;
    case 3: /* NTSC APS 3 */
      TVWriteSeqBus (dev, 0xda, tvBtNtscMacro3, sizeof (tvBtNtscMacro3));
      break;
    case 4: /* PAL APS 0 */
      TVWriteSeqBus (dev, 0xda, tvBtPalMacro0, sizeof (tvBtPalMacro0));
      break;
    case 5: /* PAL APS 1 */
    case 6:
    case 7:
      TVWriteSeqBus (dev, 0xda, tvBtPalMacro1, sizeof (tvBtPalMacro1));
      break;
  }
}

void TVBtCreate (TVEncoderObj *this, TVChip chip_type, void *ctrl)
{
  RAISE (MSG_DEBUG, "tv create bt");
  this->type = chip_type;
  this->ctrl = ctrl;
  this->hwconfig = 0;    /* No hardware config for BT */
  this->hwstate = 0x00;  /* Assume DACs are enabled, till init */
}

/* All DACs off, clock output disabled. */

/*
 *  B8-3   en_pincfg
 *  B8-2:0 config
 *  BA-7   sreset = 0 (default)
 *  BA-6   check_stat
 *  BA-5   slaver = 0 (master mode)
 *  BA-4   dacoff
 *  BA-2:0 dacdis c-a
 *  C4-7:6 estatus
 *  C4-5   eccf2
 *  C4-4   eccf1
 *  C4-3   eccgate
 *  C4-2   ecbar
 *  C4-1   dchroma
 *  C4-0   en_out
 *  C6-7   en_blanko = 1 (default) (no blank)
 *  C6-6   en_dot = 0 (default) (internal blanking)
 *  C6-5   fieldi
 *  C6-4   vsynci = 1 (active high) (default 0) (ok for nv)
 *  C6-3   hsynci = 1 (active high) (default 0) (ok for nv)
 *  C6-2:0 in_mode = 0 (24-bit RGB multiplexed)
 *  CE     out_mux c-a
 *  D4-7   mode2x = 0 (default)
 *  D4-6   div2
 *  D4-5   en_async = 0 (default)
 *  D4-4:0 (ccr stuff)
 *  D6-3:2 out_mode = 0 (default: CVBS/Y/C/Y_DLY)
 *  D6-1:0 lumadly = 0 (default)
 */

void TVBtSetPort (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  RAISE (MSG_DEBUG, "tv port bt %08X", port);

  if ((port & PORT_FORMAT_MASK) != PORT_FORMAT_RGB) {
    RAISE (MSG_ABORT, "TVBtSetPort: Only RGB multiplex format supported.");
  }
  if ((port & PORT_BLANK_POLARITY) != PORT_BLANK_LOW) {
    RAISE (MSG_ABORT, "TVBtSetPort: Only active low blank supported.");
  }
  if ((port & PORT_PCLK_MODE) != PORT_PCLK_MASTER) {
    RAISE (MSG_ABORT, "TVBtSetPort: Only clock master mode supported.");
  }
  if ((port & PORT_SYNC_DIR) == PORT_SYNC_SLAVE) {
    this->hwstate |= SetBit (5);
  } else {
    this->hwstate &= ~SetBit (5);
  }
  TVWriteBus (dev, 0xba, this->hwstate);
  TVWriteBus (dev, 0xc6, 0x00 /* in_mode */
  	      | ((port & PORT_BLANK_DIR) ? 
		  ((port & PORT_BLANK_MODE) ? SetBit(6) : 0) : SetBit (7))
	      | ((port & PORT_VSYNC_POLARITY) ? SetBit(4) : 0)
	      | ((port & PORT_HSYNC_POLARITY) ? SetBit(3) : 0));
}

void TVBtGetPort (TVEncoderObj *this, int *port)
{
  RAISE (MSG_ERROR, "TVBtGetPort not supported.");
}

void TVBtInitRegs (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  RAISE (MSG_DEBUG, "tv init bt");
  this->hwstate = 0x17;
  TVBtSetPort (this, port);
  TVWriteBus (dev, 0xb8, 0x03);
  TVWriteBus (dev, 0xc4, 0x00);
  TVWriteBus (dev, 0xce, 0x18); /* out_mux */
  TVWriteBus (dev, 0xd4, 0x00); /* Mode2x=0 */
  TVWriteBus (dev, 0xd6, 0x00); /* Out_Mode */
}

void TVBtUpdate (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  this->hwstate &= ~0x1f;
  if (state == TV_OFF) {
    this->hwstate |= 0x17; 
  } else {
    this->hwstate |= r->bt.flags3 & BT_FLAG3_DAC;
  }
  RAISE (MSG_DEBUG, "tv bt update %02X", this->hwstate);
}

void TVBtSetRegs (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  RAISE (MSG_DEBUG, "tv set bt");
  TVWriteBus (dev, 0x6e, Set8Bits(r->bt.hsynoffset));
  TVWriteBus (dev, 0x70, SetBitField(r->bt.hsynoffset,9:8,7:6)
                       | SetBitField(r->bt.hsynwidth,5:0,5:0));
  TVWriteBus (dev, 0x72, Set8Bits(r->bt.vsynoffset));
  TVWriteBus (dev, 0x74, SetBitField(r->bt.vsynoffset,10:8,5:3)
                       | SetBitField(r->bt.vsynwidth,2:0,2:0));
  TVWriteBus (dev, 0x76, Set8Bits(r->bt.h_clko));
  TVWriteBus (dev, 0x78, Set8Bits(r->bt.h_active));
  TVWriteBus (dev, 0x7a, Set8Bits(r->bt.hsync_width));
  TVWriteBus (dev, 0x7c, Set8Bits(r->bt.hburst_begin));
  TVWriteBus (dev, 0x7e, Set8Bits(r->bt.hburst_end));
  TVWriteBus (dev, 0x80, Set8Bits(r->bt.h_blanko));
  TVWriteBus (dev, 0x82, Set8Bits(r->bt.v_blanko));
  TVWriteBus (dev, 0x84, Set8Bits(r->bt.v_activeo));
  TVWriteBus (dev, 0x86, SetBitField(r->bt.v_activeo,8:8,7:7)
                       | SetBitField(r->bt.h_active,10:8,6:4)
                       | SetBitField(r->bt.h_clko,11:8,2:0));
  /* FIMXE: BT h_active only 9:8 -> 5:4 */
  TVWriteBus (dev, 0x88, Set8Bits(r->bt.h_fract));
  TVWriteBus (dev, 0x8a, Set8Bits(r->bt.h_clki));
  TVWriteBus (dev, 0x8c, Set8Bits(r->bt.h_blanki));
  TVWriteBus (dev, 0x8e, SetBitField(r->bt.h_blanki,8:8,3:3)
                       | SetBitField(r->bt.h_clki,10:8,2:0));
  /* FIXME: v_blank_dly */
  TVWriteBus (dev, 0x90, Set8Bits(r->bt.v_linesi));
  TVWriteBus (dev, 0x92, Set8Bits(r->bt.v_blanki));
  TVWriteBus (dev, 0x94, Set8Bits(r->bt.v_activei));
  TVWriteBus (dev, 0x96, SetBitField(r->bt.clpf,1:0,7:6)
                       | SetBitField(r->bt.ylpf,1:0,5:4)
                       | SetBitField(r->bt.v_activei,9:8,3:2)
                       | SetBitField(r->bt.v_linesi,9:8,1:0));
  TVWriteBus (dev, 0x98, Set8Bits(r->bt.v_scale));
  TVWriteBus (dev, 0x9a, SetBitField(r->bt.h_blanko,9:8,7:6)
                       | SetBitField(r->bt.v_scale,13:8,5:0));
  TVWriteBus (dev, 0x9c, Set8Bits(r->bt.pll_fract));
  TVWriteBus (dev, 0x9e, Set8Bits(r->bt.pll_fract >> 8));
  TVWriteBus (dev, 0xa0, 0 /* en_xclk, by_pll */ 
                       | SetBitField(r->bt.pll_int,5:0,5:0));
  TVWriteBus (dev, 0xa2, Set8Bits(r->bt.flags1));
  TVWriteBus (dev, 0xa4, Set8Bits(r->bt.sync_amp));
  TVWriteBus (dev, 0xa6, Set8Bits(r->bt.bst_amp));
  TVWriteBus (dev, 0xa8, Set8Bits(r->bt.mcr));
  TVWriteBus (dev, 0xaa, Set8Bits(r->bt.mcb));
  TVWriteBus (dev, 0xac, Set8Bits(r->bt.my));
  TVWriteBus (dev, 0xae, Set8Bits(r->bt.msc));
  TVWriteBus (dev, 0xb0, Set8Bits(r->bt.msc >> 8));
  TVWriteBus (dev, 0xb2, Set8Bits(r->bt.msc >> 16));
  TVWriteBus (dev, 0xb4, Set8Bits(r->bt.msc >> 24));
  TVWriteBus (dev, 0xb6, Set8Bits(r->bt.phase_off));
  TVBtUpdate (this, r, state);
  TVWriteBus (dev, 0xba, this->hwstate);
  TVWriteBus (dev, 0xc8, SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_YFLPF,7)
		       | SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_FFILT,6)
		       | SetBitField(r->bt.f_selc,2:0,5:3)
                       | SetBitField(r->bt.f_sely,2:0,2:0));
  TVWriteBus (dev, 0xca, SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_GMUSHY,7)
		       | SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_GMSHY,6)
		       | SetBitField(r->bt.ycoring,2:0,5:3)
                       | SetBitField(r->bt.yattenuate,2:0,2:0));
  TVWriteBus (dev, 0xcc, SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_GMUSHC,7)
		       | SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_GMSHC,6)
		       | SetBitField(r->bt.ccoring,2:0,5:3)
                       | SetBitField(r->bt.cattenuate,2:0,2:0));
  TVWriteBus (dev, 0xce, SetBitField(r->bt.out_muxa,1:0,1:0)
		       | SetBitField(r->bt.out_muxb,1:0,3:2)
                       | SetBitField(r->bt.out_muxc,1:0,5:4)
                       | SetBitField(r->bt.out_muxd,1:0,7:6));
  /* out_muxd only available for the conexant chip */
  TVWriteBus (dev, 0xd4, SetBitFlag(r->bt.flags1,BT_FLAG1_EN_ASYNC,5) 
                       | 0x00 /* rest all 0 */ );
  TVBtMacroMode (this, r);
}

void TVBtGetRegs (TVEncoderObj *this, TVEncoderRegs *r)
{
  RAISE (MSG_ERROR, "TVBtGetRegs not supported.");
}

void TVBtSetState (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  RAISE (MSG_DEBUG, "tv state bt %i", state);
  tvState = state;
  TVBtUpdate (this, r, state);
  switch (state)
  {
    case TV_OFF: 
      TVWriteBus (dev, 0xc4, 0x00); /* FIXME DIS_CHROMA */
      TVWriteBus (dev, 0xd4, 0x00); 
      TVWriteBus (dev, 0xba, this->hwstate);
      break;
    case TV_BARS: 
      TVWriteBus (dev, 0xc4, 0x05); /* FIXME DIS_CHROMA */
      TVWriteBus (dev, 0xd4, 0x00); 
      TVWriteBus (dev, 0xba, this->hwstate);
      break;
    case TV_ON: 
      TVWriteBus (dev, 0xc4, 0x01); /* FIXME DIS_CHROMA */
      TVWriteBus (dev, 0xd4, SetBitFlag(r->bt.flags1,BT_FLAG1_EN_ASYNC,5) 
		           | 0x00 /* rest all 0 */ );
      TVWriteBus (dev, 0xba, this->hwstate);
      break;
    default:
      break;
  }
  xf86usleep(1000); /* BT871 doc says: 1ms minimum */
  /* FIXME: Don't do that always */
  TVWriteBus (dev, 0x6c, 0x86); /* timing reset, en_reg_rd=0 */
  xf86usleep(1000); /* not necessary, but won't hurt ... */
}

int TVBtRead (TVEncoderObj *this, int index)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte status;

  tvBusOk = TRUE;
  switch (tvState)
  {
    case TV_OFF: 
      TVWriteBus (dev, 0xc4, 0x00 | SetBitField(index,1:0,7:6));
      break;
    case TV_BARS: 
      TVWriteBus (dev, 0xc4, 0x05 | SetBitField(index,1:0,7:6)); 
      break;
    case TV_ON: 
      TVWriteBus (dev, 0xc4, 0x01 | SetBitField(index,1:0,7:6)); 
      break;
    default:
      return -1;
  }
  TVStatusBus (dev, &status);
  if (!tvBusOk) 
    return -1;
  else 
    return status;
}

long TVBtGetStatus (TVEncoderObj *this, int index)
{ 
  return TVBtRead (this, 2) | ((tvState == TV_ON) ? 0x100 : 0);
}

TVConnect TVBtGetConnect (TVEncoderObj *this)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  int status;

  /* Enable all DACS and set check_stat */
  TVWriteBus (dev, 0xba, 0x40); 
  xf86usleep(50000); /* BT871 doc says: 20ms - 50ms */
  status = TVBtRead (this, 1);
  RAISE (MSG_DEBUG, "bt conn %02x (%s)", status, tvBusOk ? "ok" : "err");
  /* BT871 doc says we should reset check_stat */
  TVWriteBus (dev, 0xba, this->hwstate); 
  if (status >= 0) {
    switch (status & 0xe0) {
      case 0x80:  
	return CONNECT_COMPOSITE; 
	break;
      case 0x60:  
	return CONNECT_SVIDEO; 
	break;
      case 0x20: 
      case 0x40: 
      case 0xa0: /* Voodoo3 - only DAC_C via SVideo converter? */
      case 0xc0: /* Voodoo3 - only DAC_B, via internal converter (really) */
	return CONNECT_CONVERT; 
	break;
      default: 
	RAISE (MSG_WARNING, "Strange Brooktree connection status %02x.", 
	       status & 0xe0); 
      case 0xe0 : 
	return CONNECT_BOTH; 
	break;
    }
  } else {
    return CONNECT_NONE;
  }
}

/*
 * Check for Brooktree chip on device dev. Return version string if found, 
 * NULL otherwise. Don't use ReadBt.
 */

char *TVDetectBrooktree (I2CDevPtr dev, TVChip *encoder, TVState state)
{
  I2CByte status;
  char *chip;
  static char version [50];

#ifdef FAKE_PROBE_BROOKTREE
  *encoder = TV_BROOKTREE;
  snprintf (version, 50, "Brooktree Fake (%1s:%02X)", I2C_ID(dev));
  return version;
#else
#ifdef FAKE_I2C
  return NULL;
#endif
#endif  
  tvBusOk = TRUE;
  if (state != TV_OFF) {
    TVWriteBus (dev, 0xc4, 0x01);
  } else {
    TVWriteBus (dev, 0xc4, 0x00);
  }
  TVWriteBus (dev, 0x6c, 0x06); /* disable en_reg_rd, just in case */
  TVStatusBus (dev, &status);
  RAISE (MSG_DEBUG, "bt check (%i) %02x (%s)", state, status, 
	 tvBusOk ? "ok" : "err");
  if (!tvBusOk) return NULL;
  *encoder = TV_BROOKTREE;
  switch (status & 0xe0) {
    case 0x00 : chip = "868"; break;
    case 0x20 : chip = "869"; break;
    case 0x40 : chip = "870"; break;
    case 0x60 : chip = "871"; break;
    default : chip = NULL; break;
  }
  if (chip) {
    snprintf (version, 50, "Brooktree BT%s Rev %i (%1s:%02X)", 
	      chip, status & 0x1f, I2C_ID(dev));
    return version;
  } else {
    return NULL;
  }
}

TVEncoderObj tvBtTemplate = {
  type: TV_BROOKTREE, ctrl: NULL, minClock: 0, maxClock: 40500, 
  Create:     TVBtCreate,
  InitRegs:   TVBtInitRegs, 
  SetRegs:    TVBtSetRegs, 
  GetRegs:    TVBtGetRegs, 
  SetPort:    TVBtSetPort,
  GetPort:    TVBtGetPort,
  SetState:   TVBtSetState,
  GetConnect: TVBtGetConnect, 
  GetStatus:  TVBtGetStatus
};


/* 

Voodoo3 connectors:

DACA 0=CVBS (but seems to be pulled up internally; always detected)
DACB 1=Y    (or CVBS via SVideo converter type 1)
DACC 2=C    (or CVBS via SVideo converter type 2)

lmsensors: Clock slave/Sync slave

NVidia connectors:

DACA 0=CVBS
DACB 2=C
DACC 1=Y

FIXME: Must setup differently according to *card* backend.

*/

