/*
 *    dcl graphic driver for Post Script
 *
 *    (color version for only all-over filling)
 *
 *    based on z1pack.f by M. Shiotani, K. Sato, A. Numaguti, K. Ishioka
 *
 *    original  M. Shiotani (C version)
 *    93/02/01  A. Numaguti: conforming PostScript (PS-Adobe-2.0)
 *    94/01/28  S. Sakai: dcl-5.x.3
 *    94/03/15  M. Shiotani
 *
 *    Copyright (C) 2000-2004 GFD Dennou Club. All rights reserved.
 *
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#define XMAX   7500.0        /* max x  */
#define YMAX   5300.0        /* max y  */
#define TRUE   1             /* numeric value for true */
#define FALSE  0             /* numeric value for false */
#define LWDATR TRUE          /* line width  capability */
#define LTNATR TRUE          /* hard fill   capability */
#define LIMATR TRUE          /* bit image   capability */
#define LPTATR FALSE         /* mouse point capability */
#define MAXWDI 9             /* maximum number of line index */
#define MAXCLI 100           /* maximum number of color index */
#define MAXT   6             /* maximum number of tone pattern */
#define MAXNP  1000          /* maximum buffer size */
#define RGBMAX 65535         /* maximum rgb intensity */
#define A0     12.0          /* constant for tone pattern */
#define RFACT  1e8           /* constant for truncation */

static float xlwdth[MAXWDI] = { 2, 4, 6, 8, 10, 11, 12, 13, 14 };
                             /* initialization for line width */
static int itd[MAXT] = {1, 2, 4, 6, 8, 12};
static int itl[MAXT] = {1, 2, 3, 4, 6, 8};
static int rgb[MAXCLI];
static int r[MAXCLI], g[MAXCLI], b[MAXCLI];
static int imlen, ipxcl;
static int iwdidz, iwdidy, iclidz, iclidy, iwtroz, itbak;
static long np;
static int page = 0;

static float wsxwd, wsywd, wsxmnz, wsxmxz, wsymnz, wsymxz; /* window size */
static float factz, scale;

static FILE *outf;                       /* PS file */
static int color, separate, print;       /* initialized by z1dopn */
static char psfile[80], command[80], fname[80], xtitle[80];

static void z1fopn(), z1fcls();          /* internal routines */

/*------------------------- device ------------------------*/

void z1dopn_(width, height, lcolor, lsep, lprint,
	     cmap, file, title, com)
    int  *width, *height, *lcolor, *lsep, *lprint;
    char cmap[], file[], title[], com[];
{
    int ncolor, m, n;
    long int rx[MAXCLI], gx[MAXCLI], bx[MAXCLI];
    float scx, scy;
    char cmapz[80], c[80];
    FILE *stream;
    void cfnchr();

    color    = *lcolor;
    separate = *lsep | *lprint;
    print    = *lprint;

    cfnchr (command, com, 79);
    cfnchr (psfile, file, 79);
    cfnchr (xtitle, title, 79);

/* page size */

    scx   = XMAX / (*width -1);
    scy   = YMAX / (*height-1);
    scale = (scx < scy) ? scx : scy ;  /* min (scx, scy) */
    wsxwd  = (*width -1) * scale;      /* window width */
    wsywd  = (*height-1) * scale;      /* window height */
    wsxmnz = (XMAX - wsxwd) / 2;       /* lower-left  corner */
    wsymnz = (YMAX - wsywd) / 2;       /* lower-left  corner */
    wsxmxz = wsxmnz + wsxwd;           /* upper-right corner */
    wsymxz = wsymnz + wsywd;           /* upper-right corner */
    factz  = 0.0035;

/* read colormap file */

    cfnchr (cmapz, cmap, 79);

    if ((stream = fopen(cmapz, "r")) == NULL) {
	fprintf (stderr, "*** Error in z1dopn : ");
	fprintf (stderr,
		 "Allocation failed for colormap (%s).\n", cmapz);
	exit (1);
    }

    fscanf (stream, "%d : %s", &ncolor, c);
    for (n = 0; n < ncolor; n++){
	fscanf(stream, "%6ld%6ld%6ld : %s", &rx[n], &gx[n], &bx[n], c);
    }

/* open ( and activate ) workstation */

    if (color) {
	for (n = 0; n < MAXCLI; n++) {
	    m = n % ncolor;
	    rgb[n] = rx[m] * 15 / RGBMAX * 256
                   + gx[m] * 15 / RGBMAX *  16
                   + bx[m] * 15 / RGBMAX;
	    r[n] = rx[m] / 256;
	    g[n] = gx[m] / 256;
	    b[n] = bx[m] / 256;
	}
    }
}

void z1dcls_()
{
    if (!separate)
	z1fcls();
}

/*------------------------- page --------------------------*/

void z1popn_()
{
    page++;
    if (separate || page==1)
        z1fopn();

    fprintf (outf,"%%%%Page: %d %d\n", page, page);
    fputs ("gsave\n", outf);
    fputs ("[ 0.0 0.1 -0.1 0.0 562 44 ] concat\n", outf);
    fputs ("1 setlinecap\n", outf);
    fputs ("1 setlinejoin\n", outf);
    fputs ("10 setmiterlimit\n", outf);
    iwdidy = 0;
    if (color) {
        fputs ("%%BeginObject: background\n", outf);
	fputs ("N\n", outf);
	fprintf (outf, "%4.0f %4.0f M\n", wsxmnz, wsymnz);
	fprintf (outf, "%4.0f %4.0f L\n", wsxmxz, wsymnz);
	fprintf (outf, "%4.0f %4.0f L\n", wsxmxz, wsymxz);
	fprintf (outf, "%4.0f %4.0f L\n", wsxmnz, wsymxz);
	fputs ("C\n", outf);
	fprintf (outf, "%4i RGB\n", rgb[0]);
	fputs ("F\n", outf);
	iclidy = 0;
	fputs ("%%EndObject\n", outf);
    }
    itbak = 0;
    fputs ("%%BeginObject: grph1\n", outf);
}

void z1pcls_()
{
    char comline[128];

/* close page ( or screen ) */

    fputs ("%%EndObject\n", outf);
    fputs ("grestore\n", outf);
    fputs ("showpage\n", outf);

    if (separate) {
        z1fcls();
	if (print) {
	    sprintf (comline, "%s %s", command, fname);
	    system (comline);
	    sprintf (comline, "rm %s", fname);
	    system (comline);
	}
    }
}

/*------------------------- object ------------------------*/

void z1oopn_(objname, comment, Lobj, Lcom)
    char *objname, *comment;
    int Lobj, Lcom;
{
    char obj[32], com[80];
    int nc;

    void cfnchr();

    cfnchr (obj, objname, Lobj);
    cfnchr (com, comment, Lcom);

    fprintf (outf, "%%%%Begin %s [%s]\n", obj, com);
}

void z1ocls_(objname, Lobj)
    char *objname;
    int Lobj;
{
    char obj[32];

    void cfnchr();

    cfnchr (obj, objname, Lobj);

    fprintf (outf, "%%%%End %s\n", obj);
}

/*------------------------- line --------------------------*/

void z1swdi_(iwdidx)
    int *iwdidx;
{

/* set line width index */

    iwdidz = *iwdidx % 10;
    if (iwdidz == 0)
	iwdidz = 1;
}

void z1scli_(iclidx)
    int *iclidx;
{

/* set line color index */

    iclidz = *iclidx % MAXCLI;
/*  if (iclidz == 0)
	iclidz = 1; */
}

void z1gopn_()
{

/* open graphic segment */

    np = 0;
    if (iwdidz != iwdidy) {
	fprintf (outf, "%4.1f W\n", xlwdth[iwdidz-1]);
	iwdidy = iwdidz;
    }
    if (color && iclidz != iclidy) {
	fprintf (outf, "%4i RGB\n", rgb[iclidz]);
	iclidy = iclidz;
    }
    fputs ("N\n", outf);
}

void z1gmov_(wx, wy)
    float *wx, *wy;
{

/* pen-up move */

    np = np + 1;
    fprintf (outf, "%4.0f %4.0f M\n", *wx, *wy);
}

void z1gplt_(wx, wy)
    float *wx, *wy;
{

/* pen-down move */

    np = np + 1;
    fprintf (outf, "%4.0f %4.0f L\n", *wx, *wy);

    if (np >= MAXNP) {
	fputs ("S\n", outf);
	np = 0;
	fputs ("N\n", outf);
	fprintf (outf, "%4.0f %4.0f M\n", *wx, *wy);
    }
}

void z1gcls_()
{

/* close graphic segment */

    fputs ("S\n", outf);
}

/*------------------------- tone --------------------------*/

void z1gton_(np, wpx, wpy, itpat)
    int *np, *itpat;
    float wpx[], wpy[];
{
    int i, j, k, jtpat, ltpat, jtclr, lbeta, it, iz1;
    float rz1;
    double r;

/* hard fill */

    jtpat = *itpat % 1000;
    jtclr = (*itpat / 1000) % MAXCLI;
/*  if (jtclr == 0)
	jtclr = 1; */

    i = jtpat / 100;
    lbeta = jtpat == 999;

    if (MAXT < i && !lbeta) {
        fprintf (stderr, "*** Error in z1pack : ");
        fprintf (stderr, "Tone pattern no. %d is not supported.\n", jtpat);
        return;
    }

    fputs ("N\n", outf);
    fprintf (outf, "%4.0f %4.0f M\n", wpx[0], wpy[0]);
    for (it = 1; it < *np; ++it)
	fprintf (outf, "%4.0f %4.0f L\n", wpx[it], wpy[it]);
    fputs ("C\n", outf);

    fprintf (outf, "%% %3i PAT\n", jtpat);

    if (color && lbeta) {
        if (itbak != -1) {
	    fputs ("R0\n", outf);
	    itbak = -1;
        }
	fprintf (outf, "%4i RGB\n", rgb[jtclr]);
    }
    else {
	if (lbeta)
	    jtpat = 9;
	ltpat = jtpat != itbak;
	j = (jtpat - i * 100) / 10;
	k = jtpat - i * 100 - j * 10;
	if (iwtroz != 1 && 1 <= i && i <= 4)
	    i = (i + 1) % 4 + 1;
	if (k == 0)
	    return;

	if (i == 0) {
	    r = 1.0 - pow (2.0, (double) (k - 9)) * itd[j];
	    r = floor (r * RFACT + 0.5) / RFACT;
	    if (r <= 0) {
		fputs ("0 G\n", outf);
            }
	    else {
		if (ltpat) {
		    rz1 = pow (2.0, (double) (k - 1) / 2.0) * A0;
		    iz1 = ((k + 1) % 2) * 45;
		    fprintf (outf,"%6.2f %2i P0 R ", rz1, iz1);
		    itbak = jtpat;
		}
		fprintf (outf, "%5.3f G\n", r);
	    }
	}
	else {
	    r = 1.0 - pow (2.0, (double) (k - 5)) * itl[j];
	    r = floor (r * RFACT + 0.5) / RFACT;
	    if (r <= 0) {
		fputs ("0 G\n", outf);
	    }
	    else {
		if (ltpat) {
		    rz1 = pow ((double) 2, (double) (k - 1)) * A0;
		    fprintf (outf,"%6.2f 0 P%1i R ", rz1, i);
		    itbak = jtpat;
		}
		fprintf (outf, "%5.3f G\n", r);
	    }
	}
    }

    fputs ("F\n", outf);

    if (!(color && lbeta))
        fputs ("0 G\n", outf);
    iclidy = 0;
}

/*------------------------- image -------------------------*/

void z1clip_(wx1, wy1, wx2, wy2, wx3, wy3, wx4, wy4)
    float *wx1, *wy1, *wx2, *wy2, *wx3, *wy3, *wx4, *wy4;
{
    fprintf (outf, "gsave\n");
    fprintf (outf, "%4.0f %4.0f M\n", *wx1, *wy1);
    fprintf (outf, "%4.0f %4.0f L\n", *wx2, *wy2);
    fprintf (outf, "%4.0f %4.0f L\n", *wx3, *wy3);
    fprintf (outf, "%4.0f %4.0f L\n", *wx4, *wy4);
    fprintf (outf, "%4.0f %4.0f L\n", *wx1, *wy1);
    fprintf (outf, "C clip\n");
}

void z1iopn_(iwx, iwy, iwidth, iheight)
    int *iwx, *iwy, *iwidth, *iheight;
{
    fprintf (outf, "/picstr %d string def \n", *iwidth);
    fprintf (outf, "%d %d 8 \n", *iwidth, *iheight);
    fprintf (outf, "[%f 0 0 %f %f %f]\n",
       1/scale, -1/scale, -wsxmnz/scale-*iwx+0.5, wsymxz/scale-*iwy+0.5);
    fputs ("{currentfile picstr readhexstring pop}\n", outf);
    fputs ("false 3 colorimage\n", outf);

    imlen = *iwidth * *iheight;
    ipxcl = 0;
}

void z1idat_(image, nlen)
    int *nlen, image[];
{
    int i, ip;

    for (i = 0; i < *nlen; i++) {
	ipxcl++;
	ip = image[i];
	fprintf (outf, "%2.2x%2.2x%2.2x", r[ip], g[ip], b[ip]);
	if ((ipxcl % 10) == 0)
	    fputs("\n", outf);
    }
}

void z1icls_()
{
    fputs("grestore\n", outf);
    fputs("\n", outf);
}

/*------------------------- mouse -------------------------*/

void z1qpnt_(wx, wy, mb)
    float *wx, *wy;
    int *mb;
{
    *mb = 0;
}

/*---------------------- transformation -------------------*/

void z1fint_(wx, wy, iwx, iwy)
    float *wx,  *wy;
    int *iwx, *iwy;
{
    *iwx = (*wx - wsxmnz) / scale + 0.5;
    *iwy = (wsymxz - *wy) / scale + 0.5;
}

void z1iint_(iwx, iwy, wx, wy)
    float *wx,  *wy;
    int *iwx, *iwy;
{
    *wx = wsxmnz + *iwx*scale;
    *wy = wsymxz - *iwy*scale;
}

/*------------------------- inquiry -----------------------*/

void z1qwdc_(lwdatr)
    int *lwdatr;
{

/* inquire line width capability */

    *lwdatr = LWDATR;
}

void z1qclc_(lclatr)
    int *lclatr;
{

/* inquire line color capability */

    *lclatr = color;
}

void z1qtnc_(ltnatr)
    int *ltnatr;
{

/* inquire hard fill capability */

    *ltnatr = LTNATR;
}

void z1qimc_(limatr)
    int *limatr;
{

/* inquire bit image capability */

    *limatr = LIMATR;
}

void z1qptc_(lptatr)
    int *lptatr;
{

/* inquire mouse point capability */

    *lptatr = LPTATR;
}

void z1qrct_(wsxmn, wsxmx, wsymn, wsymx, fact)
    float *wsxmn, *wsxmx, *wsymn, *wsymx, *fact;
{

/* inquire workstation rectangle */

    *wsxmn = wsxmnz;
    *wsxmx = wsxmxz;
    *wsymn = wsymnz;
    *wsymx = wsymxz;
    *fact  = factz;
}

void z1srot_(iwtrot)
    int *iwtrot;
{

/* set frame rotation flag */

    iwtroz = *iwtrot;
}

/*---------- file handling (internal functions) -----------*/

static void z1fopn()
{
    if (!separate) {
	sprintf (fname, "%s.ps", psfile);
    }
    else {
	sprintf (fname, "%s_%03d.ps", psfile, page);
    }
    if ((outf = fopen(fname, "w+")) == NULL) {
	fprintf (stderr, "*** Error in z1pack : ");
	fprintf (stderr, "File < %s > cannot be opened.\n", fname);
	exit (1);
    }

    fputs ("%!PS-Adobe-2.0\n", outf);
    fprintf (outf, "%%%%Creator: %s\n", xtitle);
    fprintf (outf, "%%%%Title: %s\n", fname);
    fputs ("%%Pages: (atend)\n", outf);
    fputs ("%%BoundingBox: 31 43 563 795\n", outf);
    fputs ("%%DocumentPaperSizes: A4\n", outf);
    if (iwtroz != 1)
	fputs ("%%Orientation: Portrait\n",  outf);
    else
	fputs ("%%Orientation: Landscape\n", outf);
    fputs ("%%EndComments\n", outf);
    fputs ("%%BeginProlog\n", outf);
    fputs ("/W {setlinewidth} def\n", outf);
    fputs ("/N {newpath} def\n", outf);
    fputs ("/M {moveto} def\n", outf);
    fputs ("/L {lineto} def\n", outf);
    fputs ("/S {stroke} def\n", outf);
    fputs ("/C {closepath} def\n", outf);
    fputs ("/R {setscreen} def\n", outf);
    fputs ("/G {setgray} def\n", outf);
    fputs ("/F {fill} def\n", outf);
    fputs ("/max {2 copy lt {exch} if pop} def\n", outf);
    fputs ("/rightup   {sub abs -1 add abs} def\n", outf);
    fputs ("/rightdown {add abs -1 add abs} def\n", outf);
    fputs ("/P0 {{180 mul cos exch 180 mul cos add 2 div}} def\n", outf);
    fputs ("/P1 {{pop}} def\n", outf);
    fputs ("/P2 {{rightup}} def\n", outf);
    fputs ("/P3 {{exch pop}} def\n", outf);
    fputs ("/P4 {{rightdown}} def\n", outf);
    fputs ("/P5 {{max}} def\n", outf);
    fputs ("/P6 {{2 copy rightup 3 1 roll rightdown max}} def\n", outf);
    fputs ("currentscreen /dot exch def /ang exch def /freq exch def\n", outf);
    fputs ("/R0 {freq ang //dot setscreen} def\n", outf);
    if (color) {
	fputs ("/RGB {dup 256 idiv 15 div /r exch def\n", outf);
	fputs ("      dup 256 mod 16 idiv 15 div /g exch def\n", outf);
	fputs ("      16 mod 15 div /b exch def\n", outf);
	fputs ("      r g b setrgbcolor} def\n", outf);
    }
    fputs ("%%EndProlog\n", outf);
}

static void z1fcls()
{
    fputs ("%%Trailer\n", outf);
    fprintf (outf, "%%%%Pages: %d\n", page);
    fputs ("%%EOF\n", outf);
    fclose (outf);
}
