!= Sample program for gtool_historyauto/gtool5
!
! * 2010/08/26 Y.Morikawa
! * 2009/10/18 Y.Morikawa
! * 2003/08/21 M.Odaka
! * 2001/02/27 S.Takehiro
!
! Solving diffusion equation
! \[
!     du/dt = \kappa d^2 u/dx^2
! \]
! for giving values of $u$ at $x=[0,1]$.
!
program diffusion_cal2

  use gtool_historyauto                       ! Access module (⥸塼)
  use dc_calendar                             ! Access module (⥸塼)
  use dc_types                                ! Access module (⥸塼)

  integer, parameter     :: nx=30             ! Grid number   (åɿ)
  real(DP), parameter    :: dx=1.0/(nx-1)     ! Grid interval (åɴֳ)
  type(DC_CAL)         :: cal_gre             ! Gregorian Calendar (쥴ꥪ)
  type(DC_CAL)         :: cal_mars            ! Martian Calendar   (äݤ)
  type(DC_CAL_DATE)    :: start_date          ! Start Date (ǥ볫)
  type(DC_CAL_DATE)    :: end_date            ! End Date   (ǥ뽪λ)
  real(DP)             :: dt                  ! Time step    [sec] (ֳִ [])
  real(DP)             :: curtime             ! Current time [sec] (в [])
  real(DP)             :: endtime             ! End time     [sec] (λ [])
  real(DP)             :: dsptime             ! Output interval [day] (ϴֳ [])

  real(DP)             :: sec_of_year         ! Second of year (ǯϤᤫ)
  real(DP)             :: day_of_year         ! Day of year    (ǯϤᤫ)
  real(DP)             :: sec_of_day          ! Second of day  (Ϥᤫ)
  logical              :: flag_leapyear       ! Leap year      (ǯ)

  real(DP), dimension(nx):: x=(/(dx*(i-1),i=1,nx)/) ! X coordinate (ɸѿ)
  real(DP), dimension(nx):: temp                    ! Temperature ()
  real(DP), parameter    :: kappa=1.0e-9            ! Diffusion coefficient (ǮȻ)

  character(256):: date_str1, date_str2

  character(32):: cal_type
  integer:: month_in_year, day_in_month(1:12), hour_in_day, min_in_hour 
  integer, pointer:: day_in_month_ptr(:) =>null()
  real(DP):: sec_in_min

  call DCCalCreate( cal_type = 'Gregorian', cal = cal_gre )
  call DCCalCreate( month_in_year = 1 , &
    &               day_in_month  = (/669/), &
    &               hour_in_day   = 24, &
    &               min_in_hour   = 1 , &
    &               sec_in_min    = 3694.0_DP, &
    &               cal           = cal_mars )
                                                    ! Set calendar
                                                    ! ()

  call DCCalDateCreate( year = 2000, month =  1, day = 1, &
    &                   hour =    0, min   =  0, sec = 0.0_DP, &
    &                   date = start_date )
  call DCCalDateCreate( year = 2001, month =  1, day = 1, &
    &                   hour =    0, min   =  0, sec = 0.0_DP, &
    &                   date = end_date )
                                                    ! Set date
                                                    ! ()

  curtime = DCCalConvertByUnit(  0.0_DP, 'sec', 'sec', cal = cal_mars )
                                                    ! Set initial time 
                                                    ! ()
  endtime = DCCalDateDifference( start_date, end_date, cal = cal_mars )
                                                    ! Set end time 
                                                    ! (λ)
  dt      = DCCalConvertByUnit(  1.0_DP, 'day', 'sec', cal = cal_mars )
                                                    ! Set time step
                                                    ! (֥ƥå)
  dsptime = 10
                                                    ! Set output interval
                                                    ! (ϴֳ)

  temp = exp(-((x-0.5)/0.1)**2)                     ! Set initial value 
                                                    ! ()

  call HistoryAutoCreate( &                         ! Create output file 
    & title='Diffusion equation',                &  ! (ҥȥ꡼)
    & source='Sample program of gtool_historyauto/gtool5',   &
    & institution='GFD_Dennou Club davis project',           &
    & dims=(/'x','t'/), dimsizes=(/nx,0/),                   &
    & longnames=(/'X-coordinate','time        '/),           &
    & units=(/'m  ','day'/),                                 &
    & origin=curtime, terminus=endtime, interval=dsptime,    &
    & cal=cal_mars, start_date=start_date )

  call HistoryAutoPutAxis('x',x)                    ! Output 'x' (ѿ)

  call HistoryAutoAddVariable( &                    ! Set output variable 
    & varname='temp', dims=(/'x','t'/), &           ! (ѿ) 
    & longname='temperature', units='K', xtype='double', &
    & file='diffusion_cal2.nc' )

  call HistoryAutoPut(curtime,'temp',temp)          ! Output 'temp' (ѿ)

  write(6,*) "Calendars: "

  call DCCalInquire( cal_type, &
    &                month_in_year    = month_in_year, &
    &                day_in_month_ptr = day_in_month_ptr, &
    &                hour_in_day      = hour_in_day, &
    &                min_in_hour      = min_in_hour, &
    &                sec_in_min       = sec_in_min, &
    &                cal              = cal_mars )
                                        ! Inquire calendar
                                        ! (˴ؤ䤤碌)

  write(6,*) "       Martian Cal:   month in year:  ", month_in_year
  write(6,*) "                      days in months: ", day_in_month_ptr
  write(6,*) "                      hour in day:    ", hour_in_day
  write(6,*) "                      min in hour:    ", min_in_hour
  write(6,*) "                      sec in min:     ", sec_in_min
  deallocate(day_in_month_ptr)

  call DCCalInquire( cal_type, &
    &                month_in_year    = month_in_year, &
    &                day_in_month     = day_in_month, &
    &                hour_in_day      = hour_in_day, &
    &                min_in_hour      = min_in_hour, &
    &                sec_in_min       = sec_in_min, &
    &                cal              = cal_gre )
                                        ! Inquire calendar
                                        ! (˴ؤ䤤碌)

  write(6,*) "       Gregorian Cal: month in year:  ", month_in_year
  write(6,*) "                      days in months: ", day_in_month
  write(6,*) "                      hour in day:    ", hour_in_day
  write(6,*) "                      min in hour:    ", min_in_hour
  write(6,*) "                      sec in min:     ", sec_in_min

  write(6,*) ""

  call DCCalDateInquire( date_str1, date = start_date )
                                        ! Inquire date and set a character variable
                                        ! (䤤碌ʸѿ)

  write(6,*) "Start: ", trim(date_str1)
                                                    ! Display start date
                                                    ! (ɽ) 

  sec_of_year   = DCCalDateEvalSecOfYear( curtime, start_date, cal_mars )
                                        ! Second of year (ǯϤᤫ)
  day_of_year   = DCCalDateEvalDayOfYear( curtime, start_date, cal_mars )
                                        ! Day of year    (ǯϤᤫ)
  sec_of_day    = DCCalDateEvalSecOfDay( curtime, start_date, cal_mars )
                                        ! Second of day  (Ϥᤫ)
  flag_leapyear = DCCalDateChkLeapYear( curtime, start_date, cal_mars )
                                        ! Leap year      (ǯ)

  write(6,*) "       Martian Cal:   second of year: ", sec_of_year
  write(6,*) "                      day of year:    ", day_of_year
  write(6,*) "                      second of day:  ", sec_of_day
  write(6,*) "                      leap year:      ", flag_leapyear

  sec_of_year   = DCCalDateEvalSecOfYear( curtime, start_date, cal_gre )
                                        ! Second of year (ǯϤᤫ)
  day_of_year   = DCCalDateEvalDayOfYear( curtime, start_date, cal_gre )
                                        ! Day of year    (ǯϤᤫ)
  sec_of_day    = DCCalDateEvalSecOfDay( curtime, start_date, cal_gre )
                                        ! Second of day  (Ϥᤫ)
  flag_leapyear = DCCalDateChkLeapYear( curtime, start_date, cal_gre )
                                        ! Leap year      (ǯ)

  write(6,*) "       Gregorian Cal: second of year: ", sec_of_year
  write(6,*) "                      day of year:    ", day_of_year
  write(6,*) "                      second of day:  ", sec_of_day
  write(6,*) "                      leap year:      ", flag_leapyear


  do while ( curtime < endtime )                    ! Check termination (λȽ)

    temp(2:nx-1) = temp(2:nx-1) &                   ! Time integration (ʬ)
      & + kappa*(temp(3:nx)-2*temp(2:nx-1)+temp(1:nx-2))/dx**2*dt

    call HistoryAutoPut(curtime,'temp',temp)             ! Output 'temp' (ѿ)

    curtime = curtime + dt                          ! Progress model time (ǥʹ)
  enddo


  call DCCalDateInquire( date_str1, curtime, start_date, cal = cal_mars )
  call DCCalDateInquire( date_str2, curtime, start_date, cal = cal_gre )
                                        ! Inquire date and set a character variable
                                        ! (䤤碌ʸѿ)

  write(6,*) "End:   ", trim(date_str1), " (Mars), ", trim(date_str2), " (Gregorian)"
                                                    ! Display end date (λ) 

  sec_of_year   = DCCalDateEvalSecOfYear( curtime, start_date, cal_mars )
                                        ! Second of year (ǯϤᤫ)
  day_of_year   = DCCalDateEvalDayOfYear( curtime, start_date, cal_mars )
                                        ! Day of year    (ǯϤᤫ)
  sec_of_day    = DCCalDateEvalSecOfDay( curtime, start_date, cal_mars )
                                        ! Second of day  (Ϥᤫ)
  flag_leapyear = DCCalDateChkLeapYear( curtime, start_date, cal_mars )
                                        ! Leap year      (ǯ)

  write(6,*) "       Martian Cal:   second of year: ", sec_of_year
  write(6,*) "                      day of year:    ", day_of_year
  write(6,*) "                      second of day:  ", sec_of_day
  write(6,*) "                      leap year:      ", flag_leapyear

  sec_of_year   = DCCalDateEvalSecOfYear( curtime, start_date, cal_gre )
                                        ! Second of year (ǯϤᤫ)
  day_of_year   = DCCalDateEvalDayOfYear( curtime, start_date, cal_gre )
                                        ! Day of year    (ǯϤᤫ)
  sec_of_day    = DCCalDateEvalSecOfDay( curtime, start_date, cal_gre )
                                        ! Second of day  (Ϥᤫ)
  flag_leapyear = DCCalDateChkLeapYear( curtime, start_date, cal_gre )
                                        ! Leap year      (ǯ)

  write(6,*) "       Gregorian Cal: second of year: ", sec_of_year
  write(6,*) "                      day of year:    ", day_of_year
  write(6,*) "                      second of day:  ", sec_of_day
  write(6,*) "                      leap year:      ", flag_leapyear

  call HistoryAutoClose
  stop
end program diffusion_cal2
