module time_scheme
  use force_nbm_solv
  use force_sbl_solv

contains

subroutine time_integration( it, psiko, uko_sbl, vko_sbl,  &
  &                          psikn, ukn_sbl, vkn_sbl, pk,  &
  &                          nbm_opt, sbl_optu, sbl_optv )
!-- calculating time integration NBM and SBL, simultaneously
  use savegloval_define
  use Math_Const

  implicit none
  integer, intent(in) :: it                               ! time step
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: psiko ! psik of zo
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: uko_sbl ! uk of zo
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: vko_sbl ! vk of zo
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out) :: psikn   ! psik of zn
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out) :: ukn_sbl   ! uk of zn
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out) :: vkn_sbl   ! vk of zn
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: pk     ! pk
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout), optional :: nbm_opt
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout), optional :: sbl_optu
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout), optional :: sbl_optv
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1

  integer :: i, j, ct
  complex(kind(0d0)), dimension(kxnt,kynt) :: psikt, forcea_zeta, force_zeta
  complex(kind(0d0)), dimension(kxnt,kynt) :: ukt_sbl, vkt_sbl
  complex(kind(0d0)), dimension(kxnt,kynt) :: forcea_uk, force_uk
  complex(kind(0d0)), dimension(kxnt,kynt) :: forcea_vk, force_vk
  double precision, dimension(jynt,jxnt) :: u_nbm_isp  ! u for ISPACK
  double precision, dimension(jynt,jxnt) :: v_nbm_isp  ! v for ISPACK

  psikt=psiko
  ukt_sbl=uko_sbl
  vkt_sbl=vko_sbl
  u_nbm_isp=0.0d0
  v_nbm_isp=0.0d0
  pk=0.0d0
  force_uk=0.0d0
  force_vk=0.0d0
  forcea_uk=0.0d0
  forcea_vk=0.0d0
  force_zeta=0.0d0
  forcea_zeta=0.0d0

  select case (time_flag(1:3))

  case ('EUL')  ! Euler scheme

     call force_NBM( psiko, force_zeta,  &
  &                  pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )

     call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                  force_uk, force_vk )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,2*hynt+1
        do i=1,hxnt+1
           psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
           ukn_sbl(i,j)=uko_sbl(i,j)+dt*force_uk(i,j)
           vkn_sbl(i,j)=vko_sbl(i,j)+dt*force_vk(i,j)
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
           ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
           ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
           vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
           vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

     call smooth_val_SBL( ukn_sbl )
     call smooth_val_SBL( vkn_sbl )

  case ('RK3')  ! 3rd order Runge-Kutta scheme

     do ct=1,3

        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )
        call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                     force_uk, force_vk )

!$omp parallel default(shared)
        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 forcea_uk(i,j)=forcea_uk(i,j)+force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+force_vk(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
                 uko_sbl(i,j)=ukt_sbl(i,j)+(0.5d0*dt)*force_uk(i,j)
                 vko_sbl(i,j)=vkt_sbl(i,j)+(0.5d0*dt)*force_vk(i,j)
              end do
           end do
!$omp end do

        case (2)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 psiko(i,j)=psikt(i,j)+dt*(-forcea_zeta(i,j)+2.0d0*force_zeta(i,j))
                 uko_sbl(i,j)=ukt_sbl(i,j)+dt*(-forcea_uk(i,j)+2.0d0*force_uk(i,j))
                 vko_sbl(i,j)=vkt_sbl(i,j)+dt*(-forcea_vk(i,j)+2.0d0*force_vk(i,j))
                 forcea_zeta(i,j)=forcea_zeta(i,j)+4.0d0*force_zeta(i,j)
                 forcea_uk(i,j)=forcea_uk(i,j)+4.0d0*force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+4.0d0*force_vk(i,j)
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 forcea_uk(i,j)=forcea_uk(i,j)+force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+force_vk(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
                 ukn_sbl(i,j)=ukt_sbl(i,j)+(dt/6.0d0)*forcea_uk(i,j)
                 vkn_sbl(i,j)=vkt_sbl(i,j)+(dt/6.0d0)*forcea_vk(i,j)
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psiko(kxnt-i+2,kynt-j+2)=dconjg(psiko(i,j))
                 psiko(kxnt-i+2,j)=dconjg(psiko(i,kynt-j+2))
                 uko_sbl(kxnt-i+2,kynt-j+2)=dconjg(uko_sbl(i,j))
                 vko_sbl(kxnt-i+2,kynt-j+2)=dconjg(vko_sbl(i,j))
                 uko_sbl(kxnt-i+2,j)=dconjg(uko_sbl(i,kynt-j+2))
                 vko_sbl(kxnt-i+2,j)=dconjg(vko_sbl(i,kynt-j+2))
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
                 psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
                 ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
                 vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
                 ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
                 vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
              end do
           end do
!$omp end do

        end select

!$omp end parallel

        call smooth_val_SBL( ukn_sbl )
        call smooth_val_SBL( vkn_sbl )

     end do

  case ('RK4')  ! 4th order Runge-Kutta scheme

     do ct=1,4

        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )
        call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                     force_uk, force_vk )

!$omp parallel default(shared)

        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 forcea_uk(i,j)=forcea_uk(i,j)+force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+force_vk(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
                 uko_sbl(i,j)=ukt_sbl(i,j)+(0.5d0*dt)*force_uk(i,j)
                 vko_sbl(i,j)=vkt_sbl(i,j)+(0.5d0*dt)*force_vk(i,j)
              end do
           end do
!$omp end do

        case (2)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 forcea_uk(i,j)=forcea_uk(i,j)+2.0d0*force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+2.0d0*force_vk(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
                 uko_sbl(i,j)=ukt_sbl(i,j)+(0.5d0*dt)*force_uk(i,j)
                 vko_sbl(i,j)=vkt_sbl(i,j)+(0.5d0*dt)*force_vk(i,j)
              end do
           end do
!$omp end do

        case (3)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 forcea_uk(i,j)=forcea_uk(i,j)+2.0d0*force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+2.0d0*force_vk(i,j)
                 psiko(i,j)=psikt(i,j)+(dt)*force_zeta(i,j)
                 uko_sbl(i,j)=ukt_sbl(i,j)+(dt)*force_uk(i,j)
                 vko_sbl(i,j)=vkt_sbl(i,j)+(dt)*force_vk(i,j)
              end do
           end do
!$omp end do

        case (4)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 forcea_uk(i,j)=forcea_uk(i,j)+force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+force_vk(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
                 ukn_sbl(i,j)=ukt_sbl(i,j)+(dt/6.0d0)*forcea_uk(i,j)
                 vkn_sbl(i,j)=vkt_sbl(i,j)+(dt/6.0d0)*forcea_vk(i,j)
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2,3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psiko(kxnt-i+2,kynt-j+2)=dconjg(psiko(i,j))
                 psiko(kxnt-i+2,j)=dconjg(psiko(i,kynt-j+2))
                 uko_sbl(kxnt-i+2,kynt-j+2)=dconjg(uko_sbl(i,j))
                 vko_sbl(kxnt-i+2,kynt-j+2)=dconjg(vko_sbl(i,j))
                 uko_sbl(kxnt-i+2,j)=dconjg(uko_sbl(i,kynt-j+2))
                 vko_sbl(kxnt-i+2,j)=dconjg(vko_sbl(i,kynt-j+2))
              end do
           end do
!$omp end do

        case (4)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
                 psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
                 ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
                 vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
                 ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
                 vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
              end do
           end do
!$omp end do

        end select

!$omp end parallel

        call smooth_val_SBL( ukn_sbl )
        call smooth_val_SBL( vkn_sbl )

     end do

  case ('AB2')  ! 2nd order Adams-Bashforse scheme

     call force_NBM( psiko, force_zeta,  &
  &                  pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )
     call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                  force_uk, force_vk )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              ukn_sbl(i,j)=uko_sbl(i,j)+dt*force_uk(i,j)
              vkn_sbl(i,j)=vko_sbl(i,j)+dt*force_vk(i,j)
              nbm_opt(i,j)=force_zeta(i,j)
              sbl_optu(i,j)=force_uk(i,j)
              sbl_optv(i,j)=force_vk(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*(1.5d0*force_zeta(i,j)-0.5d0*nbm_opt(i,j))
              ukn_sbl(i,j)=uko_sbl(i,j)+dt*(1.5d0*force_uk(i,j)-0.5d0*sbl_optu(i,j))
              vkn_sbl(i,j)=vko_sbl(i,j)+dt*(1.5d0*force_vk(i,j)-0.5d0*sbl_optv(i,j))
              nbm_opt(i,j)=force_zeta(i,j)
              sbl_optu(i,j)=force_uk(i,j)
              sbl_optv(i,j)=force_vk(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
           ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
           vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
           ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
           vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

     call smooth_val_SBL( ukn_sbl )
     call smooth_val_SBL( vkn_sbl )

  case ('L-F')  ! Leap Frog scheme

     call force_NBM( psiko, force_zeta,  &
  &                  pk=pk, u_isp=u_nbm_isp, v_isp=v_nbm_isp )
     call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                  force_uk, force_vk )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              ukn_sbl(i,j)=uko_sbl(i,j)+dt*force_uk(i,j)
              vkn_sbl(i,j)=vko_sbl(i,j)+dt*force_vk(i,j)
              nbm_opt(i,j)=psiko(i,j)
              sbl_optu(i,j)=uko_sbl(i,j)
              sbl_optv(i,j)=vko_sbl(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=nbm_opt(i,j)+2.0d0*dt*force_zeta(i,j)
              ukn_sbl(i,j)=sbl_optu(i,j)+2.0d0*dt*force_uk(i,j)
              vkn_sbl(i,j)=sbl_optv(i,j)+2.0d0*dt*force_vk(i,j)
              nbm_opt(i,j)=psiko(i,j)
              sbl_optu(i,j)=uko_sbl(i,j)
              sbl_optv(i,j)=vko_sbl(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
           ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
           vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
           ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
           vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

     call smooth_val_SBL( ukn_sbl )
     call smooth_val_SBL( vkn_sbl )

!  case ('C-N')  ! Crank-Nicolson scheme

!  case ('BEU')  ! Backward Euler scheme

  end select

  psiko=psikt
  uko_sbl=ukt_sbl
  vko_sbl=vkt_sbl

end subroutine time_integration


subroutine time_integration_NBM( it, psiko, psikn, nbm_opt, pk, u_isp, v_isp )
!-- calculating time integration
  use savegloval_define
  use Math_Const

  implicit none
  integer, intent(in) :: it                               ! time step
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: psiko ! psi of zo
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out) :: psikn   ! psi of zn
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout), optional :: nbm_opt
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out), optional :: pk   ! p
  double precision, dimension(jynt,jxnt), intent(out), optional :: u_isp  ! u for ISPACK
  double precision, dimension(jynt,jxnt), intent(out), optional :: v_isp  ! v for ISPACK

  integer :: i, j, ct
  complex(kind(0d0)), dimension(kxnt,kynt) :: psikt, forcea_zeta, force_zeta

  psikt=psiko
  force_zeta=0.0d0
  forcea_zeta=0.0d0

  select case (time_flag(1:3))

  case ('EUL')  ! Euler scheme

     if(present(pk))then
        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_isp, v_isp=v_isp )
     else
        call force_NBM( psiko, force_zeta )
     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,2*hynt+1
        do i=1,hxnt+1
           psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

  case ('RK3')  ! 3rd order Runge-Kutta scheme

     do ct=1,3

        if(present(pk))then
           call force_NBM( psiko, force_zeta,  &
  &                        pk=pk, u_isp=u_isp, v_isp=v_isp )
        else
           call force_NBM( psiko, force_zeta )
        end if

!$omp parallel default(shared)
        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (2)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 psiko(i,j)=psikt(i,j)+dt*(-forcea_zeta(i,j)+2.0d0*force_zeta(i,j))
                 forcea_zeta(i,j)=forcea_zeta(i,j)+4.0d0*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psiko(kxnt-i+2,kynt-j+2)=dconjg(psiko(i,j))
                 psiko(kxnt-i+2,j)=dconjg(psiko(i,kynt-j+2))
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
                 psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
              end do
           end do
!$omp end do

        end select

!$omp end parallel

     end do

  case ('RK4')  ! 4th order Runge-Kutta scheme

     do ct=1,4

        if(present(pk))then
           call force_NBM( psiko, force_zeta,  &
  &                        pk=pk, u_isp=u_isp, v_isp=v_isp )
        else
           call force_NBM( psiko, force_zeta )
        end if

!$omp parallel default(shared)

        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (2)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (3)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 psiko(i,j)=psikt(i,j)+(dt)*force_zeta(i,j)
              end do
           end do
!$omp end do

        case (4)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 psikn(i,j)=psikt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2,3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psiko(kxnt-i+2,kynt-j+2)=dconjg(psiko(i,j))
                 psiko(kxnt-i+2,j)=dconjg(psiko(i,kynt-j+2))
              end do
           end do
!$omp end do

        case (4)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
                 psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
              end do
           end do
!$omp end do

        end select

!$omp end parallel

     end do

  case ('AB2')  ! 2nd order Adams-Bashforse scheme

     if(present(pk))then
        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_isp, v_isp=v_isp )
     else
        call force_NBM( psiko, force_zeta )
     end if

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              nbm_opt(i,j)=force_zeta(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*(1.5d0*force_zeta(i,j)-0.5d0*nbm_opt(i,j))
              nbm_opt(i,j)=force_zeta(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

  case ('L-F')  ! Leap Frog scheme

     if(present(pk))then
        call force_NBM( psiko, force_zeta,  &
  &                     pk=pk, u_isp=u_isp, v_isp=v_isp )
     else
        call force_NBM( psiko, force_zeta )
     end if

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=psiko(i,j)+dt*force_zeta(i,j)
              nbm_opt(i,j)=psiko(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              psikn(i,j)=nbm_opt(i,j)+2.0d0*dt*force_zeta(i,j)
              nbm_opt(i,j)=psiko(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           psikn(kxnt-i+2,kynt-j+2)=dconjg(psikn(i,j))
           psikn(kxnt-i+2,j)=dconjg(psikn(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

!  case ('C-N')  ! Crank-Nicolson scheme

!  case ('BEU')  ! Backward Euler scheme

  end select

  psiko=psikt

end subroutine time_integration_NBM

subroutine time_integration_SBL( it, pk, u_nbm_isp, v_nbm_isp,  &
  &                              uko_sbl, vko_sbl, ukn_sbl, vkn_sbl, sbl_optu, sbl_optv )
!-- calculating time integration
  use savegloval_define
  use Math_Const

  implicit none
  integer, intent(in) :: it                               ! time step
  complex(kind(0d0)), dimension(kxnt,kynt), intent(in) :: pk  ! pressure in NBM
  double precision, dimension(jynt,jxnt), intent(in)  :: u_nbm_isp  ! u for ISPACK
  double precision, dimension(jynt,jxnt), intent(in)  :: v_nbm_isp  ! v for ISPACK
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: uko_sbl ! u of zo
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout) :: vko_sbl ! v of zo
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out) :: ukn_sbl   ! u of zn
  complex(kind(0d0)), dimension(kxnt,kynt), intent(out) :: vkn_sbl   ! v of zn
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout), optional :: sbl_optu
  complex(kind(0d0)), dimension(kxnt,kynt), intent(inout), optional :: sbl_optv
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1
  integer :: i, j, ct
  complex(kind(0d0)), dimension(kxnt,kynt) :: ukt_sbl, vkt_sbl, forcea_uk, force_uk
  complex(kind(0d0)), dimension(kxnt,kynt) :: forcea_vk, force_vk

  ukt_sbl=uko_sbl
  vkt_sbl=vko_sbl
  force_uk=0.0d0
  force_vk=0.0d0
  forcea_uk=0.0d0
  forcea_vk=0.0d0

  select case (time_flag(1:3))

  case ('EUL')  ! Euler scheme

     call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                  force_uk, force_vk )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,2*hynt+1
        do i=1,hxnt+1
           ukn_sbl(i,j)=uko_sbl(i,j)+dt*force_uk(i,j)
           vkn_sbl(i,j)=vko_sbl(i,j)+dt*force_vk(i,j)
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
           ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
           vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
           vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

     call smooth_val_SBL( ukn_sbl )
     call smooth_val_SBL( vkn_sbl )

  case ('RK3')  ! 3rd order Runge-Kutta scheme

     do ct=1,3

        call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                     force_uk, force_vk )

!$omp parallel default(shared)

        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_uk(i,j)=forcea_uk(i,j)+force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+force_vk(i,j)
                 uko_sbl(i,j)=ukt_sbl(i,j)+(0.5d0*dt)*force_uk(i,j)
                 vko_sbl(i,j)=vkt_sbl(i,j)+(0.5d0*dt)*force_vk(i,j)
              end do
           end do
!$omp end do

        case (2)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 uko_sbl(i,j)=ukt_sbl(i,j)+dt*(-forcea_uk(i,j)+2.0d0*force_uk(i,j))
                 vko_sbl(i,j)=vkt_sbl(i,j)+dt*(-forcea_vk(i,j)+2.0d0*force_vk(i,j))
                 forcea_uk(i,j)=forcea_uk(i,j)+4.0d0*force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+4.0d0*force_vk(i,j)
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_uk(i,j)=forcea_uk(i,j)+force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+force_vk(i,j)
                 ukn_sbl(i,j)=ukt_sbl(i,j)+(dt/6.0d0)*forcea_uk(i,j)
                 vkn_sbl(i,j)=vkt_sbl(i,j)+(dt/6.0d0)*forcea_vk(i,j)
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 uko_sbl(kxnt-i+2,kynt-j+2)=dconjg(uko_sbl(i,j))
                 vko_sbl(kxnt-i+2,kynt-j+2)=dconjg(vko_sbl(i,j))
                 uko_sbl(kxnt-i+2,j)=dconjg(uko_sbl(i,kynt-j+2))
                 vko_sbl(kxnt-i+2,j)=dconjg(vko_sbl(i,kynt-j+2))
              end do
           end do
!$omp end do

        case (3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
                 vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
                 ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
                 vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
              end do
           end do
!$omp end do

        end select

!$omp end parallel

        call smooth_val_SBL( ukn_sbl )
        call smooth_val_SBL( vkn_sbl )

     end do

  case ('RK4')  ! 4th order Runge-Kutta scheme

     do ct=1,4

        call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                     force_uk, force_vk )

!$omp parallel default(shared)
        select case (ct)
        case (1)
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_uk(i,j)=forcea_uk(i,j)+force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+force_vk(i,j)
                 uko_sbl(i,j)=ukt_sbl(i,j)+(0.5d0*dt)*force_uk(i,j)
                 vko_sbl(i,j)=vkt_sbl(i,j)+(0.5d0*dt)*force_vk(i,j)
              end do
           end do
!$omp end do

        case (2)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_uk(i,j)=forcea_uk(i,j)+2.0d0*force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+2.0d0*force_vk(i,j)
                 uko_sbl(i,j)=ukt_sbl(i,j)+(0.5d0*dt)*force_uk(i,j)
                 vko_sbl(i,j)=vkt_sbl(i,j)+(0.5d0*dt)*force_vk(i,j)
              end do
           end do
!$omp end do

        case (3)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_uk(i,j)=forcea_uk(i,j)+2.0d0*force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+2.0d0*force_vk(i,j)
                 uko_sbl(i,j)=ukt_sbl(i,j)+(dt)*force_uk(i,j)
                 vko_sbl(i,j)=vkt_sbl(i,j)+(dt)*force_vk(i,j)
              end do
           end do
!$omp end do

        case (4)                         
!$omp do schedule(runtime) private(i,j)
           do j=1,2*hynt+1
              do i=1,hxnt+1
                 forcea_uk(i,j)=forcea_uk(i,j)+force_uk(i,j)
                 forcea_vk(i,j)=forcea_vk(i,j)+force_vk(i,j)
                 ukn_sbl(i,j)=ukt_sbl(i,j)+(dt/6.0d0)*forcea_uk(i,j)
                 vkn_sbl(i,j)=vkt_sbl(i,j)+(dt/6.0d0)*forcea_vk(i,j)
              end do
           end do
!$omp end do

        end select

!$omp barrier

        select case (ct)
        case (1,2,3)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 uko_sbl(kxnt-i+2,kynt-j+2)=dconjg(uko_sbl(i,j))
                 vko_sbl(kxnt-i+2,kynt-j+2)=dconjg(vko_sbl(i,j))
                 uko_sbl(kxnt-i+2,j)=dconjg(uko_sbl(i,kynt-j+2))
                 vko_sbl(kxnt-i+2,j)=dconjg(vko_sbl(i,kynt-j+2))
              end do
           end do
!$omp end do

        case (4)
!$omp do schedule(runtime) private(i,j)
           do j=2,hynt+1
              do i=2,hxnt+1
                 ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
                 vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
                 ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
                 vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
              end do
           end do
!$omp end do

        end select

!$omp end parallel

        call smooth_val_SBL( ukn_sbl )
        call smooth_val_SBL( vkn_sbl )

     end do

  case ('AB2')  ! 2nd order Adams-Bashforse scheme

     call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                  force_uk, force_vk )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              ukn_sbl(i,j)=uko_sbl(i,j)+dt*force_uk(i,j)
              vkn_sbl(i,j)=vko_sbl(i,j)+dt*force_vk(i,j)
              sbl_optu(i,j)=force_uk(i,j)
              sbl_optv(i,j)=force_vk(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              ukn_sbl(i,j)=uko_sbl(i,j)+dt*(1.5d0*force_uk(i,j)-0.5d0*sbl_optu(i,j))
              vkn_sbl(i,j)=vko_sbl(i,j)+dt*(1.5d0*force_vk(i,j)-0.5d0*sbl_optv(i,j))
              sbl_optu(i,j)=force_uk(i,j)
              sbl_optv(i,j)=force_vk(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
           vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
           ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
           vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

     call smooth_val_SBL( ukn_sbl )
     call smooth_val_SBL( vkn_sbl )

  case ('L-F')  ! Leap Frog scheme

     call force_SBL( u_nbm_isp, v_nbm_isp, pk, uko_sbl, vko_sbl,  &
  &                  force_uk, force_vk )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              ukn_sbl(i,j)=uko_sbl(i,j)+dt*force_uk(i,j)
              vkn_sbl(i,j)=vko_sbl(i,j)+dt*force_vk(i,j)
              sbl_optu(i,j)=uko_sbl(i,j)
              sbl_optv(i,j)=vko_sbl(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,2*hynt+1
           do i=1,hxnt+1
              ukn_sbl(i,j)=sbl_optu(i,j)+2.0d0*dt*force_uk(i,j)
              vkn_sbl(i,j)=sbl_optv(i,j)+2.0d0*dt*force_vk(i,j)
              sbl_optu(i,j)=uko_sbl(i,j)
              sbl_optv(i,j)=vko_sbl(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=2,hynt+1
        do i=2,hxnt+1
           ukn_sbl(kxnt-i+2,kynt-j+2)=dconjg(ukn_sbl(i,j))
           vkn_sbl(kxnt-i+2,kynt-j+2)=dconjg(vkn_sbl(i,j))
           ukn_sbl(kxnt-i+2,j)=dconjg(ukn_sbl(i,kynt-j+2))
           vkn_sbl(kxnt-i+2,j)=dconjg(vkn_sbl(i,kynt-j+2))
        end do
     end do

!$omp end do
!$omp end parallel

     call smooth_val_SBL( ukn_sbl )
     call smooth_val_SBL( vkn_sbl )

!  case ('C-N')  ! Crank-Nicolson scheme

!  case ('BEU')  ! Backward Euler scheme

  end select

  uko_sbl=ukt_sbl
  vko_sbl=vkt_sbl

end subroutine time_integration_SBL

end module time_scheme
