!= Хĥǥ
!
!= Bucket model
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: bucket_model.f90,v 1.6 2010-08-24 06:34:32 yot Exp $
! Tag Name::  $Name: dcpam5-20110228-4 $
! Copyright:: Copyright (C) GFD Dennou Club, 2008-2009. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module Bucket_Model

  ! ⥸塼 ; USE statements
  !


  ! ̷ѥ᥿
  ! Kind type parameter
  !
  use dc_types, only: DP      ! ټ¿. Double precision.

  ! ʸ ; Declaration statements
  !
  implicit none
  private

  logical, save:: bucket_model_inited = .false.
                              ! ե饰.
                              ! Initialization flag


  logical, save, public:: FlagBucketModel   ! Хĥǥ 󡿥. 
                                            ! bucket model on/off. 

  logical, save, public:: FlagBucketModelSnow   ! Хĥǥ ΰ 󡿥.
                                                ! bucket model treatment of snow on/off.

  ! ³
  ! Public procedure
  !
  public:: BucketEvap
  public:: BucketEvapAdjust
  public:: BucketIntegration
  public:: BucketPRCPAdjust
  public:: BucketModHumidCoef
  public:: BucketModQvapFlux

  ! ѿ
  ! Private variables
  !
  real(DP), save:: SoilMoistCritAmnt
                        ! <Japanese>
                        ! Critical amount of soil moisture
  real(DP), save:: SoilMoistCritAmntforEvapEff
                        ! <Japanese>
                        ! Critical amount of soil moisture for evaporation efficiency

  real(DP), save:: SoilMoistMeaningLess = -1.0d0
                        ! <Japanese>
                        ! Meaning less value for soil moisture on the ocean


  character(*), parameter:: module_name = 'bucket_model'
                              ! ⥸塼̾. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name: dcpam5-20110228-4 $' // &
    & '$Id: bucket_model.f90,v 1.6 2010-08-24 06:34:32 yot Exp $'
                              ! ⥸塼ΥС
                              ! Module version


  ! INTERFACE ʸ ; INTERFACE statements
  !


  !**************************************************************************************

contains

  !**************************************************************************************

  subroutine BucketIntegration( &
    & xy_SurfCond,                     & ! (in )
    & xy_DSoilMoistDt, xy_DSurfSnowDt, & ! (in )
    & xy_SoilMoistB, xy_SurfSnowB,     & ! (in )
    & xy_SoilMoistA, xy_SurfSnowA      & ! (out)
    )

    ! 
    ! Time control
    !
    use timeset, only: &
      & DelTime, &             ! $ \Delta t $ [s]
      & TimesetClockStart, TimesetClockStop

    ! ʻ
    ! Grid points settings
    !
    use gridset, only: imax, & ! ٳʻ.
                               ! Number of grid points in longitude
      &                jmax, & ! ٳʻ.
                               ! Number of grid points in latitude
      &                kmax    ! ľؿ.
                               ! Number of vertical level

    ! ȳɹ
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only: &
      & ThresholdSurfSnow,          &
      & TempCondWater

    integer , intent(in ) :: xy_SurfCond    ( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_DSoilMoistDt( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_DSurfSnowDt ( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SoilMoistB  ( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SurfSnowB   ( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_SoilMoistA  ( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_SurfSnowA   ( 0:imax-1, 1:jmax )

    ! ѿ
    ! Work variables
    !
    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitude
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitude


    ! ׻ַ¬
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )

    ! 
    ! Initialization
    !
    if ( .not. bucket_model_inited ) call BucketInit

    if ( .not. FlagBucketModel ) then
      xy_SoilMoistA = xy_SoilMoistB
      xy_SurfSnowA  = xy_SurfSnowB
      return
    end if


    xy_SoilMoistA = xy_SoilMoistB + xy_DSoilMoistDt * 2.0d0 * DelTime

    ! Remove negative values
    !
    do j = 1, jmax
      do i = 0, imax-1
        if( xy_SoilMoistA(i,j) < 0.0d0 ) xy_SoilMoistA(i,j) = 0.0d0
      end do
    end do

    if ( FlagBucketModelSnow ) then
      xy_SurfSnowA = xy_SurfSnowB + xy_DSurfSnowDt * 2.0d0 * DelTime

      ! Remove negative values
      !
      do j = 1, jmax
        do i = 0, imax-1
          if( xy_SurfSnowA (i,j) < 0.0d0 ) xy_SurfSnowA (i,j) = 0.0d0
        end do
      end do
    else
      xy_SurfSnowA = xy_SurfSnowB
    end if


    ! Fill meaningless value in ocean grid
    !
    do j = 1, jmax
      do i = 0, imax-1
        if ( xy_SurfCond( i, j ) == 0 ) then
          xy_SoilMoistA(i,j) = SoilMoistMeaningLess
          xy_SurfSnowA(i,j)  = SoilMoistMeaningLess
        end if
      end do
    end do


    ! ׻ַ¬
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )


  end subroutine BucketIntegration

  !**************************************************************************************

  subroutine BucketEvap( &
    & xy_SurfCond, xy_SurfEvapFlux, & ! (in )
    & xy_SoilMoistB, xy_SurfSnowB,  & ! (in )
    & xy_SoilMoistA, xy_SurfSnowA   & ! (out)
    )

    ! 
    ! Time control
    !
    use timeset, only: &
      & DelTime, &             ! $ \Delta t $ [s]
      & TimesetClockStart, TimesetClockStop

    ! ʻ
    ! Grid points settings
    !
    use gridset, only: imax, & ! ٳʻ.
                               ! Number of grid points in longitude
      &                jmax, & ! ٳʻ.
                               ! Number of grid points in latitude
      &                kmax    ! ľؿ.
                               ! Number of vertical level

    ! ȳɹ
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only: &
      & ThresholdSurfSnow,          &
      & TempCondWater

    integer , intent(in ) :: xy_SurfCond    ( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SurfEvapFlux( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SoilMoistB  ( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SurfSnowB   ( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_SoilMoistA  ( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_SurfSnowA   ( 0:imax-1, 1:jmax )

    ! ѿ
    ! Work variables
    !
    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitude
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitude


    ! ׻ַ¬
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )

    ! 
    ! Initialization
    !
    if ( .not. bucket_model_inited ) call BucketInit

    if ( .not. FlagBucketModel ) then
      xy_SoilMoistA = xy_SoilMoistB
      xy_SurfSnowA  = xy_SurfSnowB
      return
    end if


    if ( FlagBucketModelSnow ) then
      ! Evaporation is subtracted from surface snow and soil moisture
      !
      xy_SurfSnowA = xy_SurfSnowB - xy_SurfEvapFlux * 2.0d0 * DelTime
      do j = 1, jmax
        do i = 0, imax-1
          if ( xy_SurfSnowA(i,j) < 0.0d0 ) then
            xy_SoilMoistA(i,j) = xy_SoilMoistB(i,j) + xy_SurfSnowA(i,j)
            xy_SurfSnowA (i,j) = 0.0d0
          else
            xy_SoilMoistA(i,j) = xy_SoilMoistB(i,j)
          end if
        end do
      end do
    else
      ! Evaporation is subtracted from soil moisture
      !
      xy_SoilMoistA = xy_SoilMoistB - xy_SurfEvapFlux * 2.0d0 * DelTime
      xy_SurfSnowA  = xy_SurfSnowB
    end if

    ! Remove negative values
    !
    do j = 1, jmax
      do i = 0, imax-1
        if( xy_SoilMoistA(i,j) < 0.0d0 ) xy_SoilMoistA(i,j) = 0.0d0
        if( xy_SurfSnowA (i,j) < 0.0d0 ) xy_SurfSnowA (i,j) = 0.0d0
      end do
    end do

    ! Fill meaningless value in ocean grid
    !
    do j = 1, jmax
      do i = 0, imax-1
        if ( xy_SurfCond( i, j ) == 0 ) then
          xy_SoilMoistA(i,j) = SoilMoistMeaningLess
          xy_SurfSnowA(i,j)  = SoilMoistMeaningLess
        end if
      end do
    end do

    ! ׻ַ¬
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )


  end subroutine BucketEvap

  !**************************************************************************************

  subroutine BucketEvapAdjust( &
    & xy_SurfCond, xy_SurfEvapFlux, & ! (in   )
    & xy_SoilMoist, xy_SurfSnow     & ! (inout)
    )

    ! ʻ
    ! Grid points settings
    !
    use gridset, only: imax, & ! ٳʻ.
                               ! Number of grid points in longitude
      &                jmax, & ! ٳʻ.
                               ! Number of grid points in latitude
      &                kmax    ! ľؿ.
                               ! Number of vertical level

    integer , intent(in   ) :: xy_SurfCond    ( 0:imax-1, 1:jmax )
    real(DP), intent(in   ) :: xy_SurfEvapFlux( 0:imax-1, 1:jmax )
    real(DP), intent(inout) :: xy_SoilMoist   ( 0:imax-1, 1:jmax )
    real(DP), intent(inout) :: xy_SurfSnow    ( 0:imax-1, 1:jmax )

    ! ѿ
    ! Work variables
    !
    real(DP) :: xy_SoilMoistB( 0:imax-1, 1:jmax )
    real(DP) :: xy_SurfSnowB ( 0:imax-1, 1:jmax )
    real(DP) :: xy_SoilMoistA( 0:imax-1, 1:jmax )
    real(DP) :: xy_SurfSnowA ( 0:imax-1, 1:jmax )


    xy_SoilMoistB = xy_SoilMoist
    xy_SurfSnowB  = xy_SurfSnow

    call BucketEvap( &
      & xy_SurfCond, xy_SurfEvapFlux, & ! (in )
      & xy_SoilMoistB, xy_SurfSnowB,  & ! (in )
      & xy_SoilMoistA, xy_SurfSnowA   & ! (out)
      )

    xy_SoilMoist = xy_SoilMoistA
    xy_SurfSnow  = xy_SurfSnowA


  end subroutine BucketEvapAdjust

  !**************************************************************************************

  subroutine BucketPRCPAdjust( &
    & xy_SurfCond, xy_Ps, xy_SurfPRCPFlux, &  ! (in )
    & xyz_Temp, xy_SoilMoist, xy_SurfSnow  &  ! (inout)
    & )

    ! 
    ! Time control
    !
    use timeset, only: &
      & TimeN,   &             ! ƥå $ t $ λ. Time of step $ t $.
      & DelTime, &             ! $ \Delta t $ [s]
      & TimesetClockStart, TimesetClockStop

    ! ʻ
    ! Grid points settings
    !
    use gridset, only: imax, & ! ٳʻ.
                               ! Number of grid points in longitude
      &                jmax, & ! ٳʻ.
                               ! Number of grid points in latitude
      &                kmax    ! ľؿ.
                               ! Number of vertical level

    ! ɸǡ
    ! Axes data settings
    !
    use axesset, only: &
      & z_DelSigma
                              ! $ \Delta \sigma $ ().
                              ! $ \Delta \sigma $ (Full)

    ! ʪ
    ! Physical constants settings
    !
    use constants, only:  &
      & CpDry,            &
                              ! $ C_p $ [J kg-1 K-1].
                              ! 絤갵Ǯ.
                              ! Specific heat of air at constant pressure
      & Grav,             &
                              ! $ g $ [m s-2].
                              ! ϲ®.
                              ! Gravitational acceleration
      & LatentHeatFusion
                              ! $ L $ [J kg-1] .
                              ! ͻǮ.
                              ! Latent heat of fusion

    ! ȳɹ
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only: &
      & TempCondWater

    ! ҥȥǡ
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    integer , intent(in   ) :: xy_SurfCond    ( 0:imax-1, 1:jmax )
    real(DP), intent(in   ) :: xy_Ps          ( 0:imax-1, 1:jmax )
    real(DP), intent(in   ) :: xy_SurfPRCPFlux( 0:imax-1, 1:jmax )
    real(DP), intent(inout) :: xyz_Temp       ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xy_SoilMoist   ( 0:imax-1, 1:jmax )
    real(DP), intent(inout) :: xy_SurfSnow    ( 0:imax-1, 1:jmax )


    ! ѿ
    ! Work variables
    !
    real(DP) :: xy_TempIncByFusion( 0:imax-1, 1:jmax )
                              ! Temperature increase by fusion

    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitude
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitude


    ! ׻ַ¬
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )

    ! 
    ! Initialization
    !
    if ( .not. bucket_model_inited ) call BucketInit

    if ( .not. FlagBucketModel ) return


    ! Initialize an array for temperature increase by fusion
    !
    xy_TempIncByFusion = 0.0d0


    if ( FlagBucketModelSnow ) then
      ! Precipitation is added to soil moisture or surface snow
      !
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Temp(i,j,1) > TempCondWater ) then
            xy_SoilMoist(i,j) = xy_SoilMoist(i,j) &
              & + xy_SurfPRCPFlux(i,j) * 2.0d0 * DelTime
          else

            ! Heating by latent heat release by fusion.
            !   It is assumed that all of latent heat for fusion is used to heat 
            !   the lowest layer. 
            !   It should be checked whether this causes unstable layer or not. 
            !   (yot, 2010/08/17)
            !   This is commented out. (yot, 2010/08/21)
            !
            xy_TempIncByFusion(i,j) =                                     &
              & xy_SurfPRCPFlux(i,j) * LatentHeatFusion * 2.0d0 * DelTime &
              &   / ( CpDry * z_DelSigma(1) * xy_Ps(i,j) / Grav )

!!$            xyz_Temp(i,j,1) = xyz_Temp(i,j,1) + xy_TempIncByFusion(i,j)

!!$            if ( xy_TempIncByFusion(i,j) > 0.0d0 ) then
!!$              write( 6, * ) i, j, xyz_Temp(i,j,1), xy_TempIncByFusion(i,j)
!!$            end if


            xy_SurfSnow (i,j) = xy_SurfSnow (i,j) &
              & + xy_SurfPRCPFlux(i,j) * 2.0d0 * DelTime
          end if
        end do
      end do
    else
      ! Precipitation is added to soil moisture
      !
      xy_SoilMoist = xy_SoilMoist + xy_SurfPRCPFlux * 2.0d0 * DelTime
    end if

    ! Calculation of Run-off
    !
    do j = 1, jmax
      do i = 0, imax-1
        if ( xy_SoilMoist(i,j) > SoilMoistCritAmnt ) &
          & xy_SoilMoist(i,j) = SoilMoistCritAmnt
      end do
    end do

    ! Fill meaningless value in ocean grid
    !
    do j = 1, jmax
      do i = 0, imax-1
        if ( xy_SurfCond(i,j) == 0 ) then
          xy_SoilMoist(i,j) = SoilMoistMeaningLess
          xy_SurfSnow (i,j) = SoilMoistMeaningLess
        end if
      end do
    end do

    ! ҥȥǡ
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'TempIncByFusion' , xy_TempIncByFusion )

    ! ׻ַ¬
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine BucketPRCPAdjust

  !**************************************************************************************

  subroutine BucketModHumidCoef(              &
    & xy_SurfCond, xy_SoilMoist, xy_SurfSnow, & ! (in   )
    & xy_SurfHumidCoef                        & ! (inout)
    & )

    ! 
    ! Time control
    !
    use timeset, only: &
      & DelTime            ! $ \Delta t $ [s]

    ! ʻ
    ! Grid points settings
    !
    use gridset, only: imax, & ! ٳʻ.
                               ! Number of grid points in longitude
      &                jmax, & ! ٳʻ.
                               ! Number of grid points in latitude
      &                kmax    ! ľؿ.
                               ! Number of vertical level

    ! ȳɹ
    ! Setting constants of snow and sea ice
    !
    use constants_snowseaice, only: &
      & ThresholdSurfSnow

    integer , intent(in   ) :: xy_SurfCond     ( 0:imax-1, 1:jmax )
    real(DP), intent(in   ) :: xy_SoilMoist    ( 0:imax-1, 1:jmax )
    real(DP), intent(in   ) :: xy_SurfSnow     ( 0:imax-1, 1:jmax )
    real(DP), intent(inout) :: xy_SurfHumidCoef( 0:imax-1, 1:jmax )


    ! ѿ
    ! Work variables
    !
    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitude
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitude

    ! 
    ! Initialization
    !
    if ( .not. bucket_model_inited ) call BucketInit

    if ( .not. FlagBucketModel ) return


    ! Surface humidity coefficient is modified.
    !
    if ( FlagBucketModelSnow ) then
      do j = 1, jmax
        do i = 0, imax-1
          if ( xy_SurfCond(i,j) == 0 ) then
            xy_SurfHumidCoef(i,j) = xy_SurfHumidCoef(i,j)
          else if ( xy_SurfSnow(i,j) > ThresholdSurfSnow ) then
            xy_SurfHumidCoef(i,j) = 1.0d0
          else
            xy_SurfHumidCoef(i,j) = xy_SoilMoist(i,j) / SoilMoistCritAmntforEvapEff
            if ( xy_SurfHumidCoef(i,j) > 1.0d0 ) xy_SurfHumidCoef(i,j) = 1.0d0
          end if
        end do
      end do
    else
      do j = 1, jmax
        do i = 0, imax-1
          if ( xy_SurfCond(i,j) == 0 ) then
            xy_SurfHumidCoef(i,j) = xy_SurfHumidCoef(i,j)
          else
            xy_SurfHumidCoef(i,j) = xy_SoilMoist(i,j) / SoilMoistCritAmntforEvapEff
            if ( xy_SurfHumidCoef(i,j) > 1.0d0 ) xy_SurfHumidCoef(i,j) = 1.0d0
          end if
        end do
      end do
    end if


  end subroutine BucketModHumidCoef

  !**************************************************************************************

  subroutine BucketModQvapFlux(               &
    & xy_SurfCond, xy_SoilMoist, xy_SurfSnow, & ! (in   )
    & xy_SurfEvapFlux                         & ! (inout)
    & )

    ! 
    ! Time control
    !
    use timeset, only: &
      & DelTime, &             ! $ \Delta t $ [s]
      & TimesetClockStart, TimesetClockStop

    ! ʻ
    ! Grid points settings
    !
    use gridset, only: imax, & ! ٳʻ.
                               ! Number of grid points in longitude
      &                jmax, & ! ٳʻ.
                               ! Number of grid points in latitude
      &                kmax    ! ľؿ.
                               ! Number of vertical level

    integer , intent(in   ) :: xy_SurfCond     ( 0:imax-1, 1:jmax )
    real(DP), intent(in   ) :: xy_SoilMoist    ( 0:imax-1, 1:jmax )
    real(DP), intent(in   ) :: xy_SurfSnow     ( 0:imax-1, 1:jmax )
    real(DP), intent(inout) :: xy_SurfEvapFlux ( 0:imax-1, 1:jmax )


    ! ѿ
    ! Work variables
    !
    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitude
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitude


    ! ׻ַ¬
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! 
    ! Initialization
    !
    if ( .not. bucket_model_inited ) call BucketInit

    if ( .not. FlagBucketModel ) return


    if ( FlagBucketModelSnow ) then
      ! Surface water vapor flux is limited up to the water and snow amount on the land. 
      !
      do j = 1, jmax
        do i = 0, imax-1
          if ( ( xy_SurfCond    (i,j)                   >= 1      ) .and.&
            &  ( xy_SurfEvapFlux(i,j)                   >  0.0d0  ) .and.&
            &  ( xy_SurfEvapFlux(i,j) * 2.0d0 * DelTime >  &
            &       xy_SoilMoist(i,j) + xy_SurfSnow(i,j)          ) ) &
            & then
            xy_SurfEvapFlux(i,j) &
              = ( xy_SoilMoist(i,j) + xy_SurfSnow(i,j) ) / ( 2.0d0 * DelTime )
          end if
        end do
      end do
    else
      ! Surface water vapor flux is limited up to the water amount on the land. 
      !
      do j = 1, jmax
        do i = 0, imax-1
          if ( ( xy_SurfCond    (i,j)                   >= 1                 ) .and.&
            &  ( xy_SurfEvapFlux(i,j)                   >  0.0d0             ) .and.&
            &  ( xy_SurfEvapFlux(i,j) * 2.0d0 * DelTime >  xy_SoilMoist(i,j) ) ) &
            & then
            xy_SurfEvapFlux(i,j) = xy_SoilMoist(i,j) / ( 2.0d0 * DelTime )
          end if
        end do
      end do
    end if

    ! ׻ַ¬
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )


  end subroutine BucketModQvapFlux

  !**************************************************************************************

  subroutine BucketInit
    !
    ! bucket_model ⥸塼νԤޤ. 
    ! NAMELIST#bucket_model_nml ɤ߹ߤϤμ³ǹԤޤ. 
    !
    ! "bucket_model" module is initialized. 
    ! "NAMELIST#bucket_model_nml" is loaded in this procedure. 
    !

    ! ⥸塼 ; USE statements
    !

    ! NAMELIST եϤ˴ؤ桼ƥƥ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ե
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ̷ѥ᥿
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! ɸϤֹ. Unit number of standard output

    ! ʸ
    ! Character handling
    !
    use dc_string, only: StoA

    ! å
    ! Message output
    !
    use dc_message, only: MessageNotify

    ! ҥȥǡ
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable


    ! ʸ ; Declaration statements
    !
    implicit none

    integer:: unit_nml        ! NAMELIST ե륪ץֹ. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST ɤ߹߻ IOSTAT. 
                              ! IOSTAT of NAMELIST read

    ! NAMELIST ѿ
    ! NAMELIST group name
    !
    namelist /bucket_model_nml/ &
      & FlagBucketModel, &
      & SoilMoistCritAmnt, SoilMoistCritAmntforEvapEff, &
      & FlagBucketModelSnow


    ! ¹ʸ ; Executable statement
    !

    if ( bucket_model_inited ) return
!!$    call InitCheck

    ! ǥեͤ
    ! Default values settings
    !
    FlagBucketModel              = .false.

    ! Values from Manabe (1969)
    !  Manabe, Climate and the ocean circulation I. The atmospheric circulation and 
    !  the hydrology of the Earth's surface, Mon. Wea. Rev., 97, 739-774, 1969
    !
    SoilMoistCritAmnt            = 1.0d3 * 0.15d0
    SoilMoistCritAmntforEvapEff  = 1.0d3 * 0.15d0 * 0.75d0

    FlagBucketModelSnow           = .false.


    ! NAMELIST ɤ߹
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &                ! (in)
        & nml = bucket_model_nml, &    ! (out)
        & iostat = iostat_nml )        ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    if ( ( FlagBucketModelSnow ) .and. ( .not. FlagBucketModel ) ) then
      call MessageNotify( 'E', module_name, 'FlagBucketModel has to be true, when FlagBucketModelSnow is true.' )
    end if


    ! ҥȥǡϤΤΤؤѿϿ
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'TempIncByFusion', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'temperature increase by fusion', 'K' )


    !  ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )

    call MessageNotify( 'M', module_name, '  FlagBucketModel             = %y', l = (/ FlagBucketModel   /) )
    call MessageNotify( 'M', module_name, '  SoilMoistCritAmnt           = %f', d = (/ SoilMoistCritAmnt  /) )
    call MessageNotify( 'M', module_name, '  SoilMoistCritAmntforEvapEff = %f', d = (/ SoilMoistCritAmntforEvapEff /) )
    call MessageNotify( 'M', module_name, '  FlagBucketModelSnow         = %y', l = (/ FlagBucketModelSnow /) )


    bucket_model_inited = .true.


  end subroutine BucketInit

  !**************************************************************************************

end module Bucket_Model
